#pragma once

#include "racktables_actor.h"
#include "dc_matcher.h"

#include <library/cpp/monlib/metrics/histogram_collector.h>

using namespace NActors;
using namespace NMonitoring;

namespace NSolomon::NFetcher {

class TRackTablesMetrics {
public:
    explicit TRackTablesMetrics(TMetricRegistry& registry)
        : DownloadStatus_{{
                registry.Rate({{"sensor", "racktables.download.status"}, {"code", "fail"}}),
                registry.Rate({{"sensor", "racktables.download.status"}, {"code", "2xx"}}),
                registry.Rate({{"sensor", "racktables.download.status"}, {"code", "3xx"}}),
                registry.Rate({{"sensor", "racktables.download.status"}, {"code", "4xx"}}),
                registry.Rate({{"sensor", "racktables.download.status"}, {"code", "5xx"}}),
        }}
        , ParseStatus_{{
                registry.Rate({{"sensor", "racktables.parse.status"}, {"status", "fail"}}),
                registry.Rate({{"sensor", "racktables.parse.status"}, {"status", "ok"}}),
        }}
        , CacheStatus_{{
                registry.Rate({{"sensor", "racktables.cache.status"}, {"status", "fail"}}),
                registry.Rate({{"sensor", "racktables.cache.status"}, {"status", "ok"}}),
        }}
        , Resolve_{
                registry.Rate({{"sensor", "racktables.resolve"}})
        }
        , DownloadTimesMillis_{registry.HistogramRate(
                {{"sensor", "racktables.download.time_millis"}},
                ExponentialHistogram(16, 2, 1))
        }
        , ParseTimesMillis_{registry.HistogramRate(
                {{"sensor", "racktables.parse.time_millis"}},
                ExponentialHistogram(16, 2, 1))
        }
        , Rows_{registry.IntGauge({{"sensor", "racktables.rows"}})}
        , AgeMillis_{registry.IntGauge({{"sensor", "racktables.age_millis"}})}
    {
        for (auto dc: GetEnumAllValues<EDc>()) {
            ResolveDc_[static_cast<size_t>(dc)] = registry.Rate({{"sensor", "racktables.resolve.dc"}, {"dc", ToString(dc)}});
        }
    }

    void WriteCode(HttpCodes code) {
        if (code == HTTP_CODE_MAX || code < 100) {
            DownloadStatus_[0]->Inc();
        } else {
            const auto codeBucket = code / 100;
            DownloadStatus_[codeBucket - 1]->Inc();
        }
    }

    void WriteResolveDc(EDc dc) {
        ResolveDc_[static_cast<size_t>(dc)]->Inc();
        Resolve_->Inc();
    }

    void WriteParseStatus(bool ok) {
        ParseStatus_[ok]->Inc();
    }
    void WriteCacheStatus(bool ok) {
        CacheStatus_[ok]->Inc();
    }
    void WriteDownloadTime(TDuration time) noexcept {
        DownloadTimesMillis_->Record(time.MilliSeconds());
    }
    void WriteParseTime(TDuration time) noexcept {
        ParseTimesMillis_->Record(time.MilliSeconds());
    }
    void WriteRows(size_t rows) noexcept {
        Rows_->Set(rows);
    }
    void WriteAge(TDuration age) noexcept {
        AgeMillis_->Set(age.MilliSeconds());
    }

private:
    std::array<TRate*, 5> DownloadStatus_;
    std::array<TRate*, 2> ParseStatus_;
    std::array<TRate*, 2> CacheStatus_;
    std::array<TRate*, 11> ResolveDc_;
    TRate* Resolve_;
    THistogram* DownloadTimesMillis_;
    THistogram* ParseTimesMillis_;
    TIntGauge* Rows_;
    TIntGauge* AgeMillis_;
};

} // namespace solomon

