#include <solomon/services/fetcher/lib/racktables/racktables_actor.h>
#include <solomon/services/fetcher/testlib/actor_system.h>
#include <solomon/services/fetcher/testlib/http_server.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace NSolomon;

NMonitoring::TMetricRegistry Metrics;

constexpr auto DefaultRefreshInterval = TDuration::Minutes(1);

class TRackTablesTest: public ::testing::Test {
protected:
    TRackTablesTest()
        : Registry_({{"common", "label"}})
    {
    }

    void SetUp() override {
        Server_.Reset(new TTestServer);
        Runtime_ = TTestActorRuntime::CreateInited();

        TRackTablesActorConf conf{
            .RefreshInterval = DefaultRefreshInterval,
            .ConnectTimeout = TDuration::Seconds(5),
            .ReadTimeout = TDuration::Seconds(5),
            .Retries = 0,
            .Url = Server_->Address(),
            .HttpClient = CreateCurlClient({}, Metrics),
            .Registry = Registry_,
        };
        Resolver_ = Runtime_->Register(CreateRackTablesActor(conf));
        Edge_ = Runtime_->AllocateEdgeActor();
        Runtime_->WaitForBootstrap();
    }

protected:
    TTestServer* Server() {
        return Server_.Get();
    }

protected:
    THolder<TTestActorRuntime> Runtime_;
    TActorId Resolver_;
    TActorId Edge_;
    THolder<TTestServer> Server_;
    NMonitoring::TMetricRegistry Registry_;
};

const TString RESPONSE_CONTENT {
R"(2a02:6b8:0:305::/64 [Владимир]
2a02:6b8:0:306::/64 [Владимир]
2a02:6b8:0:408::/64 [Морозов]
2a02:6b8:0:409::/64 [Морозов]
2a02:6b8:0:40a::/64 [Морозов]
2a02:6b8:0:870::/64 [Ивантеевка]
2a02:6b8:0:872::/64 [Ивантеевка]
2a02:6b8:c08:5780::/57 [Сасово-1]
2a02:6b8:c1d:1380::/57 [Владимир-2]
2a02:6b8:b082:15::/64 [офис Мянтсяля]
2a02:6b8:c08:5800::/57 [Сасово-1])"
};

TEST_F(TRackTablesTest, Simple) {
    Server()->AddHandler("/", [] {
        THttpResponse resp;
        resp.SetContent(RESPONSE_CONTENT);
        return resp;
    });

    bool ok{};
    {
        const TIpv6Address testIpSas = TIpv6Address::FromString("2a02:6b8:c08:5795:0:4c7d:fd8d:d36d", ok);
        Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TRackTablesEvents::TDcResolve{testIpSas}});
        auto ev = Runtime_->GrabEdgeEvent<TRackTablesEvents::TDcResult>(Edge_);
        ASSERT_EQ(ev->Get()->Dc , EDc::SAS);
    }
    {
        const TIpv6Address testIpVla = TIpv6Address::FromString("2a02:6b8:c1d:1380:2a02:6b8:c1d:1380", ok);
        Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TRackTablesEvents::TDcResolve{testIpVla}});
        auto ev = Runtime_->GrabEdgeEvent<TRackTablesEvents::TDcResult>(Edge_);
        ASSERT_EQ(ev->Get()->Dc , EDc::VLA);
    }
    {
        const TIpv6Address testIpMan = TIpv6Address::FromString("2a02:6b8:b082:15:15:b082:6b8:2a02", ok);
        Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TRackTablesEvents::TDcResolve{testIpMan}});
        auto ev = Runtime_->GrabEdgeEvent<TRackTablesEvents::TDcResult>(Edge_);
        ASSERT_EQ(ev->Get()->Dc , EDc::MAN);
    }
    {
        const TIpv6Address testIpUnknown = TIpv6Address::FromString("2a00:1450:4010:c0b::71", ok);
        Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TRackTablesEvents::TDcResolve{testIpUnknown}});
        auto ev = Runtime_->GrabEdgeEvent<TRackTablesEvents::TDcResult>(Edge_);
        ASSERT_EQ(ev->Get()->Dc , EDc::UNKNOWN);
    }
}

const TString INVALID_RESPONSE_CONTENT {R"(Some invalid text, this text is not valid)"};

TEST_F(TRackTablesTest, Poison) {
    Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TEvents::TEvPoison});
    auto ev = Runtime_->GrabEdgeEvent<TEvents::TEvPoisonTaken>(Edge_);
    ASSERT_FALSE(Runtime_->FindActor(Resolver_));
}

TEST_F(TRackTablesTest, BadRequestErrors) {
    Server()->AddHandler("/", [] {
        THttpResponse resp(HTTP_BAD_REQUEST);
        resp.SetContent(INVALID_RESPONSE_CONTENT);
        return resp;
    });

    bool ok{};
    const TIpv6Address testIpSas = TIpv6Address::FromString("2a02:6b8:c08:5795:0:4c7d:fd8d:d36d", ok);
    auto evHandle = new IEventHandle{Resolver_, Edge_, new TRackTablesEvents::TDcResolve{testIpSas}};
    Runtime_->Send(evHandle);
    auto ev = Runtime_->GrabEdgeEvent<TRackTablesEvents::TDcResult>(Edge_, 5*DefaultRefreshInterval);
    ASSERT_FALSE(ev);
}

TEST_F(TRackTablesTest, ForbiddenErrors) {
    Server()->AddHandler("/", [] {
        THttpResponse resp(HTTP_FORBIDDEN);
        resp.SetContent(INVALID_RESPONSE_CONTENT);
        return resp;
    });

    bool ok{};
    const TIpv6Address testIpSas = TIpv6Address::FromString("2a02:6b8:c08:5795:0:4c7d:fd8d:d36d", ok);
    Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TRackTablesEvents::TDcResolve{testIpSas}});
    auto ev = Runtime_->GrabEdgeEvent<TRackTablesEvents::TDcResult>(Edge_, DefaultRefreshInterval);
    ASSERT_FALSE(ev);
}

TEST_F(TRackTablesTest, GatewayTimeOutErrors) {
    Server()->AddHandler("/", [] {
        THttpResponse resp(HTTP_GATEWAY_TIME_OUT);
        resp.SetContent(INVALID_RESPONSE_CONTENT);
        return resp;
    });

    bool ok{};
    const TIpv6Address testIpSas = TIpv6Address::FromString("2a02:6b8:c08:5795:0:4c7d:fd8d:d36d", ok);
    Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TRackTablesEvents::TDcResolve{testIpSas}});
    auto ev = Runtime_->GrabEdgeEvent<TRackTablesEvents::TDcResult>(Edge_, DefaultRefreshInterval);
    ASSERT_FALSE(ev);
}

TEST_F(TRackTablesTest, InvalidErrors) {
    Server()->AddHandler("/", [] {
        THttpResponse resp;
        resp.SetContent(INVALID_RESPONSE_CONTENT);
        return resp;
    });

    bool ok{};
    const TIpv6Address testIpSas = TIpv6Address::FromString("2a02:6b8:c08:5795:0:4c7d:fd8d:d36d", ok);
    Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TRackTablesEvents::TDcResolve{testIpSas}});
    auto ev = Runtime_->GrabEdgeEvent<TRackTablesEvents::TDcResult>(Edge_, DefaultRefreshInterval);
    ASSERT_EQ(ev->Get()->Dc , EDc::UNKNOWN);
}

TEST_F(TRackTablesTest, Recovery) {
    bool updateResponse = false;
    Server()->AddHandler("/", [&updateResponse] {
        if (!updateResponse) {
            THttpResponse resp(HTTP_GATEWAY_TIME_OUT);
            resp.SetContent(INVALID_RESPONSE_CONTENT);
            return resp;
        }
        THttpResponse resp;
        resp.SetContent(RESPONSE_CONTENT);
        return resp;
    });

    bool ok{};
    const TIpv6Address testIpSas = TIpv6Address::FromString("2a02:6b8:c08:5795:0:4c7d:fd8d:d36d", ok);
    Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TRackTablesEvents::TDcResolve{testIpSas}});
    auto ev = Runtime_->GrabEdgeEvent<TRackTablesEvents::TDcResult>(Edge_, DefaultRefreshInterval);
    ASSERT_FALSE(ev);

    updateResponse = true;
    Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TRackTablesEvents::TDcResolve{testIpSas}});
    Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TEvents::TEvWakeup});
    ev = Runtime_->GrabEdgeEvent<TRackTablesEvents::TDcResult>(Edge_);
    ASSERT_EQ(ev->Get()->Dc , EDc::SAS);
}
