#include "router.h"
#include "unknown_shard.h"

#include <solomon/services/fetcher/lib/app_data.h>
#include <solomon/services/fetcher/lib/fetcher_shard.h>
#include <solomon/services/fetcher/lib/config_updater/config_updater.h>
#include <solomon/services/fetcher/lib/sink/sink.h>
#include <solomon/services/fetcher/lib/queue/queue.h>
#include <solomon/services/fetcher/lib/cluster/cluster.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/log.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/events.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/generic/scope.h>
#include <util/string/cast.h>

using namespace std::string_view_literals;

namespace NSolomon {
    template <typename H>
    H AbslHashValue(H state, const TShardKey& key) {
        state = H::combine(std::move(state), absl::Hash<std::string_view>{}(key.ProjectId));
        state = H::combine(std::move(state), absl::Hash<std::string_view>{}(key.ClusterName));
        return H::combine(std::move(state), absl::Hash<std::string_view>{}(key.ServiceName));
    }
} // namespace NSolomon

namespace NSolomon::NFetcher {

using namespace NActors;
using namespace NMonitoring;
    template <typename H>
    H AbslHashValue(H state, const TShardData& data) {
        state = H::combine(std::move(state), absl::Hash<std::string_view>{}(data.ProjectId));
        state = H::combine(std::move(state), absl::Hash<std::string_view>{}(data.ClusterName));
        return H::combine(std::move(state), absl::Hash<std::string_view>{}(data.ServiceName));
    }

namespace {
    struct TIdHash {
        using is_transparent = void;

        size_t operator()(const TShardId& id) const noexcept {
            return id.NumId();
        }

        size_t operator()(ui32 numId) const noexcept {
            return numId;
        }
    };

    struct TIdEq {
        using is_transparent = void;

        bool operator()(ui32 numId, const TShardId& shardId) const noexcept {
            return numId == shardId.NumId();
        }

        bool operator()(const TShardId& lhs, const TShardId& rhs) const noexcept {
            return lhs == rhs;
        }
    };

    template <typename T>
    using TShardMap = absl::flat_hash_map<TShardId, T, TIdHash, TIdEq>;

    struct TKeyHash {
        using is_transparent = void;

        size_t operator()(const TShardKey& k) const noexcept {
            return absl::Hash<TShardKey>{}(k);
        }

        size_t operator()(const TShardData& d) const noexcept {
            return absl::Hash<TShardData>{}(d);
        }
    };

    struct TKeyEq {
        using is_transparent = void;

        bool operator()(const TShardData& d, const TShardKey& k) const noexcept {
            return d.ProjectId == k.ProjectId
                && d.ServiceName == k.ServiceName
                && d.ClusterName == k.ClusterName;
        }

        bool operator()(const TShardKey& lhs, const TShardKey& rhs) const noexcept {
            return lhs == rhs;
        }
    };

    struct TCounters {
        TCounters(TMetricRegistry& registry)
            : Registry_{registry}
        {
            Backlog_ = Registry_.IntGauge({{"sensor", "router.backlogSize"}});
            MemBacklog_ = Registry_.IntGauge({{"sensor", "router.backlogSizeBytes"}});
        }

        void IncDrop(const TShardId& shardId) {
            if (auto* drops = Drops_.FindPtr(shardId.NumId())) {
                (*drops)->Inc();
            } else {
                auto [it, _] = Drops_.emplace(
                    shardId.NumId(),
                    Registry_.Rate({{"sensor", "router.drops"}, {"shardId", shardId.StrId()}})
                );

                it->second->Inc();
            }
        }

        void IncBacklog(ui64 memSize) {
            Backlog_->Inc();
            MemBacklog_->Add(memSize);
        }

        void DecBacklog(ui64 memSize) {
            Backlog_->Dec();
            MemBacklog_->Add(-memSize);
        }

    private:
        TMetricRegistry& Registry_;
        THashMap<TShardId::TNumId, TRate*> Drops_;
        IIntGauge* Backlog_;
        IIntGauge* MemBacklog_;
    };

    THolder<TEvSinkWrite> CreateSinkWriteEvent(TQueueEntry&& q, TShardId shardId, TString projectId) {
        THolder<TEvSinkWrite> ev{new TEvSinkWrite};

        ev->ShardData.ShardId = std::move(shardId);
        ev->ShardData.ProjectId = std::move(projectId);

        ev->ShardData.Host = std::move(q.Host);
        ev->ShardData.Url = std::move(q.Url);
        ev->ShardData.Interval = std::move(q.Interval);
        ev->ShardData.Format = std::move(q.Format);
        ev->ShardData.Labels = std::move(q.Labels);

        ev->ShardData.Data = std::move(q.Data);
        ev->ShardData.Instant = std::move(q.Instant);

        ev->ShardData.PrevData = std::move(q.PrevData);
        ev->ShardData.PrevInstant = std::move(q.PrevInstant);

        ev->ShardData.SourceId = std::move(q.SourceId);

        return ev;
    };

    class TRouterActor: public TActorBootstrapped<TRouterActor> {
    public:
        TRouterActor(TRouterActorConf conf, TMetricRegistry& registry)
            : Sinks_{std::move(conf.Peers)}
            , Counters_{registry}
            , UrlBacklogLimit_{conf.UrlBacklogLimit}
            , ShardBacklogLimit_{conf.ShardBacklogLimit}
            , UnknownShardHandlerId_{conf.UnknownShardHandlerId}
            , Cluster_{conf.Cluster}
        {
        }

        void Bootstrap(const TActorContext&) {
            Become(&TRouterActor::StateWork);
            Send(MakeConfigUpdaterId(), new TEvents::TEvSubscribe);
        }

        STFUNC(StateWork) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvSinkWrite, OnSinkWrite);
                hFunc(TEvConfigChanged, OnConfigChanged);
                hFunc(TEvRouterStatePeersRequest, OnStatePeersRequest);
                hFunc(TEvRouterStateBacklogsRequest, OnStateBacklogsRequest);
                hFunc(TEvRouterStateShardsRequest, OnStateRouterShardsRequest);
                hFunc(TEvShardCreated, OnShardCreated);
            }
        }

        void OnShardCreated(const TEvShardCreated::TPtr& evPtr) {
            if (!Cluster_) {
                return;
            }

            auto&& ev = *evPtr->Get();
            auto loc = Cluster_->NodeByFqdn(ev.Host);
            PcsToId_[ev.ShardKey] = ev.ShardId;

            if (!loc) {
                MON_WARN(Router, "Shard created with an unknown location: " << ev.Host);
                return;
            }

            AddShard(std::move(ev.ShardId), std::move(ev.ShardKey), *loc);
        }

        void OnStatePeersRequest(const TEvRouterStatePeersRequest::TPtr& ev) {
            TVector<TClusterNode> peers;

            for (auto&&[loc, _]: Sinks_) {
                peers.push_back(loc);
            }

            Send(ev->Sender, new TEvRouterStatePeersResponse(std::move(peers)));
        }

        void OnStateBacklogsRequest(const TEvRouterStateBacklogsRequest::TPtr& ev) {
            TVector<TRouterState::TBacklog> backlogs;

            for (auto&& [shardId, queue]: WriteBacklog_) {
                backlogs.push_back({
                                        .ShardId = shardId,
                                        .MemorySize = queue->MemorySize(),
                                        .EntrySize = queue->EntrySize(),
                                        .IsFull = queue->IsFull(),
                                    });
            }

            Send(ev->Sender, new TEvRouterStateBacklogsResponse(std::move(backlogs)));
        }

        void OnStateRouterShardsRequest(const TEvRouterStateShardsRequest::TPtr& ev) {
            TVector<TRouterState::TShard> routerShards;

            for (auto&& [shardId, info]: ShardMap_) {
                routerShards.push_back({
                                           .Id = shardId,
                                           .Location = info.Location,
                                       });
            }

            Send(ev->Sender, new TEvRouterStateShardsResponse(std::move(routerShards)));
        }

        void OnConfigChanged(const TEvConfigChanged::TPtr& evPtr) {
            auto&& ev = *evPtr->Get();

            for (auto&& shard: ev.Added) {
                // doesn't make sense for agent shards since they don't provide data directly
                if (shard.Type() != EFetcherShardType::Simple) {
                    return;
                }

                OnShardAdded(shard);
            }

            for (auto&& shard: ev.Changed) {
                if (shard.Type() != EFetcherShardType::Simple) {
                    return;
                }

                OnShardChanged(shard);
            }

            for (auto&& shardInfo: ev.Removed) {
                if (shardInfo.Type != EFetcherShardType::Simple) {
                    return;
                }

                OnShardRemoved(shardInfo.Id);
            }
        }

        void OnShardChanged(const TFetcherShard& shard) {
            auto&& shardId = shard.Id();
            auto&& loc = shard.Location();

            auto it = ShardMap_.find(shardId);
            if (it == ShardMap_.end()) {
                OnShardAdded(shard);
                return;
            }

            auto&& info = it->second;

            if (info.Location != loc) {
                info.Location = shard.Location();
                auto sinkId = SinkForLocation(loc);
                if (!sinkId) {
                    MON_WARN(Router, "Sink is not configured for " << loc);
                }

                info.ActorId = sinkId ? *sinkId : TActorId{};
            }
        }

        void OnShardAdded(const TFetcherShard& shard) {
            Y_VERIFY_DEBUG(shard.IsValid());


            TShardKey shardKey{ToString(shard.ProjectId()), shard.Cluster()->Name(), ToString(shard.ServiceName())};
            PcsToId_.emplace(shardKey, shard.Id());

            auto loc = shard.Location();
            if (loc.IsUnknown()) {
                return;
            }

            TString projectId{shard.ProjectId()};

            AddShard(shard.Id(), std::move(shardKey), loc);
        }

        void AddShard(TShardId shardId, const TShardKey& shardKey, const TClusterNode& loc) {
            auto&& projectId = shardKey.ProjectId;

            auto sinkId = SinkForLocation(loc);
            if (!sinkId) {
                MON_ERROR(Router, "Sink is not configured for " << loc);
                return;
            }

            auto [it, isInserted] = ShardMap_.try_emplace(
                shardId,
                std::move(loc), *sinkId, projectId
            );

            Y_VERIFY_DEBUG(isInserted);
            if (!isInserted) {
                MON_ERROR(Router, "Shard " << shardId << " is already present in the map. NumId clash?");
                return;
            }

            const auto flushedCount = FlushBacklog(shardId, shardKey, projectId, *sinkId);
            if (flushedCount > 0) {
                MON_INFO(Router, "Writing " << flushedCount << " backlog entries into the sink for " << shardId);
            }
        }

        void OnShardRemoved(const TShardId& shardId) {
            ShardMap_.erase(shardId);

            DropBacklog(shardId);
        }

        bool TryAssignShardId(TShardData& data) {
            TShardKey k{data.ProjectId, data.ClusterName, data.ServiceName};
            if (auto it = PcsToId_.find(k); it != PcsToId_.end()) {
                data.ShardId = it->second;
                return true;
            }

            return false;
        }

        TEvDataReceived::EResult WriteToBacklog(TEvSinkWrite& ev, TActorId sender) {
            auto& shardData = ev.ShardData;

            TEvDataReceived::EResult result;

            auto [it, isInserted] = WriteBacklog_.try_emplace(
                shardData.ShardId,
                MakeHolder<TShardQueue>(UrlBacklogLimit_, ShardBacklogLimit_, CreateFakeCounters())
            );

            if (isInserted) {
                MON_DEBUG(Router, "Created a backlog queue for " << shardData.ShardId);
            }

            auto& queue = *it->second;

            TQueueEntry entry{std::move(ev), sender};
            const auto dataSize = entry.Size();
            // we don't know the location of the shard yet -- put it into a backlog
            if (queue.Push(std::move(entry))) {
                Counters_.IncBacklog(dataSize);
                MON_INFO(Router, "Writing data for " << shardData.ShardId << " to backlog");
                result = TEvDataReceived::EResult::Postponed;
            } else {
                MON_WARN(Router, "Cannot write more data for " << shardData.ShardId << " to backlog, dropping");
                Counters_.IncDrop(shardData.ShardId);
                result = TEvDataReceived::EResult::Dropped;
            }

            return result;
        }

        void HandleUnknownShard(TEvSinkWrite& ev) {
            auto& data = ev.ShardData;
            TShardKey key{data.ProjectId, data.ClusterName, data.ServiceName};
            Send(UnknownShardHandlerId_, new TEvUnknownShard{key, data.ProviderAccountId});
            UnknownShardBacklog_.Add(std::move(key), std::move(ev));
        }

        void OnSinkWrite(TEvSinkWrite::TPtr evPtr) {
            auto& ev = *evPtr->Get();
            auto& data = ev.ShardData;

            TEvDataReceived::EResult result;
            Y_DEFER {
                Send(evPtr->Sender, new TEvDataReceived{result});
            };

            if (!data.ShardId.IsValid() && !TryAssignShardId(data)) {
                Y_VERIFY_DEBUG(data.ProjectId && data.ServiceName && data.ClusterName);
                result = TEvDataReceived::EResult::Postponed;
                HandleUnknownShard(ev);
                return;
            }

            if (auto it = ShardMap_.find(data.ShardId); it != ShardMap_.end()) {
                auto& info = it->second;
                TActivationContext::Send(evPtr->Forward(info.ActorId));
                result = TEvDataReceived::EResult::Written;
                return;
            }

            if (!data.ShardId.IsValid()) {
                HandleUnknownShard(ev);
                result = TEvDataReceived::EResult::Postponed;
            } else {
                result = WriteToBacklog(ev, evPtr->Sender);
            }
        }

    private:
        std::optional<TActorId> SinkForLocation(const TClusterNode& loc) {
            auto it = Sinks_.find(loc);

            if (Y_UNLIKELY(it == Sinks_.end())) {
                // looks like cluster is misconfigured
                return {};
            }

            return it->second;
        }

        void DropBacklog(const TShardId& shardId) {
            if (auto it = WriteBacklog_.find(shardId); it != WriteBacklog_.end()) {
                Counters_.DecBacklog(it->second->MemorySize());
                WriteBacklog_.erase(it);
                MON_WARN(Router, "Dropping backlog for " << shardId);
            }
        }

        ui32 FlushBacklog(const TShardId& shardId, const TShardKey& shardKey, const TString& projectId, TActorId sinkId) {
            ui32 i = 0;

            if (auto it = WriteBacklog_.find(shardId); it != WriteBacklog_.end()) {
                auto& q = it->second;
                for (; !q->IsEmpty(); ++i) {
                    auto entry = q->Pop();
                    Send(sinkId, CreateSinkWriteEvent(std::move(entry), shardId, projectId).Release());
                }

                WriteBacklog_.erase(it);
            }

            auto unknownQueue = UnknownShardBacklog_.Extract(shardKey);

            for (auto& entry: unknownQueue) {
                Send(sinkId, CreateSinkWriteEvent(std::move(entry), shardId, projectId).Release());
                ++i;
            }

            return i;
        }

    private:
        struct TShardInfo {
            TShardInfo(TClusterNode loc, TActorId actor, TString projectId)
                : Location{std::move(loc)}
                , ActorId{std::move(actor)}
                , ProjectId{std::move(projectId)}
            {
            }

            TClusterNode Location;
            TActorId ActorId;
            TString ProjectId;
        };

        THashMap<TClusterNode, TActorId> Sinks_;
        TShardMap<TShardInfo> ShardMap_;
        TShardMap<THolder<TShardQueue>> WriteBacklog_;
        absl::flat_hash_map<TShardKey, TShardId, TKeyHash, TKeyEq> PcsToId_;

        TCounters Counters_;

        ui32 UrlBacklogLimit_;
        ui32 ShardBacklogLimit_;

        TUnknownShardBacklog UnknownShardBacklog_;
        TActorId UnknownShardHandlerId_;
        const IClusterMap* Cluster_;
    };

} // namespace
    IActor* CreateRouterActor(TRouterActorConf conf, TMetricRegistry& registry) {
        return new TRouterActor{std::move(conf), registry};
    }

    TActorId MakeRouterServiceId() {
        static constexpr TStringBuf SRV_ID = "RouterSrv\0"sv;
        return TActorId(0, SRV_ID);
    }
} // namespace NSolomon::NFetcher
