#include "unknown_shard.h"

#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>

using namespace NActors;
using namespace NSolomon::NCoremon;

namespace NSolomon::NFetcher {
namespace {
    constexpr auto EvResponse = TEvents::ES_PRIVATE;
    struct TEvResponse: public TEventLocal<TEvResponse, EvResponse> {
        TEvResponse(TErrorOr<TShardCreated, TApiCallError>&& resp, TShardKey key, TActorId receiver)
            : Response{std::move(resp)}
            , ShardKey{std::move(key)}
            , Receiver{receiver}
        {
        }

        TErrorOr<TShardCreated, TApiCallError> Response;
        TShardKey ShardKey;
        TActorId Receiver;
    };

    constexpr auto RECORD_RETENTION = TDuration::Minutes(2);
    struct TRequestRecord {
        TRequestRecord(TInstant startTime, TActorId requester)
            : StartTime{startTime}
            , Requester{requester}
        {
        }

        TInstant StartTime;
        TActorId Requester;
    };

    class TShardCreatorActor: public TActorBootstrapped<TShardCreatorActor> {
    public:
        TShardCreatorActor(IProcessingClusterClientPtr client, const IClusterMap& cluster)
            : Client_{std::move(client)}
            , Cluster_{cluster}
        {
        }

        void Bootstrap() {
            Become(&TThis::StateFunc);
            PingSelf();
        }

        void PingSelf() {
            Schedule(RECORD_RETENTION / 5, new TEvents::TEvWakeup);
        }

        STFUNC(StateFunc) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvUnknownShard, OnUnknownShard);
                hFunc(TEvResponse, OnResponse);
                cFunc(TEvents::TSystem::PoisonPill, OnPoisonPill);
                cFunc(TEvents::TSystem::Wakeup, OnWakeup);
            }
        }

        STFUNC(StateDying) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvResponse, OnResponse);
            }
        }

        void OnPoisonPill() {
            IsDying_ = true;
            if (InFlightRequests_ == 0) {
                PassAway();
                return;
            }

            Become(&TThis::StateDying);
        }

        void OnUnknownShard(const TEvUnknownShard::TPtr& ev) {
            if (InFlight_.find(ev->Get()->ShardKey) != InFlight_.end()) {
                return;
            }

            auto client = Leader_.IsUnknown()
                ? Client_->GetAnyClient()
                : Client_->GetClient(Leader_);

            if (!client) {
                MON_ERROR(Router, "Failed to get a coremon client!");
                return;
            }

            TActorSystem* as = TActorContext::ActorSystem();

            auto key = ev->Get()->ShardKey;
            InFlight_.emplace(
                std::piecewise_construct,
                std::forward_as_tuple(key),
                std::forward_as_tuple(as->Timestamp(), ev->Sender)
            );

            auto self = SelfId();
            auto recv = ev->Sender;

            client->CreateShard(
                            key.ProjectId,
                            key.ServiceName,
                            key.ClusterName,
                            ev->Get()->CreatedBy)
                    .Subscribe([as, self, key, recv] (auto f) {
                        as->Send(self, new TEvResponse{f.ExtractValue(), key, recv});
                    });

            ++InFlightRequests_;
        }

        void OnResponse(TEvResponse::TPtr ev) {
            Y_VERIFY_DEBUG(InFlightRequests_ > 0);
            --InFlightRequests_;
            if (IsDying_ && InFlightRequests_ <= 0) {
                PassAway();
                return;
            }

            auto&& resp = ev->Get()->Response;
            if (resp.Fail()) {
                MON_ERROR(Router, "Failed to create shard " << ev->Get()->ShardKey << ": " << resp.Error().Message());
                return;
            }

            auto&& val = resp.Extract();
            auto leader = Cluster_.NodeByFqdn(val.Leader);
            if (leader) {
                Leader_ = *leader;
            } else {
                MON_ERROR(Router, "Unknown leader node FQDN " << val.Leader);
            }

            MON_INFO(Router, "Successfully created shard: " << val.ShardId);
            Send(ev->Get()->Receiver, new TEvShardCreated{
                TShardId{val.ShardId, val.NumId},
                std::move(ev->Get()->ShardKey),
                std::move(val.AssignedToHost)
            });
        }

        void OnWakeup() {
            auto now = TActivationContext::Now();
            TVector<TShardKey> toErase;
            for (auto it = InFlight_.begin(); it != InFlight_.end(); ++it) {
                if (it->second.StartTime + RECORD_RETENTION < now) {
                    toErase.push_back(it->first);
                }
            }

            for (auto&& key: toErase) {
                InFlight_.erase(key);
            }

            PingSelf();
        }

    private:
        IProcessingClusterClientPtr Client_;
        const IClusterMap& Cluster_;
        TClusterNode Leader_;
        int InFlightRequests_{0};
        bool IsDying_{false};

        // avoid duplicate requests
        THashMap<TShardKey, TRequestRecord> InFlight_;
    };

} // namespace
    IActor* CreateShardCreatorActor(IProcessingClusterClientPtr client, const IClusterMap& cluster) {
        return new TShardCreatorActor{std::move(client), cluster};
    }
} // namespace NSolomon::NFetcher
