#include <solomon/services/fetcher/lib/config_updater/config_updater.h>
#include <solomon/services/fetcher/lib/fetcher_shard.h>
#include <solomon/services/fetcher/lib/router/router.h>
#include <solomon/services/fetcher/lib/router/unknown_shard.h>
#include <solomon/services/fetcher/lib/sink/sink.h>
#include <solomon/services/fetcher/testlib/actor_system.h>
#include <solomon/services/fetcher/testlib/db.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/testlib/test_runtime.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/hash_set.h>
#include <util/string/cast.h>
#include <util/system/hostname.h>

#include <utility>

using namespace testing;
using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;

namespace NSolomon::NFetcher {
    std::ostream& operator<<(std::ostream& os, TEvDataReceived::EResult r) {
        switch (r) {
            case TEvDataReceived::EResult::Written:
                os << "Written";
                break;
            case TEvDataReceived::EResult::Postponed:
                os << "Postponed";
                break;
            case TEvDataReceived::EResult::Dropped:
                os << "Dropped";
                break;
        }

        return os;
    }
}

struct TMockShardUpdater: TActor<TMockShardUpdater> {
    TMockShardUpdater(TFetcherActorRuntime& runtime)
        : TActor<TMockShardUpdater>(&TThis::StateWork)
        , Runtime_{runtime}
    {
    }

    STFUNC(StateWork) {
        Y_UNUSED(ctx);
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvents::TEvSubscribe, OnSubscribe);
            hFunc(TEvents::TEvUnsubscribe, OnUnsubscribe);
        }
    }

    void SetShards(TVector<TFetcherShard> shards) {
        TVector<TFetcherShard> added;
        TVector<TFetcherShard> changed;
        TVector<TInfoOfAShardToRemove> removed;

        for (auto&& shard: shards) {
            auto it = FindIf(Shards_, [&shard] (auto&& s) {
                return s.Id() == shard.Id();
            });

            const auto contains = it != Shards_.end();

            if (contains) {
                changed.push_back(shard);
                Shards_.erase(it);
            } else {
                added.push_back(shard);
            }
        }

        for (const auto& shard: Shards_) {
            removed.emplace_back(shard.Id(), shard.Type());
        }

        Shards_ = std::move(shards);
        Send<TEvConfigChanged>(added, changed, removed);
    }

private:
    template <typename TEv, typename... TArgs>
    void Send(TArgs&&... args) {
        for (auto id: Subscribers_) {
            Runtime_.Send(new IEventHandle(id, TActorId(), new TEv{std::forward<TArgs>(args)...}));
        }
    }

    void OnSubscribe(const TEvents::TEvSubscribe::TPtr& ev) {
        Subscribers_.insert(ev->Sender);
        Runtime_.Send(new IEventHandle(ev->Sender, SelfId(), new TEvConfigChanged({Shards_}, {}, {})));
    }

    void OnUnsubscribe(const TEvents::TEvUnsubscribe::TPtr& ev) {
        Subscribers_.erase(ev->Sender);
    }

private:
    THashSet<TActorId> Subscribers_;
    TVector<TFetcherShard> Shards_;
    TFetcherActorRuntime& Runtime_;
};

class TRouterTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorRuntime_.Reset(new TFetcherActorRuntime);
        ShardUpdater_ = new TMockShardUpdater{*ActorRuntime_};
        ActorRuntime_->AddLocalService(
            MakeConfigUpdaterId(),
            TActorSetupCmd{ShardUpdater_, TMailboxType::Simple, 0}
        );

        ActorRuntime_->Initialize();
        UnknownHandler_ = ActorRuntime_->AllocateEdgeActor();

        PopulatePeers();

        TRouterActorConf conf {
            .UrlBacklogLimit = 0,
            .ShardBacklogLimit = 5,
            .Peers = Peers_,
            .UnknownShardHandlerId = UnknownHandler_,
        };

        Router_ = ActorRuntime_->Register(CreateRouterActor(conf, Registry_));
        ActorRuntime_->WaitForBootstrap();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

protected:
    THolder<TEvSinkWrite> MakeWrite(TShardId shardId, TStringBuf projectId, TString data) {
        auto writeEv = MakeHolder<TEvSinkWrite>();
        writeEv->ShardData.ShardId = std::move(shardId);
        writeEv->ShardData.ProjectId = ToString(projectId);
        writeEv->ShardData.Data = std::move(data);
        return writeEv;
    }

    void PopulatePeers() {
        LocalSink_ = ActorRuntime_->AllocateEdgeActor(0);

        Peers_ = {
            {Local(), LocalSink_},
            {Node(1), ActorRuntime_->AllocateEdgeActor(0)},
            {Node(2), ActorRuntime_->AllocateEdgeActor(0)},
        };
    }

    // NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
    TFetcherShard MakeShard(TClusterNode loc = Local(), const TShardId& shardId = {}) {
        auto shard = MakeAtomicShared<NDb::NModel::TShardConfig>(MakeShardConfig());
        if (shardId.IsValid()) {
            shard->Id = shardId.StrId();
            shard->NumId = shardId.NumId();
        } else {
            shard->NumId = 1;
        }

        auto cluster = MakeAtomicShared<NDb::NModel::TClusterConfig>(MakeClusterConfig());
        auto service = MakeAtomicShared<NDb::NModel::TServiceConfig>(MakeServiceConfig());

        return CreateSimpleShard(shard, cluster, service, std::move(loc));
    }

    static TClusterNode Local() {
        return {GetFQDNHostName(), 0, 0};
    }

    static TClusterNode Node(i32 nodeId) {
        return {TStringBuilder() << "host" << nodeId, nodeId, 0};
    }

protected:
    TMetricRegistry Registry_;
    THolder<TFetcherActorRuntime> ActorRuntime_;
    THashMap<TClusterNode, TActorId> Peers_;
    TActorId Router_;
    TActorId LocalSink_;
    TActorId UnknownHandler_;

    TMockShardUpdater* ShardUpdater_{nullptr};
    TShardId::TNumId NumId_{1};
};

TEST_F(TRouterTest, Simple) {
    auto fetcherShard = MakeShard();

    auto writeEv = MakeHolder<TEvSinkWrite>();
    writeEv->ShardData.ShardId = fetcherShard.Id();
    writeEv->ShardData.ProjectId = fetcherShard.ProjectId();
    writeEv->ShardData.Data = "hello";

    ShardUpdater_->SetShards({std::move(fetcherShard)});

    const auto sender = ActorRuntime_->AllocateEdgeActor(0);
    ActorRuntime_->Send(new IEventHandle(Router_, sender, writeEv.Release()));

    auto dataRecv = ActorRuntime_->GrabEdgeEvent<TEvDataReceived>(sender);
    ASSERT_THAT(dataRecv->Get()->Result, Eq(TEvDataReceived::EResult::Written));

    auto sinkWrite = ActorRuntime_->GrabEdgeEvent<TEvSinkWrite>(LocalSink_);
    ASSERT_THAT(sinkWrite->Get()->ShardData.Data, StrEq("hello"));
}

TEST_F(TRouterTest, ShardBacklogLimit) {
    auto fetcherShard = MakeShard();
    const auto sender = ActorRuntime_->AllocateEdgeActor(0);

    for (auto i = 0; i < 10; ++i) {
        auto writeEv = MakeHolder<TEvSinkWrite>();
        writeEv->ShardData.ShardId = fetcherShard.Id();
        writeEv->ShardData.ProjectId = fetcherShard.ProjectId();
        writeEv->ShardData.Data = "hello";

        ActorRuntime_->Send(new IEventHandle(Router_, sender, writeEv.Release()));

        auto dataRecv = ActorRuntime_->GrabEdgeEvent<TEvDataReceived>(sender);
        if (i > 4) {
            ASSERT_THAT(dataRecv->Get()->Result, Eq(TEvDataReceived::EResult::Dropped));
        } else {
            ASSERT_THAT(dataRecv->Get()->Result, Eq(TEvDataReceived::EResult::Postponed));
        }
    }
}

TEST_F(TRouterTest, BacklogIsFlushedWhenShardGetsKnown) {
    auto fetcherShard = MakeShard();
    const auto sender = ActorRuntime_->AllocateEdgeActor(0);

    constexpr auto MSG_COUNT = 5;

    // write some data to a shard which is not yet configured
    for (auto i = 0; i < MSG_COUNT; ++i) {
        auto writeEv = MakeWrite(
                fetcherShard.Id(),
                fetcherShard.ProjectId(),
                TStringBuilder() << "hello" << i);

        ActorRuntime_->Send(new IEventHandle(Router_, sender, writeEv.Release()));

        auto dataRecv = ActorRuntime_->GrabEdgeEvent<TEvDataReceived>(sender);
        ASSERT_THAT(dataRecv->Get()->Result, Eq(TEvDataReceived::EResult::Postponed));
    }

    // now configure this shard
    ShardUpdater_->SetShards({std::move(fetcherShard)});

    // expect that all messages from the backlog are flushed into a corresponding sink
    for (auto i = 0; i < MSG_COUNT; ++i) {
        auto sinkWrite = ActorRuntime_->GrabEdgeEvent<TEvSinkWrite>(LocalSink_);
        ASSERT_THAT(sinkWrite->Get()->ShardData.Data, StrEq(TStringBuilder() << "hello" << i));
    }
}

TEST_F(TRouterTest, ShardMovesBetweenNodes) {
    auto first = MakeShard(Node(1), {"first", 1});
    auto second = MakeShard(Node(2), {"second", 2});

    ShardUpdater_->SetShards({first, second});

    auto firstNodeSink = Peers_[Node(1)];
    auto secondNodeSink = Peers_[Node(2)];

    for (auto i = 0; i < 4; ++i) {
        auto& shard = (i & 1 == 0) ? first: second;
        TString data = TStringBuilder() <<  "hello" << i;

        auto writeEv = MakeWrite(shard.Id(), shard.ProjectId(), data);
        ActorRuntime_->Send(new IEventHandle(Router_, TActorId(), writeEv.Release()));

        auto sinkId = (i & 1 == 0) ? firstNodeSink : secondNodeSink;
        auto dataRecv = ActorRuntime_->GrabEdgeEvent<TEvSinkWrite>(sinkId);
        ASSERT_THAT(dataRecv->Get()->ShardData.Data, data);
        ASSERT_THAT(dataRecv->Get()->ShardData.ShardId, shard.Id());
    }

    // remove both shards -- writes should go to backlog now
    ShardUpdater_->SetShards({});

    const auto sender = ActorRuntime_->AllocateEdgeActor(0);
    for (auto i = 0; i < 4; ++i) {
        auto& shard = (i & 1 == 0) ? first: second;
        TString data = TStringBuilder() <<  "hello" << i;

        auto writeEv = MakeWrite(shard.Id(), shard.ProjectId(), data);
        ActorRuntime_->Send(new IEventHandle(Router_, sender, writeEv.Release()));

        auto dataRecv = ActorRuntime_->GrabEdgeEvent<TEvDataReceived>(sender);
        ASSERT_THAT(dataRecv->Get()->Result, Eq(TEvDataReceived::EResult::Postponed));
    }

    // now swap locations for the shards
    first = MakeShard(Node(2), {"first", 1});
    second = MakeShard(Node(1), {"second", 2});

    ShardUpdater_->SetShards({first, second});

    // wait messages from backlog
    for (auto i = 0; i < 4; ++i) {
        auto& shard = (i & 1 == 0) ? first: second;
        auto sinkId = (i & 1 == 0) ? secondNodeSink : firstNodeSink;
        TString data = TStringBuilder() <<  "hello" << i;

        auto dataRecv = ActorRuntime_->GrabEdgeEvent<TEvSinkWrite>(sinkId);
        ASSERT_THAT(dataRecv->Get()->ShardData.Data, data);
        ASSERT_THAT(dataRecv->Get()->ShardData.ShardId, shard.Id());
    }

    // write some new data for these shards
    for (auto i = 0; i < 4; ++i) {
        auto& shard = (i & 1 == 0) ? first: second;
        TString data = TStringBuilder() <<  "hello" << i;

        auto writeEv = MakeWrite(shard.Id(), shard.ProjectId(), data);
        ActorRuntime_->Send(new IEventHandle(Router_, TActorId(), writeEv.Release()));

        auto sinkId = (i & 1 == 0) ? secondNodeSink : firstNodeSink;
        auto dataRecv = ActorRuntime_->GrabEdgeEvent<TEvSinkWrite>(sinkId);
        ASSERT_THAT(dataRecv->Get()->ShardData.Data, data);
        ASSERT_THAT(dataRecv->Get()->ShardData.ShardId, shard.Id());
    }
}

TEST_F(TRouterTest, UnknownHandlerGetsCalled) {
    auto fetcherShard = MakeShard();

    auto writeEv = MakeHolder<TEvSinkWrite>();
    writeEv->ShardData.ServiceName = ToString(fetcherShard.ServiceName());
    writeEv->ShardData.ClusterName = ToString(fetcherShard.Cluster()->Name());
    writeEv->ShardData.ProjectId = fetcherShard.ProjectId();
    writeEv->ShardData.Data = "hello";
    ActorRuntime_->Send(new IEventHandle(Router_, {}, writeEv.Release()));

    auto ev = ActorRuntime_->GrabEdgeEvent<TEvUnknownShard>(UnknownHandler_);

    TShardKey expected{
            ToString(fetcherShard.ProjectId()),
            ToString(fetcherShard.Cluster()->Name()),
            ToString(fetcherShard.ServiceName()),
    };

    ASSERT_THAT(ev->Get()->ShardKey, Eq(expected));
}

TEST_F(TRouterTest, UnknownBacklogIsFlushed) {
    auto fetcherShard = MakeShard();

    auto writeEv = MakeHolder<TEvSinkWrite>();
    writeEv->ShardData.ServiceName = ToString(fetcherShard.ServiceName());
    writeEv->ShardData.ClusterName = ToString(fetcherShard.Cluster()->Name());
    writeEv->ShardData.ProjectId = fetcherShard.ProjectId();
    writeEv->ShardData.Data = "hello";
    ActorRuntime_->Send(new IEventHandle(Router_, {}, writeEv.Release()));

    auto ev = ActorRuntime_->GrabEdgeEvent<TEvUnknownShard>(UnknownHandler_);

    ShardUpdater_->SetShards({std::move(fetcherShard)});
    auto sinkWrite = ActorRuntime_->GrabEdgeEvent<TEvSinkWrite>(LocalSink_);
    ASSERT_THAT(sinkWrite->Get()->ShardData.Data, StrEq(TStringBuilder() << "hello"));
}

TEST_F(TRouterTest, WriteViaAgentShard) {
    auto fetcherShard = MakeShard();

    auto writeEv = MakeHolder<TEvSinkWrite>();
    writeEv->ShardData.ServiceName = ToString(fetcherShard.ServiceName());
    writeEv->ShardData.ClusterName = ToString(fetcherShard.Cluster()->Name());
    writeEv->ShardData.ProjectId = fetcherShard.ProjectId();
    writeEv->ShardData.Data = "hello";
    ShardUpdater_->SetShards({std::move(fetcherShard)});
    ActorRuntime_->Send(new IEventHandle(Router_, {}, writeEv.Release()));

    auto sinkWrite = ActorRuntime_->GrabEdgeEvent<TEvSinkWrite>(LocalSink_);
}
