#include <solomon/libs/cpp/cluster_map/cluster.h>
#include <solomon/services/fetcher/lib/router/unknown_shard.h>
#include <solomon/services/fetcher/testlib/actor_system.h>
#include <solomon/services/fetcher/testlib/coremon.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/stream/mem.h>
#include <util/system/hostname.h>

using namespace testing;
using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace NActors;
using namespace NMonitoring;

static const TString CLUSTER = TStringBuilder()
    << FQDNHostName() << ":0";

class TShardCreatorTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorRuntime_ = MakeActorSystem();
        ActorRuntime_->Initialize();

        TMemoryInput is{CLUSTER};
        TMockMapper m;
        Cluster_ = TClusterMapBase::Load(is, &m);
        MockCoremon_ = std::make_shared<TMockCoremonCluster>(Cluster_);
        TestActor_ = ActorRuntime_->Register(
            CreateShardCreatorActor(WrapClusterClient(MockCoremon_), *Cluster_)
        );

        ActorRuntime_->WaitForBootstrap();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

protected:
    TMockCoremonClient* LocalClient() {
        return static_cast<TMockCoremonClient*>(MockCoremon_->Get(Cluster_->Local().Endpoint));
    }

protected:
    TMetricRegistry Registry_;
    THolder<TFetcherActorRuntime> ActorRuntime_;
    std::shared_ptr<TMockCoremonCluster> MockCoremon_;
    TIntrusivePtr<TClusterMapBase> Cluster_;
    TActorId TestActor_;
};

TEST_F(TShardCreatorTest, MultipleCreations) {
    ActorRuntime_->Send(TestActor_, {}, THolder(new TEvUnknownShard{{"project1", "cluster", "service"}}));
    ActorRuntime_->Send(TestActor_, {}, THolder(new TEvUnknownShard{{"project", "cluster1", "service"}}));
    ActorRuntime_->Send(TestActor_, {}, THolder(new TEvUnknownShard{{"project", "cluster", "service1"}}));

    ASSERT_THAT(LocalClient()->CreateRequests, UnorderedElementsAreArray({
        TMockCoremonClient::TCreateShard{"project1", "cluster", "service", ""},
        TMockCoremonClient::TCreateShard{"project", "cluster1", "service", ""},
        TMockCoremonClient::TCreateShard{"project", "cluster", "service1", ""}
    }));
}

TEST_F(TShardCreatorTest, NoDoubleCreations) {
    ActorRuntime_->Send(TestActor_, {}, THolder(new TEvUnknownShard{{"project", "cluster", "service"}}));
    ASSERT_THAT(LocalClient()->CreateRequests, SizeIs(1));
    ASSERT_THAT(LocalClient()->CreateRequests[0].ProjectId, "project");
    ASSERT_THAT(LocalClient()->CreateRequests[0].ClusterName, "cluster");
    ASSERT_THAT(LocalClient()->CreateRequests[0].ServiceName, "service");

    ActorRuntime_->Send(TestActor_, {}, THolder(new TEvUnknownShard{{"project", "cluster", "service"}}));
    ASSERT_THAT(LocalClient()->CreateRequests, SizeIs(1));
}

TEST_F(TShardCreatorTest, NotifiesOnCreation) {
    const TShardKey shardKey{"project", "cluster", "service"};

    auto sender = ActorRuntime_->AllocateEdgeActor();
    ActorRuntime_->Send(TestActor_, sender, MakeHolder<TEvUnknownShard>(shardKey));

    auto evPtr = ActorRuntime_->GrabEdgeEvent<TEvShardCreated>(sender);
    auto&& ev = *evPtr->Get();
    auto expectedId = TShardId{"projectservicecluster", 42};
    ASSERT_THAT(ev.ShardId, expectedId);
    ASSERT_THAT(ev.ShardKey, shardKey);
    ASSERT_THAT(ev.Host, FQDNHostName());
}
