#include "counters.h"

#include <library/cpp/monlib/metrics/fake.h>
#include <library/cpp/monlib/metrics/metric_sub_registry.h>

#include <util/generic/algorithm.h>

using namespace NMonitoring;
using namespace yandex::solomon::common;

namespace NSolomon::NFetcher {
namespace {
    struct TShardMetrics: public IShardMetrics {
        struct TContentTypeCounters {
            void ReportUnspecified();
            void ReportFormat(EFormat fmt, ECompression compression);

            IRate* Json;
            IRate* Spack;
            IRate* SpackLz4;
            IRate* SpackZlib;
            IRate* SpackZstd;
            IRate* Unknown;

            IRate* NotSpecified;
        };

        struct TCounters: private TMoveOnly {
            std::shared_ptr<IMetricRegistry> Registry;

            std::array<IRate*, UrlStatusType_ARRAYSIZE> Metrics;
            IRate* MetricsParsed;
            IRate* ResponseBytes;

            IIntGauge* UrlCount;
            IIntGauge* Inflight;
            IIntGauge* Idle;
            IHistogram* DownloadDuration;
            IRate* Postponed;
            IRate* Success;
            IRate* Fail;
            IRate* Timeout;

            IHistogram* DecodeTime;
            IIntGauge* DecodeInflight;

            TContentTypeCounters ContentType;

            static std::shared_ptr<TCounters> ForTotal(IMetricRegistry* registry) {
                TMetricSubRegistry subRegistry{{{"projectId", "total"}, {"shardId", "total"}}, registry};
                auto counters = std::make_shared<TCounters>();
                counters->InitMetrics(&subRegistry);
                return counters;
            }

            static std::shared_ptr<TCounters> ForProject(IMetricRegistry* registry, TStringBuf projectId) {
                TMetricSubRegistry subRegistry{{{"projectId", projectId}, {"shardId", "total"}}, registry};
                auto counters = std::make_shared<TCounters>();
                counters->InitMetrics(&subRegistry);
                return counters;
            }

            static std::shared_ptr<TCounters> ForShard(TStringBuf projectId, TStringBuf shardId) {
                auto counters = std::make_shared<TCounters>();
                counters->Registry = std::make_shared<TMetricRegistry>(TLabels{
                    {"projectId", projectId},
                    {"shardId", shardId},
                });
                counters->InitMetrics(counters->Registry.get());
                return counters;
            }

        private:
            void InitMetrics(IMetricRegistry* registry) {
                ResponseBytes = registry->Rate(MakeLabels<TString>({{"sensor", "fetch.responseByteSize"}}));

                for (size_t i = 0; i < UrlStatusType_ARRAYSIZE; ++i) {
                    auto&& name = UrlStatusType_Name(UrlStatusType(i));
                    if (name.empty()) {
                        continue;
                    }

                    Metrics[i] = registry->Rate(MakeLabels<TString>({
                        {"sensor", "fetch.urlStatus"},
                        {"type", UrlStatusType_Name(UrlStatusType(i))}
                    }));
                }

                DownloadDuration = registry->HistogramRate(
                        MakeLabels<TString>({{"sensor", "fetch.downloadMillis"}}),
                        ExponentialHistogram(13, 2, 16));

                DecodeTime = registry->HistogramRate(
                        MakeLabels<TString>({{"sensor", "decode.us"}}),
                        ExponentialHistogram(20, 2, 16));

                DecodeInflight = registry->IntGauge(MakeLabels<TString>({{"sensor", "decode.inflight"}}));
                UrlCount = registry->IntGauge(MakeLabels<TString>({{"sensor", "fetch.urlCount"}}));
                Inflight = registry->IntGauge(MakeLabels<TString>({{"sensor", "fetch.inflight"}}));
                Idle = registry->IntGauge(MakeLabels<TString>({{"sensor", "fetch.idle"}}));
                Postponed = registry->Rate(MakeLabels<TString>({{"sensor", "fetch.postponed"}}));
                Success = registry->Rate(MakeLabels<TString>({{"sensor", "fetch.success"}}));
                Fail = registry->Rate(MakeLabels<TString>({{"sensor", "fetch.fail"}}));
                Timeout = registry->Rate(MakeLabels<TString>({{"sensor", "fetch.timeout"}}));

                ContentType.Json = registry->Rate(MakeLabels<TString>({
                    {"sensor", "fetch.contentType"},
                    {"type", "JSON"}
                }));

                ContentType.Spack = registry->Rate(MakeLabels<TString>({
                    {"sensor", "fetch.contentType"},
                    {"type", "SPACK"}
                }));

                ContentType.SpackLz4 = registry->Rate(MakeLabels<TString>({
                    {"sensor", "fetch.contentType"},
                    {"type", "SPACK_LZ4"}
                }));

                ContentType.SpackZlib = registry->Rate(MakeLabels<TString>({
                    {"sensor", "fetch.contentType"},
                    {"type", "SPACK_ZLIB"}
                }));

                ContentType.SpackZstd = registry->Rate(MakeLabels<TString>({
                    {"sensor", "fetch.contentType"},
                    {"type", "SPACK_ZSTD"}
                }));

                ContentType.Unknown = registry->Rate(MakeLabels<TString>({
                    {"sensor", "fetch.contentType"},
                    {"type", "UNKNOWN_CONTENT_TYPE"}
                }));

                ContentType.NotSpecified = registry->Rate(MakeLabels<TString>({
                    {"sensor", "fetch.unspecifiedContentType"}
                }));
            }
        };

        TShardMetrics(
                std::shared_ptr<TCounters> plain,
                std::shared_ptr<TCounters> project,
                std::shared_ptr<TCounters> total,
                std::shared_ptr<TDcCounters> dc);

        void ReportStatus(UrlStatusType status) override {
            const auto idx = static_cast<size_t>(status);

            if (Plain_) {
                Y_VERIFY_DEBUG(Plain_->Metrics[idx]);
                Plain_->Metrics[idx]->Inc();
            }

            Total_->Metrics[idx]->Inc();
            ProjectTotal_->Metrics[idx]->Inc();
        }

        void ReportMetricsParsed(size_t count, EDc dc) override {
            DcCounters_->ReportMetrics(dc, count);
        }

        void ReportResponseSize(size_t size, EDc dc, TStringBuf projectId) override {
            if (Plain_) {
                Plain_->ResponseBytes->Add(size);
            }
            Total_->ResponseBytes->Add(size);
            ProjectTotal_->ResponseBytes->Add(size);
            DcCounters_->ReportBytes(dc, size, projectId);
        }

        void IncDecodeInflight() override {
            if (Plain_) {
                Plain_->DecodeInflight->Inc();
            }
            ProjectTotal_->DecodeInflight->Inc();
            Total_->DecodeInflight->Inc();
        }

        void IncDecodeInflightMultishard() override {
            if (Plain_) {
                Plain_->DecodeInflight->Inc();
            }
            Total_->DecodeInflight->Inc();
            ProjectTotal_->DecodeInflight->Inc();
        }

        void DecDecodeInflight() override {
            if (Plain_) {
                Plain_->DecodeInflight->Dec();
            }
            ProjectTotal_->DecodeInflight->Dec();
            Total_->DecodeInflight->Dec();
        }

        void DecDecodeInflightMultishard() override {
            if (Plain_) {
                Plain_->DecodeInflight->Dec();
            }
            Total_->DecodeInflight->Dec();
            ProjectTotal_->DecodeInflight->Dec();
        }

        void DecodeTime(TDuration duration) override {
            if (Plain_) {
                Plain_->DecodeTime->Record(static_cast<double>(duration.MicroSeconds()));
            }
            Total_->DecodeTime->Record(static_cast<double>(duration.MicroSeconds()));
            ProjectTotal_->DecodeTime->Record(static_cast<double>(duration.MicroSeconds()));
        }

        void DecodeTimeMultishard(TDuration duration) override {
            if (Plain_) {
                Plain_->DecodeTime->Record(static_cast<double>(duration.MicroSeconds()));
            }
            Total_->DecodeTime->Record(static_cast<double>(duration.MicroSeconds()));
            ProjectTotal_->DecodeTime->Record(static_cast<double>(duration.MicroSeconds()));
        }

        void ReportUnspecified() override {
            if (Plain_) {
                Plain_->ContentType.ReportUnspecified();
            }
            Total_->ContentType.ReportUnspecified();
            ProjectTotal_->ContentType.ReportUnspecified();
        }

        void ReportFormat(EFormat fmt, ECompression compression) override {
            if (Plain_) {
                Plain_->ContentType.ReportFormat(fmt, compression);
            }
            Total_->ContentType.ReportFormat(fmt, compression);
            ProjectTotal_->ContentType.ReportFormat(fmt, compression);
        }

        void AddUrlCount(i64 val) override {
            if (Plain_) {
                Plain_->UrlCount->Add(val);
            }
            Total_->UrlCount->Add(val);
            ProjectTotal_->UrlCount->Add(val);
        }

        void AddInflight(i64 val) override {
            if (Plain_) {
                Plain_->Inflight->Add(val);
            }
            Total_->Inflight->Add(val);
            ProjectTotal_->Inflight->Add(val);
        }

        void AddIdle(i32 val) override {
            if (Plain_) {
                Plain_->Idle->Add(val);
            }
            Total_->Idle->Add(val);
            ProjectTotal_->Idle->Add(val);
        }

        void IncInflight() override {
            AddInflight(1);
        }

        void DecInflight() override {
            AddInflight(-1);
        }

        void RecordDownloadDuration(ui64 millis) override {
            if (Plain_) {
                Plain_->DownloadDuration->Record(static_cast<double>(millis));
            }
            Total_->DownloadDuration->Record(static_cast<double>(millis));
            ProjectTotal_->DownloadDuration->Record(static_cast<double>(millis));
        }

        void Postponed() override {
            if (Plain_) {
                Plain_->Postponed->Inc();
            }
            Total_->Postponed->Inc();
            ProjectTotal_->Postponed->Inc();
        }

        void DownloadSuccess() override {
            if (Plain_) {
                Plain_->Success->Inc();
            }
            Total_->Success->Inc();
            ProjectTotal_->Success->Inc();
        }

        void DownloadFail() override {
            if (Plain_) {
                Plain_->Fail->Inc();
            }
            Total_->Fail->Inc();
            ProjectTotal_->Fail->Inc();
        }

        void DownloadTimeout() override {
            if (Plain_) {
                Plain_->Timeout->Inc();
            }
            Total_->Timeout->Inc();
            ProjectTotal_->Timeout->Inc();
        }

        void Accept(TInstant time, IMetricConsumer* consumer) const override {
            if (Plain_) {
                Plain_->Registry->Accept(time, consumer);
            }
            // other metrics (project, total and dc) are reported by global registry
        }

        void Append(TInstant time, IMetricConsumer* consumer) const override {
            if (Plain_) {
                Plain_->Registry->Append(time, consumer);
            }
            // other metrics (project, total and dc) are reported by global registry
        }

    private:
        std::shared_ptr<TCounters> Plain_;
        std::shared_ptr<TCounters> ProjectTotal_;
        std::shared_ptr<TCounters> Total_;
        std::shared_ptr<TDcCounters> DcCounters_;
    };

    class TShardMetricFactory: public IShardMetricFactory {
    public:
        TShardMetricFactory(IMetricRegistry& registry, EMetricVerbosity verbosity)
            : Registry_{registry}
            , Total_{TShardMetrics::TCounters::ForTotal(&registry)}
            , DcCounters_{std::make_shared<TDcCounters>(registry)}
            , Verbosity_{verbosity}
        {
        }

        IShardMetricsPtr MetricsFor(const TString& projectId, const TString& shardId) override {
            return std::make_shared<TShardMetrics>(
                    GetShardCounters(projectId, shardId),
                    GetProjectCounters(projectId),
                    Total_,
                    DcCounters_);
        }

    private:
        std::shared_ptr<TShardMetrics::TCounters> GetShardCounters(TStringBuf projectId, TStringBuf shardId) {
            switch (Verbosity_) {
                case EMetricVerbosity::All:
                    return TShardMetrics::TCounters::ForShard(projectId, shardId);

                case EMetricVerbosity::OnlyTotal:
                    return nullptr;
            }

            Y_UNREACHABLE();
        }

        std::shared_ptr<TShardMetrics::TCounters> GetProjectCounters(TStringBuf projectId) {
            auto& projectCounters = ProjectCounterMap_[projectId];
            if (!projectCounters) {
                projectCounters = TShardMetrics::TCounters::ForProject(&Registry_, projectId);
            }
            return projectCounters;
        }

    private:
        IMetricRegistry& Registry_;
        std::shared_ptr<TShardMetrics::TCounters> Total_;
        THashMap<TString, std::shared_ptr<TShardMetrics::TCounters>> ProjectCounterMap_;
        std::shared_ptr<TDcCounters> DcCounters_;
        EMetricVerbosity Verbosity_;
    };

} // namespace

    TShardMetrics::TShardMetrics(
            std::shared_ptr<TCounters> plain,
            std::shared_ptr<TCounters> project,
            std::shared_ptr<TCounters> total,
            std::shared_ptr<TDcCounters> dc)
        : Plain_{std::move(plain)}
        , ProjectTotal_{std::move(project)}
        , Total_{std::move(total)}
        , DcCounters_{std::move(dc)}
    {
    }

    void TShardMetrics::TContentTypeCounters::ReportUnspecified() {
        NotSpecified->Inc();
    }

    void TShardMetrics::TContentTypeCounters::ReportFormat(EFormat fmt, ECompression compression) {
        if (fmt == EFormat::JSON) {
            Json->Inc();
        } else if (fmt == EFormat::SPACK) {
            switch (compression) {
            case ECompression::UNKNOWN:
            case ECompression::IDENTITY:
                Spack->Inc();
                break;
            case ECompression::ZLIB:
                SpackZlib->Inc();
                break;
            case ECompression::LZ4:
                SpackLz4->Inc();
                break;
            case ECompression::ZSTD:
                SpackZstd->Inc();
                break;
            }
        } else {
            Unknown->Inc();
        }
    }

    THolder<IShardMetricFactory> CreateShardMetricFactory(IMetricRegistry& registry, EMetricVerbosity verbosity) {
        return MakeHolder<TShardMetricFactory>(registry, verbosity);
    }
} // namespace NSolomon::NFetcher
