#pragma once

#include <solomon/services/fetcher/lib/metric_verbosity.h>

#include <solomon/libs/cpp/solomon_env/dc.h_serialized.h>
#include <solomon/libs/cpp/sync/rw_lock.h>
#include <solomon/protos/common/url_info.pb.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/metric.h>
#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/generic/serialized_enum.h>
#include <util/generic/map.h>

#include <array>

namespace NSolomon::NFetcher {
    using TPerDcRate = std::array<NMonitoring::IRate*, GetEnumItemsCount<EDc>()>;

    inline TPerDcRate CreatePerDcCounters(NMonitoring::IMetricFactory& factory, TStringBuf projectId, TStringBuf metric) {
        TPerDcRate arr;
        for (auto dc: GetEnumNames<EDc>()) {
            arr[static_cast<size_t>(dc.first)] = factory.Rate(NMonitoring::MakeLabels({
                {"sensor", metric},
                {"projectId", projectId},
                {"shardId", "total"},
                {"dc", dc.second},
            }));
        }

        return arr;
    }

    struct TDcCounters {
        static_assert(static_cast<size_t>(EDc::UNKNOWN) == 0);

        class TProjectCounters {
        public:
            TProjectCounters(NMonitoring::IMetricFactory& factory)
                : Factory_{factory}
            {
            }

            void Report(TStringBuf projectId, EDc dc, ui64 metrics) {
                {
                    auto counters = Counters_.Read();
                    if (auto it = counters->find(projectId); it != counters->end()) {
                        it->second[static_cast<size_t>(dc)]->Add(metrics);
                        return;
                    }
                }

                auto counters = Counters_.Write();
                auto [it, _] = counters->emplace(
                    TString{projectId},
                    CreatePerDcCounters(Factory_, projectId, TStringBuf("fetch.metricsWritten"))
                );
                it->second[static_cast<size_t>(dc)]->Add(metrics);
            }

        private:
            struct THash {
                using is_transparent = void;
                size_t operator()(const TString& s) const noexcept {
                    return absl::Hash<std::string_view>{}(s);
                }

                size_t operator()(TStringBuf s) const noexcept {
                    return absl::Hash<std::string_view>{}(s);
                }
            };

            struct TEq {
                using is_transparent = void;
                bool operator()(TStringBuf sb, TStringBuf s) const noexcept {
                    return sb == s;
                }

                bool operator()(TStringBuf sb, const TString& s) const noexcept {
                    return sb == s;
                }

                bool operator()(const TString& lhs, const TString& rhs) const noexcept {
                    return lhs == rhs;
                }
            };

            NSync::TLightRwLock<absl::flat_hash_map<TString, TPerDcRate, THash, TEq>> Counters_;
            NMonitoring::IMetricFactory& Factory_;
        };

        TDcCounters(NMonitoring::IMetricRegistry& registry)
            : PerProject{registry}
        {
            Metrics = CreatePerDcCounters(registry, "total", "fetch.metricsWritten");
            Bytes = CreatePerDcCounters(registry, "total", "fetch.bytesReceived");
        }

        void ReportMetrics(EDc dc, ui64 metricsWritten) {
            Metrics[static_cast<size_t>(dc)]->Add(metricsWritten);
        }

        void ReportBytes(EDc dc, ui64 bytes, TStringBuf projectId) {
            Bytes[static_cast<size_t>(dc)]->Add(bytes);
            if (projectId) {
                PerProject.Report(projectId, dc, bytes);
            }
        }

        TPerDcRate Metrics;
        TPerDcRate Bytes;
        TProjectCounters PerProject;
    };

    class IShardMetrics: public NMonitoring::IMetricSupplier {
    public:
        virtual ~IShardMetrics() = default;
        virtual void ReportStatus(yandex::solomon::common::UrlStatusType status) = 0;
        virtual void ReportMetricsParsed(size_t count, EDc dc) = 0;
        virtual void ReportResponseSize(size_t size, EDc dc, TStringBuf sb) = 0;

        virtual void ReportUnspecified() = 0;

        virtual void ReportFormat(NMonitoring::EFormat fmt, NMonitoring::ECompression compression) = 0;

        // multi shard methods also add metrics to plain and projectTotal counters, 'cause there are not as many
        // service providers as basic shards
        virtual void IncDecodeInflight() = 0;
        virtual void IncDecodeInflightMultishard() = 0;
        virtual void DecDecodeInflight() = 0;
        virtual void DecDecodeInflightMultishard() = 0;
        virtual void DecodeTime(TDuration duration) = 0;
        virtual void DecodeTimeMultishard(TDuration duration) = 0;

        virtual void AddUrlCount(i64 val) = 0;
        virtual void AddInflight(i64 val) = 0;
        virtual void IncInflight() = 0;
        virtual void DecInflight() = 0;
        virtual void AddIdle(i32 val) = 0;
        virtual void RecordDownloadDuration(ui64 millis) = 0;
        virtual void Postponed() = 0;
        virtual void DownloadSuccess() = 0;
        virtual void DownloadFail() = 0;
        virtual void DownloadTimeout() = 0;
    };

    using IShardMetricsPtr = std::shared_ptr<IShardMetrics>;

    // This class is not thread-safe
    struct IShardMetricFactory {
        virtual ~IShardMetricFactory() = default;
        virtual IShardMetricsPtr MetricsFor(const TString& projectId, const TString& shardId) = 0;
    };

    THolder<IShardMetricFactory> CreateShardMetricFactory(NMonitoring::IMetricRegistry& registry, EMetricVerbosity verbosity);

} // namespace NSolomon::NFetcher
