#include "shard_handler.h"

#include <solomon/services/fetcher/lib/url/url.h>

#include <util/string/builder.h>

namespace NSolomon::NFetcher {
    TFetcherShardHandler::TFetcherShardHandler(const TFetcherShard& shard, IClusterUpdateListener& listener)
        : Shard_{shard}
        , Listener_{listener}
    {
    }

    void TFetcherShardHandler::SetTicketProvider(NAuth::NTvm::ITicketProvider* ticketProvider) {
        TicketProvider_ = ticketProvider;
    }

    TString TFetcherShardHandler::AsString() const {
        TStringBuilder os;
        for (auto&& [group, hosts]: GroupToHosts_) {
            os << group->Name() << ":";
            for (auto&& [url, _]: hosts) {
                os << " " << url;
            }
        }

        return os;
    }

    void TFetcherShardHandler::DropRemovedGroups(const TResolveResults& cluster) {
        // to avoid multiple lookups for same keys we'll store iterators
        using TIter = decltype(GroupToHosts_)::iterator;

        TVector<TIter> toRemove;
        for (auto it = GroupToHosts_.begin(); it != GroupToHosts_.end(); ++it) {
            const auto found = FindIf(cluster.begin(), cluster.end(), [&] (auto&& group) {
                return group.first->Name() == it->first->Name();
            }) != cluster.end();

            if (found) {
                continue;
            }

            toRemove.push_back(it);
        }

        for (auto&& it: toRemove) {
            auto&& [resolver, hosts] = *it;
            for (auto&& host: hosts) {
                Listener_.OnHostRemoved(host.first, host.second);
            }

            FailScores_.erase(resolver);
            GroupToHosts_.erase(it);
        }
    }

    void TFetcherShardHandler::AddHost(const THostAndLabels& hostAndLabels, const IHostGroupResolverPtr& resolver) {
        auto url = MakeUrl(Shard_, hostAndLabels);

        GroupToHosts_[resolver].emplace(url, hostAndLabels);
        Listener_.OnHostAdded(url, hostAndLabels);
    }

    void TFetcherShardHandler::RemoveHost(const TString& url, const IHostGroupResolverPtr& resolver) {
        auto& group = GroupToHosts_.at(resolver);
        auto it = group.find(url);

        if (it != group.end()) {
            Listener_.OnHostRemoved(it->first, it->second);
            group.erase(it);
        }
    }

    void TFetcherShardHandler::UpdateCluster(const TResolveResults& cluster) {
        DropRemovedGroups(cluster);

        TVector<ui32> tvmIds;
        if (auto tvmId = Shard_.TvmId()) {
            tvmIds.emplace_back(*tvmId);
        }

        for (auto&& [resolver, result]: cluster) {
            if (result.Fail()) {
                auto&& name = resolver->Name();
                auto&& msg = result.Error().Message();

                auto& failScore = FailScores_[resolver];
                failScore.IncFail();
                Listener_.OnResolveError(name, msg, failScore);
                continue;
            } else {
                FailScores_[resolver].IncSuccess();
            }

            auto& currentHosts = GroupToHosts_[resolver];
            auto newHosts = result.Value();

            TVector<const TString*> removed;
            for (auto&& [url, hostAndLabels]: currentHosts) {
                auto it = newHosts.find(hostAndLabels);
                if (it == newHosts.end()) {
                    removed.emplace_back(&url);
                } else {
                    newHosts.erase(it);
                }
            }

            for (auto&& r: removed) {
                RemoveHost(*r, resolver);
            }

            // these are the newly added ones
            for (auto& host: newHosts) {
                if (host.TvmDestId) {
                    tvmIds.emplace_back(*host.TvmDestId);
                }

                AddHost(std::move(host), resolver);
            }

            if (!tvmIds.empty() && TicketProvider_ != nullptr) {
                TicketProvider_->AddDestinationIds(std::move(tvmIds));
                tvmIds = {};
            }
        }
    }
} // namespace NSolomon::NFetcher
