#pragma once

#include "shard_resolver.h"

#include <solomon/services/fetcher/lib/host_groups/host_and_labels.h>

#include <util/generic/vector.h>

namespace NSolomon::NFetcher {
    class TFailScore {
    public:
        void IncSuccess() {
            ++Total_;
        }

        void IncFail() {
            ++Total_;
            ++Fail_;
        }

        double Rate() const {
            return double(Fail_) / double(Total_);
        }

        size_t Total() const {
            return Total_;
        }

    private:
        size_t Total_{0};
        size_t Fail_{0};
    };

    struct IClusterUpdateListener {
        virtual ~IClusterUpdateListener() = default;
        virtual void OnHostAdded(const TString& url, const THostAndLabels& host) = 0;
        virtual void OnHostRemoved(const TString& url, const THostAndLabels& host) = 0;
        virtual void OnResolveError(TStringBuf resolverName, TStringBuf message, const TFailScore& failScore) = 0;
    };

    class TFetcherShardHandler {
    public:
        TFetcherShardHandler(const TFetcherShard& shard, IClusterUpdateListener& listener);
        void UpdateCluster(const TResolveResults& result);

        TString AsString() const;

        const auto& FailScores() const {
            return FailScores_;
        }

        const auto& GroupToHosts() const {
            return GroupToHosts_;
        }

        void SetTicketProvider(NAuth::NTvm::ITicketProvider* ticketProvider);

    private:
        void RemoveHost(const TString& url, const IHostGroupResolverPtr& resolver);
        void AddHost(const THostAndLabels& hostAndLabels, const IHostGroupResolverPtr& resolver);
        void DropRemovedGroups(const TResolveResults& cluster);

        const TFetcherShard& Shard_;
        IClusterUpdateListener& Listener_;
        NAuth::NTvm::ITicketProvider* TicketProvider_{nullptr};

        struct TCompareByName {
            bool operator()(const IHostGroupResolverPtr& rhs, const IHostGroupResolverPtr& lhs) const {
                return lhs->Name() == rhs->Name();
            }
        };

        struct TNameHash {
            size_t operator()(const IHostGroupResolverPtr& r) const {
                return ::THash<TString>()(r->Name());
            }
        };

        using TGroupMap = THashMap<IHostGroupResolverPtr, THashMap<TString, THostAndLabels>, TNameHash, TCompareByName>;
        using TFailScoreMap = THashMap<IHostGroupResolverPtr, TFailScore, TNameHash, TCompareByName>;

        TGroupMap GroupToHosts_;
        TFailScoreMap FailScores_;
    };
} // namespace NSolomon::NFetcher
