#include "shard_manager.h"
#include "shard_resolver.h"
#include "shard_stat.h"
#include "events.h"
#include "local_events.h"
#include "counters.h"
#include "shard_handler.h"

#include <solomon/services/fetcher/lib/app_data.h>
#include <solomon/services/fetcher/lib/config_updater/config_updater.h>
#include <solomon/services/fetcher/lib/dns/continuous_resolver.h>
#include <solomon/services/fetcher/lib/fetcher_shard.h>
#include <solomon/services/fetcher/lib/url/url.h>

#include <solomon/libs/cpp/http/server/handlers/metrics.h>
#include <solomon/libs/cpp/limiter/limiter.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <util/generic/hash.h>
#include <util/generic/ptr.h>
#include <util/string/cast.h>

#include <utility>

#define TRACING_OUTPUT(ctxOrAs, msg) \
    do { \
        if (Y_UNLIKELY(Tracing_)) { \
            MON_INFO_C(ctxOrAs, Tracing, msg); \
        } \
    } while (0)

using namespace NActors;
using namespace NMonitoring;
using namespace NThreading;

namespace NSolomon::NFetcher {
namespace {
    ILimiterPtr CreateLimiter(ui64 limit) {
        return limit == 0
            ? CreateFakeLimiter()
            : NSolomon::CreateLimiter(limit);
    }

    struct TUrlData {
        TUrlData(THostAndLabels hostAndLabels)
            : HostAndLabels{std::move(hostAndLabels)}
        {
        }

        THostAndLabels HostAndLabels;
    };

    class TFetcherShardActor: public TActorBootstrapped<TFetcherShardActor>, public IClusterUpdateListener {
    public:
        explicit TFetcherShardActor(TShardActorConfig conf)
            : DataSinkId_{conf.DataSinkId}
            , ContinuousResolverId_{conf.ContinuousResolverId}
            , StatCollectorId_{conf.StatCollectorId}
            , AuthGatekeeperId_{conf.AuthGatekeeperId}
            , ClusterManagerId_{conf.ClusterManagerId}
            , ShardConf_{std::move(conf.ShardConf)}
            , Handler_{ShardConf_, *this}
            , ShardId_{ShardConf_.Id()}
            , Metrics_{std::move(conf.Metrics)}
            , UrlFactory_{conf.UrlFactory}
            , Limiter_{conf.Limiter ? conf.Limiter : CreateFakeLimiter()}
        {
        }

        void Bootstrap() {
            Become(&TThis::StateWork);
            Send(ClusterManagerId_, new TEvResolveCluster{ShardConf_.Cluster()});

            const TAppData& appData = GetAppData();
            Handler_.SetTicketProvider(appData.TicketProvider.Get());
            UrlDownloadPoolId_ = appData.FetchConfig().DownloadPoolId;
            UrlParsePoolId_ = appData.FetchConfig().ParsePoolId;
        }

        STFUNC(StateWork) {
            switch (ev->GetTypeRewrite()) {
                CFunc(TEvents::TSystem::PoisonPill, OnPoisonPill);
                hFunc(TEvClusterResolved, OnClusterResolved);
                CFunc(TShardEvents::EvStartTracing, OnStartTracing);
                CFunc(TShardEvents::EvStopTracing, OnStopTracing);
                hFunc(TEvShardChanged, OnShardChanged);
                hFunc(NSelfMon::TEvPageDataReq, OnSelfMon);
            }
        }

        void OnSelfMon(NSelfMon::TEvPageDataReq::TPtr& ev) {
            if (TString url = TString(ev->Get()->Param("url")); !url.empty()) {
                CGIUnescape(url);
                if (auto it = UrlActorMap_.find(url); it != UrlActorMap_.end()) {
                    TActivationContext::Send(ev->Forward(it->second));
                    return;
                }

                ::yandex::monitoring::selfmon::Page page;
                page.set_title(TStringBuilder{} << "Shard " << ShardId_);
                auto* code = page.mutable_grid()->add_rows()->add_columns()->mutable_component()->mutable_code();
                code->set_content("Cannot find URL " + url);
                Send(ev->Sender, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
                return;
            }

            ::yandex::monitoring::selfmon::Page page;
            page.set_title(TStringBuilder{} << "Shard " << ShardId_);
            auto* grid = page.mutable_grid();

            auto activeTab = ev->Get()->Param("tab");
            auto* tabs = grid->add_rows()->add_columns()->mutable_component()->mutable_tabs();
            if (auto* tab = tabs->add_tabs()) {
                tab->set_active(activeTab.empty() || activeTab == "urls");
                if (auto* ref = tab->mutable_reference()) {
                    ref->set_title("URLs");
                    ref->set_page("/shards");
                    ref->set_args("shardId=" + ShardId_.StrId() + "&tab=urls");
                }
            }
            if (auto* tab = tabs->add_tabs()) {
                tab->set_active(activeTab == "config");
                if (auto* ref = tab->mutable_reference()) {
                    ref->set_title("Config");
                    ref->set_page("/shards");
                    ref->set_args("shardId=" + ShardId_.StrId() + "&tab=config");
                }
            }

            if (activeTab.empty() || activeTab == "urls") {
                auto* table = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
                table->set_numbered(true);

                auto* urlColumn = table->add_columns();
                urlColumn->set_title("URL");
                auto* urlValues = urlColumn->mutable_reference();

                for (const auto& [url, _]: UrlActorMap_) {
                    auto* ref = urlValues->add_values();
                    ref->set_title(url);
                    ref->set_page("/shards");
                    ref->set_args(TStringBuilder{} << "shardId=" << ShardId_.StrId() << "&url=" << CGIEscapeRet(url));
                }
            }

            if (activeTab == "config") {
                auto* code = grid->add_rows()->add_columns()->mutable_component()->mutable_code();
                TStringStream ss;
                ShardConf_.DumpConfig(&ss);
                code->set_lang("yaml");
                code->set_content(ss.Str());
            }

            Send(ev->Sender, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        }

        void OnStartTracing(const TActorContext& ctx) {
            Tracing_ = true;

            TStringStream os;
            os << ShardId_ << " groups:\n";
            os << Handler_.AsString();

            TRACING_OUTPUT(ctx, os.Str());

            for (auto&& [_, aid]: UrlActorMap_) {
                Send(aid, new TEvStartTracing{{}});
            }
        }

        void OnStopTracing(const TActorContext&) {
            Tracing_ = false;

            for (auto&& [_, aid]: UrlActorMap_) {
                Send(aid, new TEvStopTracing{{}});
            }
        }

        void OnPoisonPill(const TActorContext&) {
            for (auto&& [_, urlActor]: UrlActorMap_) {
                Send(urlActor, new TEvents::TEvPoisonPill);
            }

            Send(ClusterManagerId_, new TEvClusterUnsubscribe{ShardConf_.Cluster()->Id()});
            PassAway();
        }

        void OnClusterResolved(const TEvClusterResolved::TPtr& ev) {
            auto&& cluster = *ev->Get()->Result;
            Handler_.UpdateCluster(cluster);
        }

        void OnHostAdded(const TString& url, const THostAndLabels& hostAndLabels) override {
            auto [it, isNew] = UrlActorMap_.emplace(url, TActorId{});

            if (!isNew) {
                MON_INFO(FetcherShard, "Duplicate URL in shard " << ShardConf_.Id() << ": " << url);
                return;
            }

            auto urlParser = UrlFactory_->CreateUrlParser(ShardConf_, hostAndLabels.Host, Metrics_);
            auto urlParserId = Register(urlParser.release(), TMailboxType::HTSwap, UrlParsePoolId_);

            auto* urlActor = CreateUrlActor(
                    hostAndLabels,
                    ShardConf_,
                    *UrlFactory_,
                    urlParserId,
                    ContinuousResolverId_,
                    DataSinkId_,
                    StatCollectorId_,
                    AuthGatekeeperId_,
                    Metrics_,
                    Limiter_,
                    GetAppData().DataHttpClient());
            it->second = Register(urlActor, TMailboxType::HTSwap, UrlDownloadPoolId_);

            if (Y_UNLIKELY(Tracing_)) {
                Send(it->second, new TEvStartTracing{{}});
            }
        }

        void OnHostRemoved(const TString& url, const THostAndLabels&) override {
            auto it = UrlActorMap_.find(url);

            Y_VERIFY_DEBUG(it != UrlActorMap_.end(), "No actor for %s", url.c_str());
            if (Y_LIKELY(it != UrlActorMap_.end())) {
                Send(it->second, new TEvents::TEvPoisonPill);
                UrlActorMap_.erase(it);
            }
        }

        void OnResolveError(TStringBuf resolverName, TStringBuf message, const TFailScore& failScore) override {
            if ((failScore.Total() == 1) || (failScore.Rate() < 0.8)) {
                MON_WARN(FetcherShard, "Failed to resovle " << resolverName << ": " << message);
            }
        }

    private:
        void OnShardChanged(const TEvShardChanged::TPtr& ev) {
            auto& shard = ev->Get()->Shard;
            Y_VERIFY_DEBUG(shard.Id() == ShardConf_.Id());
            ShardConf_ = std::move(shard);

            for (auto&& [_, urlActor]: UrlActorMap_) {
                Send(urlActor, new TEvShardChanged{ShardConf_});
            }
        }

    private:
        TActorId DataSinkId_;
        TActorId ContinuousResolverId_;
        TActorId StatCollectorId_;
        TActorId AuthGatekeeperId_;
        TActorId ClusterManagerId_;

        TFetcherShard ShardConf_;
        TFetcherShardHandler Handler_;
        TShardId ShardId_;

        ui32 UrlDownloadPoolId_{0};
        ui32 UrlParsePoolId_{0};
        THashMap<TString, TActorId> UrlActorMap_;

        IShardMetricsPtr Metrics_;
        std::shared_ptr<IFetcherUrlFactory> UrlFactory_;
        ILimiterPtr Limiter_;
        bool Tracing_{false};
    };

    class TShardManager: public TActorBootstrapped<TShardManager> {
    public:
        explicit TShardManager(const TShardManagerConfig& config)
            : SinkId_{config.DataSinkId}
            , AuthGatekeeperId_{config.AuthGatekeeperId}
            , ClusterManagerId_{config.ClusterManagerId}
            , Limiter_{NFetcher::CreateLimiter(config.MaxInflight)}
            , YasmWhiteList_(config.YasmWhiteList)
            , YasmPrefix_(config.YasmPrefix)
            , IamServiceId_(config.IamServiceId)
        {
        }

        void Bootstrap() {
            NSelfMon::RegisterPage(*TActorContext::ActorSystem(), "/shards", "Shards", SelfId());

            Become(&TThis::StateFunc);
            Send(MakeConfigUpdaterId(), new TEvents::TEvSubscribe);

            const TAppData& appData = GetAppData();
            UrlFactory_ = CreateFetcherUrlFactory(
                appData.TicketProvider,
                GetIamTokenProvider(appData),
                appData.ClusterInfo(),
                YasmWhiteList_,
                YasmPrefix_);
            MetricFactory_ = CreateShardMetricFactory(*appData.Metrics, appData.MetricVerbosity());
        }

        STFUNC(StateFunc) {
            switch (ev->GetTypeRewrite()) {
                HFunc(TEvConfigChanged, OnConfigChanged);

                HFunc(TEvShardStatsRequest, OnApiRequest);
                HFunc(TEvShardsHealthRequest, OnApiRequest);
                HFunc(TEvTargetsStatusRequest, OnApiRequest);
                hFunc(NSelfMon::TEvPageDataReq, OnSelfMon);

                HFunc(TEvStartTracing, OnTracingRequest<TEvStartTracing>);
                HFunc(TEvStopTracing, OnTracingRequest<TEvStopTracing>);
            }
        }

    private:
        NCloud::ITokenProviderPtr GetIamTokenProvider(const TAppData& appData) const {
            if (IamServiceId_.empty()) {
                return {};
            }
            auto iamTokenProvider = appData.GetIamTokenProvider(IamServiceId_);
            Y_VERIFY(iamTokenProvider, "IAM token provider for account id %s doesn't exist", IamServiceId_.c_str());
            return iamTokenProvider;
        }

        void OnSelfMon(NSelfMon::TEvPageDataReq::TPtr& ev) {
            if (TString shardId = TString(ev->Get()->Param("shardId")); !shardId.empty()) {
                if (auto numIdIt = StrToNumId_.find(shardId); numIdIt != StrToNumId_.end()) {
                    TShardId id{shardId, numIdIt->second};
                    if (auto it = ShardInfo_.find(id); it != ShardInfo_.end()) {
                        TActivationContext::Send(ev->Forward(it->second.FetcherShardId));
                        return;
                    }
                }

                if (auto it = ServiceProviderShardsInfo_.find(shardId); it != ServiceProviderShardsInfo_.end()) {
                    TActivationContext::Send(ev->Forward(it->second.FetcherShardId));
                    return;
                }

                ::yandex::monitoring::selfmon::Page page;
                page.set_title("Shards");
                auto* code = page.mutable_grid()->add_rows()->add_columns()->mutable_component()->mutable_code();
                code->set_content("Cannot find shard " + shardId);
                Send(ev->Sender, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
                return;
            }

            ::yandex::monitoring::selfmon::Page page;
            page.set_title("Shards");
            auto* grid = page.mutable_grid();

            auto* table = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
            table->set_numbered(true);

            auto* numIdColumn = table->add_columns();
            numIdColumn->set_title("NumId");
            auto* numIdValues = numIdColumn->mutable_int64();

            auto* shardIdColumn = table->add_columns();
            shardIdColumn->set_title("ShardId");
            auto* shardIdValues = shardIdColumn->mutable_reference();

            auto* urlCountColumn = table->add_columns();
            urlCountColumn->set_title("URL Count");
            auto* urlCountValues = urlCountColumn->mutable_uint64();

            auto addShards = [&]<typename TContainer>(const TContainer& shards, auto retrieveNumId, auto retrieveStrId) {
                for (const auto& [id, info]: shards) {
                    numIdValues->add_values(retrieveNumId(id));
                    auto strId = retrieveStrId(id);

                    auto* ref = shardIdValues->add_values();
                    ref->set_title(strId);
                    ref->set_page("/shards");
                    ref->set_args("shardId=" + strId);

                    auto stats = info.StatCollector->ShardStats();
                    urlCountValues->add_values(stats.UrlCount);
                }
            };

            addShards(ShardInfo_, [](const auto& id) { return id.NumId(); }, [](const auto& id) { return id.StrId(); });
            addShards(ServiceProviderShardsInfo_, [](const auto&) { return 0; }, std::identity{});

            Send(ev->Sender, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        }

        void OnApiRequest(const TEvShardStatsRequest::TPtr& ev, const TActorContext&) {
            TVector<TShardStats> stats;
            stats.reserve(ShardInfo_.size() + ServiceProviderShardsInfo_.size());

            for (auto&& it: ShardInfo_) {
                stats.push_back(it.second.StatCollector->ShardStats());
            }

            for (auto&& it: ServiceProviderShardsInfo_) {
                stats.push_back(it.second.StatCollector->ShardStats());
            }

            Send(ev->Sender, new TEvShardStatsResponse{std::move(stats)});
        }

        void OnApiRequest(const TEvShardsHealthRequest::TPtr& ev, const TActorContext&) {
            TVector<TShardHealth> healths;
            healths.reserve(ShardInfo_.size() + ServiceProviderShardsInfo_.size());

            for (auto&& it: ShardInfo_) {
                healths.push_back(it.second.StatCollector->ShardHealth());
            }

            for (auto&& it: ServiceProviderShardsInfo_) {
                healths.push_back(it.second.StatCollector->ShardHealth());
            }

            Send(ev->Sender, new TEvShardsHealthResponse{std::move(healths)});
        }

        void OnApiRequest(const TEvTargetsStatusRequest::TPtr& ev, const TActorContext& ctx) {
            const TShardInfo* info;
            auto retrieveShardInfo = [&]<typename TContainer>(const TContainer& c, const auto& key) {
                if (auto it = c.find(key); it != c.end()) {
                    info = &(it->second);
                } else {
                    MON_WARN(FetcherShard, "Requested status for " << key << " but it's not known");
                }
            };

            if (ev->Get()->NumId) {
                retrieveShardInfo(ShardInfo_, ev->Get()->NumId);
            } else if (ev->Get()->ServiceProviderId) {
                retrieveShardInfo(ServiceProviderShardsInfo_, MakeAgentShardId(ev->Get()->ServiceProviderId));
            } else {
                MON_WARN(FetcherShard, "Neither NumId or ServiceProviderId is specified");
            }

            if (!info) {
                Send(ev->Sender, new TEvNotFound);
                return;
            }

            ctx.Send(ev->Forward(info->StatCollectorId));
        }

        template <typename TEv>
        void OnTracingRequest(const typename TEv::TPtr& ev, const TActorContext& ctx) {
            const auto it = FindIf(ShardInfo_, [&] (auto&& kv) {
                return kv.first.StrId() == ev->Get()->ShardId;
            });

            if (it == ShardInfo_.end()) {
                Send(ev->Sender, new TEvNotFound);
                return;
            }

            ctx.Send(ev->Forward(it->second.FetcherShardId));
            Send(ev->Sender, new TEvTracingOk);
        }

        void OnConfigChanged(const TEvConfigChanged::TPtr& ev, const TActorContext& ctx) {
            for (auto&& shard: ev->Get()->Added) {
                OnShardAdded(shard, ctx);
            }

            for (auto&& shard: ev->Get()->Changed) {
                OnShardChanged(shard, ctx);
            }

            for (auto&& shardInfo: ev->Get()->Removed) {
                OnShardRemoved(shardInfo);
            }
        }

        void OnShardAdded(const TFetcherShard& shard, const TActorContext& ctx) {
            const auto isKnown = shard.Type() == EFetcherShardType::Agent
                    ? ServiceProviderShardsInfo_.find(shard.Id().StrId()) != ServiceProviderShardsInfo_.end()
                    : ShardInfo_.find(shard.Id()) != ShardInfo_.end();

            if (!isKnown && shard.IsLocal()) {
                AddShard(shard, ctx);
            }
        }

        void OnShardRemoved(const TInfoOfAShardToRemove& shardInfo) {
            bool wasThereSmthToRemove = shardInfo.Type == EFetcherShardType::Agent
                    ? RemoveServiceProviderShard(shardInfo.Id.StrId())
                    : RemoveShard(shardInfo.Id);

            if (!wasThereSmthToRemove) {
                return;
            }

            MON_DEBUG(FetcherShard, "Killing the actor for shard " << shardInfo.Id);
        }

        void OnShardChanged(TFetcherShard shard, const TActorContext& ctx) {
            TShardInfo* shardInfo;
            bool isKnown = false;
            bool isServiceProviderShard = shard.Type() == EFetcherShardType::Agent;

            if (isServiceProviderShard) {
                auto it = ServiceProviderShardsInfo_.find(shard.Id().StrId());

                if (it != ServiceProviderShardsInfo_.end()) {
                    shardInfo = std::addressof(it->second);
                    isKnown = true;
                }
            } else {
                auto it = ShardInfo_.find(shard.Id());

                if (it != ShardInfo_.end()) {
                    shardInfo = std::addressof(it->second);
                    isKnown = true;
                }
            }

            // moved to this node
            if (!isKnown && shard.IsLocal()) {
                AddShard(shard, ctx);
            // moved from this node
            } else if (isKnown && !shard.IsLocal()) {
                MON_DEBUG(FetcherShard, "Shard " << shard.Id() << " moved away from this host");

                if (isServiceProviderShard) {
                    RemoveServiceProviderShard(shard.Id().StrId());
                } else {
                    RemoveShard(shard.Id());
                }
            // same location, but some settings have changed
            } else if (isKnown && shard.IsLocal()) {
                MON_DEBUG(FetcherShard, "Notifying about shard change " << shard.Id());
                Send(shardInfo->FetcherShardId, new TEvShardChanged{shard});
            }
        }

        bool RemoveServiceProviderShard(const TString& shardStrId) {
            auto it = ServiceProviderShardsInfo_.find(shardStrId);
            if (it == ServiceProviderShardsInfo_.end()) {
                return false;
            }

            it->second.Kill(TActorContext::AsActorContext());
            ServiceProviderShardsInfo_.erase(it);

            return true;
        }

        bool RemoveShard(const TShardId& shardId) {
            auto it = ShardInfo_.find(shardId);
            if (it == ShardInfo_.end()) {
                return false;
            }

            it->second.Kill(TActorContext::AsActorContext());
            ShardInfo_.erase(it);
            StrToNumId_.erase(shardId.StrId());

            return true;
        }

    private:
        void AddShard(TFetcherShard shard, const TActorContext& ctx) {
            Y_VERIFY_DEBUG(shard.IsValid());
            if (!shard.IsPull()) {
                return;
            }

            const auto shardId = shard.Id();
            auto&& strId = shardId.StrId();
            bool isServiceProviderShard = shard.Type() == EFetcherShardType::Agent;

            if (!isServiceProviderShard) {
                StrToNumId_[strId] = shardId.NumId();
            }

            auto metrics = MetricFactory_->MetricsFor(ToString(shard.ProjectId()), shardId.StrId());
            NSolomon::NHttp::RegisterMetricSupplier(std::static_pointer_cast<IMetricSupplier>(metrics));

            TShardActorConfig conf{
                .ShardConf = std::move(shard),
                .Metrics = metrics,
                .UrlFactory = UrlFactory_,
                .Limiter = Limiter_,
                .DataSinkId = SinkId_,
                .ContinuousResolverId = MakeDnsResolverId(),
                .AuthGatekeeperId = AuthGatekeeperId_,
                .ClusterManagerId = ClusterManagerId_,
            };

            auto [collector, collectorId] = CreateShardStatCollector(ctx, shardId.NumId());
            conf.StatCollectorId = collectorId;

            const auto shardActorId = Register(new TFetcherShardActor{conf});

            TShardInfo info {
                .FetcherShardId = shardActorId,
                .StatCollectorId = conf.StatCollectorId,
                .ShardMetrics = metrics,
                .StatCollector = collector,
            };

            bool isNew = isServiceProviderShard
                    ? ServiceProviderShardsInfo_.emplace(shardId.StrId(), std::move(info)).second
                    : ShardInfo_.emplace(shardId, std::move(info)).second;
            Y_VERIFY_DEBUG(isNew);
            Y_UNUSED(isNew);

            MON_INFO(FetcherShard, "Spawning an actor for shard " << shardId);
        }

    private:
        struct TShardInfo {
            void Kill(const TActorContext& ctx) {
                ctx.Send(FetcherShardId, new TEvents::TEvPoisonPill);
                ctx.Send(StatCollectorId, new TEvents::TEvPoisonPill);
            }

            TActorId FetcherShardId;
            TActorId StatCollectorId;
            IShardMetricsPtr ShardMetrics;
            IShardStatCollector* StatCollector;
        };

        THashMap<TString, TShardInfo> ServiceProviderShardsInfo_;
        THashMap<TShardId, TShardInfo> ShardInfo_;
        TActorId SinkId_;
        TActorId AuthGatekeeperId_;
        TActorId ClusterManagerId_;
        ILimiterPtr Limiter_;
        std::shared_ptr<IFetcherUrlFactory> UrlFactory_;
        THolder<IShardMetricFactory> MetricFactory_;
        IYasmItypeWhiteListPtr YasmWhiteList_;
        TString YasmPrefix_;
        TString IamServiceId_;
        THashMap<TString, TShardId::TNumId> StrToNumId_;
    };

} // namespace

IActor* CreateShardManagerActor(const TShardManagerConfig& conf) {
    return new TShardManager{conf};
}

IActor* CreateFetcherShardActor(TShardActorConfig conf) {
    return new TFetcherShardActor{std::move(conf)};
}

} // namespace NSolomon::NFetcher
