#include "shard_stat.h"
#include "status_view.h"

#include <solomon/services/fetcher/lib/download/download.h>

#include <solomon/libs/cpp/glob/glob.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>

#include <util/generic/hash.h>
#include <util/generic/algorithm.h>

using namespace NActors;
using namespace NMonitoring;

namespace NSolomon::NFetcher {
namespace {
    THostPredicate BuildFilterForRequest(const TEvTargetsStatusRequest& ev) {
        THostPredicate predicate;

        if (ev.Status != yandex::solomon::common::UNKNOWN_STATUS_TYPE) {
            predicate.Conditions.push_back([status=ev.Status] (const TUrlAndShardsStatus& s) {
                bool doesUrlHaveSatus = s.Url.UrlStatus == status;
                bool doesSomeShardHaveStatus = false;

                for (const auto& [_, shard]: s.ShardIdToShardStatus) {
                    if (shard.Status == status) {
                        doesSomeShardHaveStatus = true;
                        break;
                    }
                }

                return doesUrlHaveSatus || doesSomeShardHaveStatus;
            });
        }

        if (ev.NotOk) {
            predicate.Conditions.push_back([] (const TUrlAndShardsStatus& s) {
                auto okStatus = yandex::solomon::common::OK;
                bool isUrlNotOk = s.Url.UrlStatus != okStatus;
                bool isSomeShardNotOk = false;

                for (const auto& [_, shard]: s.ShardIdToShardStatus) {
                    if (shard.Status != okStatus) {
                        isSomeShardNotOk = true;
                        break;
                    }
                }

                return isUrlNotOk || isSomeShardNotOk;
            });
        }

        if (ev.Dc != EDc::UNKNOWN) {
            predicate.Conditions.push_back([&] (const TUrlAndShardsStatus& s) {
                return ev.Dc == s.Url.Dc;
            });
        }

        if (ev.HostGlob.empty()) {
            // do nothing
        } else if (IsGlob(ev.HostGlob)) {
            TStringBuf glob = ev.HostGlob;
            predicate.Conditions.push_back([glob] (const TUrlAndShardsStatus& s) {
                return IsGlobMatch(glob, s.Url.Host);
            });
        } else {
            TStringBuf host = ev.HostGlob;
            predicate.Conditions.push_back([host] (const TUrlAndShardsStatus& s) {
                return s.Url.Host.find(host) != TString::npos;
            });
        }

        return predicate;
    }

    class TShardStatCollector:
        public TActorBootstrapped<TShardStatCollector>,
        public IShardStatCollector
    {
    public:
        explicit TShardStatCollector(ui32 numId)
            : NumId_{numId}
        {
        }

        void Bootstrap(const TActorContext&) {
            Become(&TThis::StateWork);
        }

        STFUNC(StateWork) {
            switch (ev->GetTypeRewrite()) {
                HFunc(TEvTargetsStatusRequest, OnTargetStatusRequest);
                HFunc(TEvReportUrlStatus, OnUrlStatus);
                hFunc(TEvReportShardStatus, OnShardStatus);
                hFunc(TEvRemoveStats, OnRemove);
                cFunc(TEvents::TSystem::PoisonPill, PassAway);
            }
        }

        TShardHealth ShardHealth() const override {
            TShardHealth sh;
            sh.NumId = NumId_;
            std::tie(sh.UrlsOk, sh.UrlsFail) = StatusCounts();
            return sh;
        }

        TShardStats ShardStats() const override {
            return TShardStats{
                .NumId = NumId_,
                .UrlCount = Statuses_.size(),
            };
        }

    private:
        std::pair<ui64, ui64> StatusCounts() const {
            std::pair<ui64, ui64> p{0, 0};
            auto& [ok, fail] = p;

            for (auto&& [_, v]: Statuses_) {
                if (v.Url.UrlStatus == yandex::solomon::common::OK) {
                    ++ok;
                } else {
                    ++fail;
                }
            }

            return p;
        }

        void OnRemove(const TEvRemoveStats::TPtr& ev) {
            auto&& url = ev->Get()->Url;
            const auto count = Statuses_.erase(url);

            if (count == 0) {
                MON_WARN(ShardManager, "Failed to remove " << url << " from stats");
            }
        }

        void OnTargetStatusRequest(const TEvTargetsStatusRequest::TPtr& evPtr, const TActorContext&) {
            auto&& ev = *evPtr->Get();

            const auto predicate = BuildFilterForRequest(ev);
            auto view = TStatusViewBuilder{Statuses_}
                .SetPredicate(predicate)
                .SetOffset(ev.Offset)
                .SetLimit(ev.Limit)
                .Build();

            Send(evPtr->Sender, new TEvTargetsStatusResponse{std::move(view.Statuses), view.Total});
        }

        void OnUrlStatus(const TEvReportUrlStatus::TPtr& ev, const TActorContext&) {
            auto& status = ev->Get()->Status;
            auto& urlAndShardsStatus = Statuses_[status.Url];
            urlAndShardsStatus.Epoch = ev->Get()->Epoch;
            urlAndShardsStatus.Url = std::move(ev->Get()->Status);
        }

        void OnShardStatus(const TEvReportShardStatus::TPtr& evPtr) {
            auto& urlAndShardsStatus = Statuses_[evPtr->Get()->Url];
            if (urlAndShardsStatus.Epoch == 0) {
                MON_ERROR(
                        ShardManager,
                        "didn't find a status for url " << evPtr->Get()->Url << ", while a shard status for shard "
                                                        << evPtr->Get()->Status.ShardId << " was already present");
                return;
            }

            // TODO(ivanzhukov): support an epoch for per-shard statuses
            // so we can delete statuses that weren't updated for too long
            urlAndShardsStatus.ShardIdToShardStatus[evPtr->Get()->Status.ShardId] = std::move(evPtr->Get()->Status);
        }

        TMap<TString, TUrlAndShardsStatus> Statuses_;
        ui32 NumId_;
    };
} // namespace

    std::pair<IShardStatCollector*, TActorId> CreateShardStatCollector(const TActorContext& ctx, ui32 numId) {
        auto* collector = new TShardStatCollector{numId};
        const auto aid = ctx.RegisterWithSameMailbox(collector);

        return std::make_pair(collector, aid);
    }
} // namespace NSolomon::NFetcher
