#pragma once

#include "shard_stat.h"

#include <util/generic/vector.h>
#include <util/generic/map.h>

namespace NSolomon::NFetcher {
    struct THostPredicate {
        bool operator()(const TUrlAndShardsStatus& status) const {
            return AllOf(Conditions, [&] (auto&& c) {
                return c(status);
            });
        }

        TVector<std::function<bool(const TUrlAndShardsStatus&)>> Conditions;
    };

    struct TStatusView {
        TVector<TUrlAndShardsStatus> Statuses;
        ui64 Total{0};
    };

    struct TStatusViewBuilder {
        TStatusViewBuilder(const TMap<TString, TUrlAndShardsStatus>& statuses)
            : Statuses_{statuses}
        {
        }

        TStatusViewBuilder& SetPredicate(THostPredicate predicate) {
            Predicate_ = std::move(predicate);
            return *this;
        }

        TStatusViewBuilder& SetOffset(ui64 offset) {
            Offset_ = offset;
            return *this;
        }

        TStatusViewBuilder& SetLimit(ui64 limit) {
            Limit_ = limit;
            return *this;
        }

        TStatusView Build() {
            ui64 total{0};
            const auto startIdx = Offset_;
            auto count = Statuses_.size() - startIdx;

            if (Limit_ != 0) {
                count = Min<ui64>(Limit_, count);
            }

            const auto endIdx = startIdx + count;

            TVector<TUrlAndShardsStatus> statuses;
            statuses.reserve(count);
            for (auto&& [k, v]: Statuses_) {
                if (!Predicate_(v)) {
                    continue;
                }

                if (total >= startIdx && total < endIdx) {
                    statuses.push_back(v);
                }

                // NOTE: don't terminate this loop since we have to know the
                // total number of elements matching the predicate

                ++total;
            }

            return {
                .Statuses = std::move(statuses),
                .Total = total,
            };
        }

    private:
        THostPredicate Predicate_;
        const TMap<TString, TUrlAndShardsStatus>& Statuses_;
        ui64 Offset_{0};
        ui64 Limit_{0};
    };
} // namespace NSolomon::NFetcher
