#include <solomon/services/fetcher/lib/shard_manager/shard_handler.h>

#include <solomon/services/fetcher/lib/url/url.h>
#include <solomon/services/fetcher/testlib/fetcher_shard.h>
#include <solomon/services/fetcher/testlib/tvm.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <utility>

using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace NThreading;
using namespace testing;

struct TTestListener: public IClusterUpdateListener {
    void OnHostAdded(const TString& url, const THostAndLabels& host) override {
        Added.push_back(std::make_pair(url, host));
    }

    void OnHostRemoved(const TString& url, const THostAndLabels& host) override {
        Removed.push_back(std::make_pair(url, host));
    }

    void OnResolveError(TStringBuf, TStringBuf, const TFailScore&) override {}

    TVector<std::pair<TString, THostAndLabels>> Added;
    TVector<std::pair<TString, THostAndLabels>> Removed;
};

struct TMockResolver: IHostGroupResolver {
    TMockResolver(TString name)
        : Name_{std::move(name)}
    {
    }

    TAsyncResolveResult Resolve() noexcept override {
        return MakeFuture<TResolveResult>(TVector<THostAndLabels>{});
    }

    const TString& Name() const override {
        return Name_;
    }

private:
    TString Name_;
};

TResolveResults MakeResolveResults(const THashMap<TString, THashSet<THostAndLabels>>& results) {
    THashMap<IHostGroupResolverPtr, TClusterResolveResult> res;

    for (auto&& [name, hosts]: results) {
        auto&& val = TClusterResolveResult::FromValue(hosts);
        res.emplace(new TMockResolver{name}, std::move(val));
    }

    return res;
}

auto first = THostAndLabels{"hostname.com"};
auto second = THostAndLabels{"hostname2.com"};

class TShardHandlerTest: public ::testing::Test {
public:
    void SetUp() override {
        MockShard_ = ShardFactory_.MakeShard();
        Shard_.Reset(new TFetcherShard{MockShard_});
        Listener_.Reset(new TTestListener);
        Handler_.Reset(new TFetcherShardHandler{*Shard_, *Listener_});
    }

protected:
    void InsertGroup() {
        auto r = MakeResolveResults({{"foo", { first, second }}});
        Handler_->UpdateCluster(r);

        Listener_->Added.clear();
    }

protected:
    TFetcherShardFactory ShardFactory_;
    TIntrusiveConstPtr<TMockShard> MockShard_;
    THolder<TFetcherShard> Shard_;
    THolder<TTestListener> Listener_;
    THolder<TFetcherShardHandler> Handler_;
};

TEST_F(TShardHandlerTest, HostAddedToGroup) {
    auto r = MakeResolveResults({{"foo", { first, second }}});
    Handler_->UpdateCluster(r);

    ASSERT_THAT(Listener_->Added, UnorderedElementsAre(
            std::make_pair(MakeUrl(*Shard_, first), first),
            std::make_pair(MakeUrl(*Shard_, second), second)
    ));
}

TEST_F(TShardHandlerTest, HostRemovedFromGroup) {
    InsertGroup();

    auto r = MakeResolveResults({{"foo", { first }}});
    Handler_->UpdateCluster(r);

    ASSERT_THAT(Listener_->Removed, UnorderedElementsAre(
            std::make_pair(MakeUrl(*Shard_, second), second)
    ));

    ASSERT_THAT(Listener_->Added, IsEmpty());
}

TEST_F(TShardHandlerTest, GroupRemoved) {
    InsertGroup();

    auto r = MakeResolveResults({{ }});
    Handler_->UpdateCluster(r);

    ASSERT_THAT(Listener_->Removed, UnorderedElementsAre(
            std::make_pair(MakeUrl(*Shard_, first), first),
            std::make_pair(MakeUrl(*Shard_, second), second)
    ));

    ASSERT_THAT(Listener_->Added, IsEmpty());
}

TEST_F(TShardHandlerTest, GroupContentsChanged) {
    InsertGroup();
    auto newFirst = THostAndLabels{"hostname4.com"};
    auto newSecond = THostAndLabels{"hostname3.com"};

    auto r = MakeResolveResults({{"foo", { newFirst, newSecond }}});
    Handler_->UpdateCluster(r);

    ASSERT_THAT(Listener_->Removed, UnorderedElementsAre(
            std::make_pair(MakeUrl(*Shard_, first), first),
            std::make_pair(MakeUrl(*Shard_, second), second)
    ));

    ASSERT_THAT(Listener_->Added, UnorderedElementsAre(
            std::make_pair(MakeUrl(*Shard_, newFirst), newFirst),
            std::make_pair(MakeUrl(*Shard_, newSecond), newSecond)
    ));
}

TEST_F(TShardHandlerTest, GroupReadded) {
    InsertGroup();

    Handler_->UpdateCluster(MakeResolveResults({{ }}));

    ASSERT_THAT(Listener_->Removed, UnorderedElementsAre(
            std::make_pair(MakeUrl(*Shard_, first), first),
            std::make_pair(MakeUrl(*Shard_, second), second)
    ));
    ASSERT_THAT(Listener_->Added, IsEmpty());

    Listener_->Removed.clear();

    auto r = MakeResolveResults({{"foo", { first, second }}});
    Handler_->UpdateCluster(r);

    ASSERT_THAT(Listener_->Added, UnorderedElementsAre(
            std::make_pair(MakeUrl(*Shard_, first), first),
            std::make_pair(MakeUrl(*Shard_, second), second)
    ));
    ASSERT_THAT(Listener_->Removed, IsEmpty());
}

TEST_F(TShardHandlerTest, HostsAreNotRemovedOnError) {
    InsertGroup();
    TResolveResults r;
    r.emplace(new TMockResolver{"foo"}, TClusterResolveResult::FromError("something bad has happened!"));
    Handler_->UpdateCluster(r);

    ASSERT_THAT(Listener_->Removed, IsEmpty());
    ASSERT_THAT(Listener_->Added, IsEmpty());
}

TEST_F(TShardHandlerTest, TvmTicketFromShardGetsAddedToClient) {
    MockShard_->MutCluster().ShardSettings.PullOrPush = TPullSettings{.TvmDestId = 100500};

    TMockTicketProvider ticketProvider;
    Handler_->SetTicketProvider(&ticketProvider);

    auto r = MakeResolveResults({{"foo", { first, second }}});
    Handler_->UpdateCluster(r);
    ASSERT_THAT(ticketProvider.Added, ElementsAre(100500));
}

TEST_F(TShardHandlerTest, ResolversUpdated) {
    auto r0 = MakeResolveResults({{"foo", { first, second }}, {"bar", { first, second }}});
    auto r1 = MakeResolveResults({{"foo", { second }}, {"bar", { first }}});

    Handler_->UpdateCluster(r0);
    Handler_->UpdateCluster(r1);
    ASSERT_EQ(Handler_->GroupToHosts().size(), 2ul);
    ASSERT_EQ(Handler_->FailScores().size(), 2ul);

    for (auto&& [resolver, result]: r1) {
        THashMap<TString, THostAndLabels> resultMap;
        for (auto&& hostAndLabels: result.Value()){
            resultMap.emplace(MakeUrl(*Shard_, hostAndLabels), hostAndLabels);
        }
        ASSERT_EQ(Handler_->GroupToHosts().at(resolver), resultMap);
        ASSERT_EQ(Handler_->FailScores().at(resolver).Total(), 2ul);
        ASSERT_DOUBLE_EQ(Handler_->FailScores().at(resolver).Rate(), 0);
    }
}
