#include <solomon/services/fetcher/lib/shard_manager/shard_resolver.h>

#include <solomon/services/fetcher/testlib/actor_system.h>
#include <solomon/services/fetcher/testlib/db.h>
#include <solomon/services/fetcher/testlib/http_server.h>
#include <solomon/services/fetcher/config/resolver_config.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/http/misc/httpcodes.h>

#include <util/system/event.h>
#include <util/string/split.h>

using namespace testing;
using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace NSolomon::NDb;

namespace NSolomon::NFetcher {
    // allow gtest to print pretty values instead of bytes
    std::ostream& operator<<(std::ostream& os, const THostAndLabels& h) {
        auto s = h.ToString();
        os.write(s.begin(), s.size());

        return os;
    }
}

namespace {

const TString GROUP_RESPONSE{R"(solomon-pre-front-sas-00.search.yandex.net
solomon-pre-front-sas-01.search.yandex.net
)"};

const TString HOST_LIST_RESPONSE = R"(my1.yandex.net label1=foo label2=bar
my2.yandex.net
my3.yandex.net
)";

TVector<THostAndLabels> ParseManyHosts(TStringBuf lines, const TLabels& labels = {}) {
    TVector<THostAndLabels> result;
    for (auto&& line: StringSplitter(lines).Split('\n').SkipEmpty()) {
        auto h = THostAndLabels::FromString(line.Token());

        for (auto&& l: labels) {
            h.Labels.Add(l);
        }

        result.push_back(h);
    }

    return result;
}

template <typename... TArgs>
TVector<THostAndLabels> MergeExpectedLists(TArgs&&... xs) {
    TVector<THostAndLabels> result;

    for (auto&& x: {xs...}) {
        Copy(x.begin(), x.end(), std::back_inserter(result));
    }

    return result;
}

const auto HOST_LIST_EXPECTED = ParseManyHosts(HOST_LIST_RESPONSE);
const auto CONDUCTOR_EXPECTED = ParseManyHosts(GROUP_RESPONSE);

const auto ALL_EXPECTED = MergeExpectedLists(HOST_LIST_EXPECTED, CONDUCTOR_EXPECTED);

const auto HOST_LIST_EXPECTED_WITH_LABELS = HostAndLabelsListFromString(HOST_LIST_RESPONSE);
const auto ALL_EXPECTED_WITH_LABELS = MergeExpectedLists(
    ParseManyHosts(GROUP_RESPONSE, TLabels{{"hello", "world"}}),
    HOST_LIST_EXPECTED_WITH_LABELS
);

class TConductorServer: public TTestServer {
public:
    TConductorServer() {
        AddHandler("/api/groups2hosts/solomon_pre", [] {
            THttpResponse resp;
            resp.SetContent(GROUP_RESPONSE);
            return resp;
        });
    }
};

class TTextFileServer: public TTestServer {
public:
    TTextFileServer() {
        AddHandler("/", [] {
            THttpResponse resp;
            resp.SetContent(HOST_LIST_RESPONSE);
            return resp;
        });
    }
};

struct TServerMock {
    TServerMock() {
        TextFile.Reset(new TTextFileServer);
        Conductor.Reset(new TConductorServer);
        ConductorConfig.SetConductorUrl(Conductor->Address());
    }

    NMonitoring::TMetricRegistry MetricRegistry;
    IHttpClientPtr HttpClient{CreateCurlClient({}, MetricRegistry)};

    TConductorClientConfig ConductorConfig;
    THolder<TConductorServer> Conductor;
    THolder<TTextFileServer> TextFile;
};

void AssertHostsAre(const TResolveResults& results, const TVector<THostAndLabels>& expected) {
    THashSet<THostAndLabels> actual;
    for (auto&& [resolver, result]: results) {
        if (!result.Success()) {
            continue;
        }

        auto&& v = result.Value();
        actual.insert(v.begin(), v.end());
    }

    ASSERT_THAT(actual, UnorderedElementsAreArray(expected));
}

struct TMockResolverFactory: IHostResolverFactory {
    TMockResolverFactory(TServerMock& mock)
        : Mock_{mock}
    {
    }

    IHostGroupResolverPtr CreateConductorGroupResolver(TConductorConfig config) override {
        auto conductorConf = TConductorResolverConfig{"solomon_pre", Mock_.HttpClient.Get()}
            .SetClientConfig(Mock_.ConductorConfig)
            .SetClusterConfig(config);
        return ::CreateConductorGroupResolver(conductorConf, Registry_);
    }

    IHostGroupResolverPtr CreateConductorTagResolver(TConductorConfig config) override {
        auto conductorConf = TConductorResolverConfig{"solomon_pre", Mock_.HttpClient.Get()}
            .SetClientConfig(Mock_.ConductorConfig)
            .SetClusterConfig(config);

        return ::CreateConductorTagResolver(conductorConf, Registry_);
    }

    IHostGroupResolverPtr CreateHostUrlResolver(THostListUrlConfig config) override {
        auto hostUrlConf = THostUrlResolverConfig(THostListUrlConfig(Mock_.TextFile->Address()), Mock_.HttpClient.Get())
            .SetClusterConfig(config);

        return ::CreateHostUrlResolver(hostUrlConf, Registry_);
    }

    IHostGroupResolverPtr CreateQloudResolver(TQloudConfig) override { Y_FAIL(); }
    IHostGroupResolverPtr CreateYpResolver(TYpConfig) override { Y_FAIL(); }
    IHostGroupResolverPtr CreateNetworkResolver(TNetworkConfig) override { Y_FAIL(); }
    IHostGroupResolverPtr CreateNannyResolver(TNannyConfig) override { Y_FAIL(); }
    IHostGroupResolverPtr CreateInstanceGroupResolver(TInstanceGroupConfig) override { Y_FAIL(); }
    IHostGroupResolverPtr CreateHostPatternResolver(THostPatternConfig) override { Y_FAIL(); }
    IHostGroupResolverPtr CreateAgentGroupResolver(TString, TVector<THostAndLabels>) override { Y_FAIL(); }
    IHostGroupResolverPtr CreateYasmAgentGroupResolver(TVector<TString>, TVector<TString>) override { Y_FAIL(); }
    IHostGroupResolverPtr CreateCloudDnsResolver(TString, TCloudDnsConfig) override { Y_FAIL(); }

private:
    TServerMock& Mock_;
    NMonitoring::TMetricRegistry Registry_;
};

} // namespace

class TClusterManagerTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorSystem_ = MakeActorSystem(true);
        ActorSystem_->Initialize();

        Registry_.Reset(new TMetricRegistry);
        Mocks_.Reset(new TServerMock);
        TestActor_ = ActorSystem_->AllocateEdgeActor();
        auto factory = MakeIntrusive<TMockResolverFactory>(*Mocks_);
        ClusterManager_ = ActorSystem_->Register(CreateClusterManager(*Registry_, factory, {}));

        Tasks_ = Registry_->IntGauge(TLabels{{"sensor", "clusterManager.taskCount"}});
    }

protected:
    NDb::NModel::TClusterConfig MakeCluster() {
        auto cluster = MakeClusterConfig();
        cluster.HostUrls = TStringBuilder() << R"([{"url":")" << Mocks_->TextFile->Address() << R"(","labels":[], "ignorePorts":false}])";
        cluster.ConductorGroups = R"([{"group":"solomon_pre","labels":["hello=world"]}])";
        cluster.ConductorTags.clear();
        cluster.NannyGroups.clear();
        cluster.YpClusters.clear();
        cluster.QloudGroups.clear();
        cluster.Hosts.clear();
        return cluster;
    }

protected:
    TActorId TestActor_;
    TActorId ClusterManager_;
    THolder<TFetcherActorRuntime> ActorSystem_;
    THolder<TMetricRegistry> Registry_;
    THolder<TServerMock> Mocks_;

    IIntGauge* Tasks_;
};

TEST_F(TClusterManagerTest, ResolveOnce) {
    auto cluster = MakeCluster();

    {
        auto ev = MakeHolder<TEvResolveCluster>(CreateCluster(cluster));
        ev->SubscriptionType = TEvResolveCluster::ESubscriptionType::Once;

        ActorSystem_->Send(new IEventHandle{ClusterManager_, TestActor_, ev.Release()});
        ASSERT_THAT(Tasks_->Get(), Eq(0));
    }

    {
        auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(TestActor_);
        ASSERT_THAT(ev->Get()->Result->size(), Eq(2u));
        ASSERT_THAT(Tasks_->Get(), Eq(0));
        AssertHostsAre(*ev->Get()->Result, ALL_EXPECTED_WITH_LABELS);
    }
}

TEST_F(TClusterManagerTest, MultipleSubscribesOnStart) {
    auto cluster = MakeCluster();
    const int count = 4;
    TVector<TActorId> requesters;

    for (auto i = 0; i < count; ++i) {
        requesters.push_back(ActorSystem_->AllocateEdgeActor());
        auto ev = MakeHolder<TEvResolveCluster>(CreateCluster(cluster));
        ActorSystem_->Send(new IEventHandle{ClusterManager_, requesters[i], ev.Release()});
    }

    for (auto i = 0; i < count; ++i) {
        auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(requesters[i]);
        ASSERT_THAT(ev->Get()->Result->size(), Eq(2u));
        AssertHostsAre(*ev->Get()->Result, ALL_EXPECTED_WITH_LABELS);
    }
}

TEST_F(TClusterManagerTest, ResolveSubscribe) {
    auto cluster = MakeCluster();

    {
        auto ev = MakeHolder<TEvResolveCluster>(CreateCluster(cluster));
        ActorSystem_->Send(new IEventHandle{ClusterManager_, TestActor_, ev.Release()});
    }

    {
        auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(TestActor_);
        ASSERT_THAT(ev->Get()->Result->size(), Eq(2u));
        ASSERT_THAT(Tasks_->Get(), Eq(1));
        AssertHostsAre(*ev->Get()->Result, ALL_EXPECTED_WITH_LABELS);
    }

    {
        auto ev = MakeHolder<TEvResolveCluster>(CreateCluster(cluster));
        ActorSystem_->Send(new IEventHandle{ClusterManager_, TestActor_, ev.Release()});
    }

    {
        // check that a new task is not spawned for the same cluster
        auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(TestActor_);
        ASSERT_THAT(ev->Get()->Result->size(), Eq(2u));
        ASSERT_THAT(Tasks_->Get(), Eq(1));
        AssertHostsAre(*ev->Get()->Result, ALL_EXPECTED_WITH_LABELS);
    }
}

TEST_F(TClusterManagerTest, ResolveOnceEmpty) {
    auto cluster = MakeClusterConfig();
    cluster.HostUrls.clear();
    cluster.ConductorGroups.clear();
    cluster.ConductorTags.clear();
    cluster.NannyGroups.clear();
    cluster.YpClusters.clear();
    cluster.QloudGroups.clear();

    {
        auto ev = MakeHolder<TEvResolveCluster>(CreateCluster(cluster));
        ev->SubscriptionType = TEvResolveCluster::ESubscriptionType::Once;

        ActorSystem_->Send(new IEventHandle{ClusterManager_, TestActor_, ev.Release()});
    }

    {
        auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(TestActor_);
        ASSERT_TRUE(ev->Get()->Result->empty());
        ASSERT_THAT(Tasks_->Get(), Eq(0));
    }

    // no project/cluster ID
    {
        cluster.ProjectId.clear();
        cluster.Id.clear();
        auto ev = MakeHolder<TEvResolveCluster>(CreateCluster(cluster));
        ev->SubscriptionType = TEvResolveCluster::ESubscriptionType::Once;

        ActorSystem_->Send(new IEventHandle{ClusterManager_, TestActor_, ev.Release()});
    }

    {
        auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(TestActor_);
        ASSERT_TRUE(ev->Get()->Result->empty());
        ASSERT_THAT(Tasks_->Get(), Eq(0));
    }
}

TEST_F(TClusterManagerTest, HostsAreUpdatedAfterClusterModification) {
    auto cluster = MakeCluster();

    {
        auto ev = MakeHolder<TEvResolveCluster>(CreateCluster(cluster));
        ActorSystem_->Send(new IEventHandle{ClusterManager_, TestActor_, ev.Release()});
    }

    {
        auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(TestActor_);
        ASSERT_THAT(ev->Get()->Result->size(), Eq(2u));
        ASSERT_THAT(Tasks_->Get(), Eq(1));
        AssertHostsAre(*ev->Get()->Result, ALL_EXPECTED_WITH_LABELS);
    }

    {
        cluster.ConductorGroups.clear();
        ActorSystem_->Send(ClusterManager_, MakeHolder<TEvClustersChanged>(TVector{CreateCluster(cluster)}));

        auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(TestActor_);
        ASSERT_THAT(ev->Get()->Result->size(), Eq(1u));
        ASSERT_THAT(Tasks_->Get(), Eq(1));
        AssertHostsAre(*ev->Get()->Result, HOST_LIST_EXPECTED_WITH_LABELS);
    }
}

class TShardResolverTest: public ::testing::Test {
public:
    static constexpr auto CONDUCTOR_RESOLVER_IDX = 0;
    static constexpr auto HOST_URL_IDX = 1;

    void SetUp() override {
        ActorSystem_ = MakeActorSystem(true);
        ActorSystem_->Initialize();
        Mocks_.Reset(new TServerMock);

        TextFileServer_ = Mocks_->TextFile.Get();
        ConductorServer_ = Mocks_->Conductor.Get();

        auto conductorConf = TConductorResolverConfig{"solomon_pre", Mocks_->HttpClient.Get()}
            .SetClientConfig(Mocks_->ConductorConfig);
        auto hostUrlConf = THostUrlResolverConfig(THostListUrlConfig(TextFileServer_->Address()), Mocks_->HttpClient.Get());

        Resolvers_.push_back(CreateConductorGroupResolver(conductorConf, MetricRegistry_));
        Resolvers_.push_back(CreateHostUrlResolver(hostUrlConf, MetricRegistry_));

        TestActor_ = ActorSystem_->AllocateEdgeActor();

        FillDatabase();
    }

    void TearDown() override {
        Resolvers_.clear();
        TextFileServer_->Stop();
        ConductorServer_->Stop();
        ActorSystem_.Reset();
    }

protected:
    void FillDatabase() {
        auto service = MakeServiceConfig();
        service.Id = "service_id_0";
        service.Name = "service_name_0";

        auto cluster = MakeClusterConfig();
        cluster.Id = "service_id_0";
        cluster.Name = "service_name_0";
        cluster.HostUrls = TStringBuilder() << R"([{"url":")" << TextFileServer_->Address() << R"(","labels":[], "ignorePorts":false}])";
        cluster.ConductorGroups = R"([{"group":"solomon_pre","labels":["hello=world"]}])";
        cluster.ConductorTags.clear();
        cluster.NannyGroups.clear();
        cluster.YpClusters.clear();
        cluster.QloudGroups.clear();

        // TODO: add pattern
        cluster.Hosts.clear();

        auto shard = MakeShardConfig();
        shard.ClusterId = cluster.Id;
        shard.ClusterName = cluster.Name;

        FetcherShard_ = MakeHolder<TFetcherShard>(CreateSimpleShard(
            MakeAtomicShared<NModel::TShardConfig>(shard),
            MakeAtomicShared<NModel::TClusterConfig>(cluster),
            MakeAtomicShared<NModel::TServiceConfig>(service),
            {}
        ));
    }

protected:
    THashMap<IHostGroupResolverPtr, TClusterResolveResult> Result_;
    TVector<IHostGroupResolverPtr> Resolvers_;

    THolder<TServerMock> Mocks_;
    TConductorServer* ConductorServer_{nullptr};
    TTextFileServer* TextFileServer_{nullptr};
    NMonitoring::TMetricRegistry MetricRegistry_;

    THolder<TFetcherShard> FetcherShard_;
    TFetcherConfig FetcherConfig_;

    TActorId TestActor_;
    THolder<TFetcherActorRuntime> ActorSystem_;
};

TEST_F(TShardResolverTest, AllClustersAreResolved) {
    ActorSystem_->Register(ResolveCluster(
            Resolvers_,
            TestActor_
    ));

    TVector<THostAndLabels> hosts;
    auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(TestActor_);
    for (auto&& [_, result]: *ev->Get()->Result) {
        ASSERT_TRUE(result.Success());
        Copy(result.Value().begin(), result.Value().end(), std::back_inserter(hosts));
    }

    ASSERT_THAT(hosts, UnorderedElementsAreArray(ALL_EXPECTED));
}

TEST_F(TShardResolverTest, ClustersArePartiallyResolved) {
    TextFileServer_->AddHandler("/", [] {
        THttpResponse r;
        Sleep(TDuration::MilliSeconds(200));
        r.SetContent(GROUP_RESPONSE);
        return r;
    });

    ActorSystem_->Register(ResolveCluster(
            Resolvers_,
            TestActor_,
            TDuration::MilliSeconds(40)
    ));

    auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(TestActor_);
    auto&& result = *ev->Get()->Result;
    ASSERT_THAT(result.size(), Eq(2u));

    auto& conductorResult = result.at(Resolvers_[CONDUCTOR_RESOLVER_IDX]);
    auto& hostUrlResult = result.at(Resolvers_[HOST_URL_IDX]);
    ASSERT_TRUE(conductorResult.Success());
    ASSERT_FALSE(hostUrlResult.Success());
};

TEST_F(TShardResolverTest, FailedResolve) {
    auto handler = [] {
        THttpResponse r;
        r.SetHttpCode(HTTP_INTERNAL_SERVER_ERROR);
        r.SetContent("internal error");
        return r;
    };

    TextFileServer_->AddHandler("/", handler);
    ConductorServer_->AddHandler("/api/groups2hosts/solomon_pre", handler);

    ActorSystem_->Register(ResolveCluster(
            Resolvers_,
            TestActor_,
            TDuration::MilliSeconds(40)
    ));

    auto ev = ActorSystem_->GrabEdgeEvent<TEvClusterResolved>(TestActor_);
    auto&& result = *ev->Get()->Result;
    ASSERT_THAT(result.size(), Eq(2u));

    auto& conductorResult = result.at(Resolvers_[CONDUCTOR_RESOLVER_IDX]);
    auto& hostUrlResult = result.at(Resolvers_[HOST_URL_IDX]);
    ASSERT_FALSE(conductorResult.Success());
    ASSERT_FALSE(hostUrlResult.Success());
    Sleep(TDuration::Seconds(1));
}
