#include <solomon/services/fetcher/lib/shard_manager/status_view.h>

#include <solomon/libs/cpp/glob/glob.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace testing;
using namespace NSolomon::NFetcher;
using namespace NSolomon;

TMap<TString, TUrlAndShardsStatus> MakeStatusMap(ui64 count) {
    TMap<TString, TUrlAndShardsStatus> result;
    for (auto i = 0u; i < count; ++i) {
        result.emplace(ToString(i), TUrlAndShardsStatus{ .Url = {.Host = ToString(i) } });
    }

    return result;
}

TEST(TStatusViewTest, LimitOffset) {
    auto statusMap = MakeStatusMap(100);

    {
        auto view = TStatusViewBuilder{statusMap}
            .SetLimit(10)
            .Build();

        ASSERT_THAT(view.Statuses, SizeIs(10));
        ASSERT_THAT(view.Total, Eq(100u));
    }

    {
        auto view = TStatusViewBuilder{statusMap}
            .SetLimit(10)
            .SetOffset(101)
            .Build();

        ASSERT_THAT(view.Statuses, SizeIs(0));
        ASSERT_THAT(view.Total, Eq(100u));
    }

    {
        auto view = TStatusViewBuilder{statusMap}
            .SetLimit(10)
            .SetOffset(98)
            .Build();

        ASSERT_THAT(view.Statuses, SizeIs(2));
        ASSERT_THAT(view.Total, Eq(100u));
    }

    {
        for (auto i: {0, 2, 90, 37}) {
            auto view = TStatusViewBuilder{statusMap}
                .SetLimit(10)
                .SetOffset(i)
                .Build();

            ASSERT_THAT(view.Statuses, SizeIs(10));
            ASSERT_THAT(view.Total, Eq(100u));
        }
    }

    {
        for (auto i: {0, 2, 90, 37}) {
            auto view = TStatusViewBuilder{statusMap}
                .SetLimit(10)
                .SetOffset(i)
                .Build();

            ASSERT_THAT(view.Statuses, SizeIs(10));
            ASSERT_THAT(view.Total, Eq(100u));
        }
    }
}

TEST(TStatusViewTest, Predicate) {
    auto statusMap = MakeStatusMap(100);

    {
        THostPredicate pred;
        pred.Conditions.push_back([] (auto&& status) {
            return IsGlobMatch("2*", status.Url.Host);
        });

        // expected matches are 2, 20, 21, ...
        auto view = TStatusViewBuilder{statusMap}
            .SetLimit(1)
            .SetOffset(2)
            .SetPredicate(pred)
            .Build();

        ASSERT_THAT(view.Statuses, SizeIs(1));
        ASSERT_THAT(view.Total, Eq(11u));
        ASSERT_THAT(view.Statuses[0].Url.Host, StrEq("21"));
    }

    {
        THostPredicate pred;
        pred.Conditions.push_back([] (auto&&) {
            return false;
        });

        // expected matches are 2, 20, 21, ...
        auto view = TStatusViewBuilder{statusMap}
            .SetLimit(1)
            .SetOffset(2)
            .SetPredicate(pred)
            .Build();

        ASSERT_THAT(view.Statuses, SizeIs(0));
        ASSERT_THAT(view.Total, Eq(0u));
    }
}
