#include "sink.h"

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <library/cpp/monlib/encode/text/text.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>

#include <util/stream/mem.h>

namespace NSolomon::NFetcher {
namespace {
    using namespace NActors;
    using namespace NMonitoring;

    class TDebugDataSink: public TActorBootstrapped<TDebugDataSink> {
    public:
        void Bootstrap(const TActorContext&) {
            Become(&TThis::StateFunc);
        }

        STFUNC(StateFunc) {
            Y_UNUSED(ctx);

            switch (ev->GetTypeRewrite()) {
                HFunc(TEvSinkWrite, OnDataReceived);
            }
        }

        void OnDataReceived(const TEvSinkWrite::TPtr& ev, const TActorContext&) {
            auto& shardData = ev->Get()->ShardData;
            auto format = shardData.Format;
            auto data = std::move(shardData.Data);

            auto textEncoder = EncoderText(&Cout);

            Cout << "-------\n";
            Cout << "Received data for shard{";

            if (shardData.ShardId.IsValid()) {
                Cout << "id=" << shardData.ShardId.StrId() << "; numId=" << shardData.ShardId.NumId();
            } else {
                Cout << "project=" << shardData.ProjectId
                     << "; cluster=" << shardData.ClusterName
                     << "; service=" << shardData.ServiceName;
            }

            Cout << "} at " << TInstant::Now() << Endl;

            try {
                switch (format) {
                case NMonitoring::EFormat::JSON:
                        DecodeJson(data, textEncoder.Get());
                    break;
                case NMonitoring::EFormat::SPACK: {
                        TMemoryInput in{data};
                        DecodeSpackV1(&in, textEncoder.Get());
                    break;
                }
                default:
                    Cerr << "Unsupported format: " << ui8(format) << Endl;
                }
            } catch (...) {
                Cerr << "Error while decoding: " << CurrentExceptionMessage();
            }
        }
    };

    class TNullDataSink: public TActor<TNullDataSink> {
    public:
        TNullDataSink()
            : TActor<TNullDataSink>(&TNullDataSink::StateFunc)
        {
        }

        STFUNC(StateFunc) {
            switch (ev->GetTypeRewrite()) {
                HFunc(TEvSinkWrite, OnDataReceived);
            }
        }

        void OnDataReceived(const TEvSinkWrite::TPtr&, const TActorContext&) {
        }
    };

} // namespace

NActors::IActor* CreateDebugDataSink() {
    return new TDebugDataSink;
}

NActors::IActor* CreateNullDataSink() {
    return new TNullDataSink;
}

} // namespace NSolomon::NFetcher
