#include "factory.h"

#include <solomon/services/fetcher/lib/fetcher_shard.h>

#include <library/cpp/int128/int128.h>
#include <library/cpp/digest/crc32c/crc32c.h>

namespace NSolomon::NFetcher {
namespace {
    ui32 ExtractHostAddr(TIpv6Address addr) {
        auto lo = GetLow(ui128(addr));
        return lo & ~ui32{0};
    }

    class TSourceIdFactory: public ISourceIdFactory {
    public:
        TSourceIdFactory(const TFetcherShard* s = nullptr)
            : ShardConf_{s}
        {
        }

        // 64-bit source id is represented as follows:
        // | 16 bit url hash | 16 bit port | 32 bit lower bits of IPv6
        // Motivation:
        // * Host is indentified by lower 32 bits of its address https://nda.ya.ru/t/RDEa6E4w3VxU6B
        // * It is possible that shard contains multiple endpoints for a single ID and different ports, so we add port
        // * Also it is possible that multiple endpoints are served via a single IP:port tuple (e.g. using nginx), so we add a hash from the url
        TSourceId Create(TIpv6Address addr, ui16 port, TStringBuf url) const override {
            ui32 urlHash = Crc32c(url.data(), url.size());
            ui32 hostAddr = ExtractHostAddr(addr);

            TSourceId sourceId{};
            sourceId |= hostAddr;
            sourceId |= ui64(port) << 32;
            sourceId |= ui64(urlHash) << 48;

            return sourceId;
        }

    private:
        [[ maybe_unused ]]
        const TFetcherShard* ShardConf_;
    };
} // namespace
    ISourceIdFactoryPtr CreateSourceIdFactory(const TFetcherShard& shard) {
        return MakeHolder<TSourceIdFactory>(&shard);
    }

    ISourceIdFactoryPtr DefaultSourceIdFactory() {
        return MakeHolder<TSourceIdFactory>();
    }
} // namespace NSolomon::NFetcher
