#include "fetcher_url.h"
#include "headers.h"
#include "interval.h"
#include "shard_state.h"
#include "ts_args.h"

#include <solomon/services/fetcher/lib/dc_label/dc_label.h>

#include <solomon/libs/cpp/backoff/jitter.h>

#include <util/digest/numeric.h>
#include <util/string/cast.h>

using namespace NMonitoring;
using namespace std::chrono_literals;
using namespace yandex::solomon::common;

namespace NSolomon::NFetcher {
namespace {
    constexpr TStringBuf HOST_LABEL_NAME = "host";

    ui16 EffectivePort(const TFetcherShard& shard, const THostAndLabels& hostAndLabels) {
        auto port = hostAndLabels.Port
            ? hostAndLabels.Port
            : shard.Port();

        port += hostAndLabels.PortShift;
        return port;
    }

} // namespace

    TString MakeUrl(const TFetcherShard& shard, const THostAndLabels& hostAndLabels, TMaybeIp maybeIp = Nothing()) {
        TStringBuilder builder;

        auto protocol = shard.Protocol();
        if (protocol == NDb::NModel::EPullProtocol::HTTPS) {
            builder << "https://";
        } else {
            builder << "http://";
        }

        // with http we can use cached IP address
        TIpv6Address ip;
        if (hostAndLabels.IpAddress) {
            ip = *hostAndLabels.IpAddress;
        } else if (maybeIp) {
            ip = *maybeIp;
        }

        if (ip.IsValid()) {
            if (ip.Type() == TIpv6Address::Ipv6) {
                builder << '[' << ip.ToString(false) << ']';
            } else {
                builder << ip.ToString(false);
            }
        } else {
            builder << hostAndLabels.Host;
        }

        auto port = EffectivePort(shard, hostAndLabels);
        builder << ':' << port << shard.UrlPath();
        return builder;
    }

    TErrorOr<THeaders, TGenericError> TFetcherUrlBase::MakeHeaders() const {
        THeaders result;
        auto&& id = ClusterInfo_.Id();
        if (!id.empty()) {
            result[NAME_FETCHER_ID] = id;
            result[NAME_CLUSTER_ID] = id;
        } else {
            result[NAME_FETCHER_ID] = VALUE_FETCHER_ID;
        }

        result[NAME_ACCEPT] = VALUE_ACCEPT;
        result[NAME_ACCEPT_ENCODING] = VALUE_ACCEPT_ENCODING;

        auto ok = MakeHeadersImpl(result);
        if (!ok.Success()) {
            return ok.PassError<THeaders>();
        }

        if (!HostAndLabels_.Host.empty()) {
            result[NAME_HOST] = HostAndLabels_.Host;
        }

        if (auto ret = AddSecurityHeaders(result); !ret.Success()) {
            return ret.PassError<THeaders>();
        }

        return result;
    }

    TErrorOr<std::optional<TString>, TGenericError> TFetcherUrlBase::GetTvmTicket() const {
        auto tvmId = Shard_.TvmId();
        if (auto hostId = HostAndLabels_.TvmDestId) {
            tvmId = hostId;
        }

        std::optional<TString> ticket;

        if (!tvmId) {
            return ticket;
        } else if (TicketProvider_ == nullptr) {
            return TErrorOr<std::optional<TString>, TGenericError>::FromError("TVM is not configured");
        }

        auto ticketResult = TicketProvider_->GetTicket(*tvmId);
        if (!ticketResult.Success()) {
            return ticketResult.PassError<std::optional<TString>>();
        }

        return TErrorOr<std::optional<TString>, TGenericError>::FromValue(
            ticketResult.Extract()
        );
    }

    TErrorOr<std::optional<TString>, TGenericError> TFetcherUrlBase::GetIamToken() const {
        std::optional<TString> token;
        if (!IamTokenProvider_) {
            return token;
        }

        auto iamToken = IamTokenProvider_->Token();
        token = iamToken->Token();
        return token;
    }

    const NMonitoring::TLabels& TFetcherUrlBase::HostLabels() const {
        return HostAndLabels_.Labels;
    }

    TString GetHostLabel(const THostAndLabels& hostAndLabels, const bool useFqdn) {
        if (hostAndLabels.Host) {
            if (useFqdn) {
                return hostAndLabels.Host;
            }
            // get short domain name
            TStringBuf sb{hostAndLabels.Host};
            return TString{sb.Before('.')};
        } else if (hostAndLabels.IpAddress) {
            auto hostLabel = hostAndLabels.Labels.Find(HOST_LABEL_NAME);
            return hostLabel
                ? TString{TStringBuf{hostLabel->Value()}.Before('.')}
                : hostAndLabels.IpAddress->ToString(false);
        }

        Y_FAIL();
    }

    TString TFetcherUrlBase::Fqdn() const {
        return HostAndLabels_.Host
            ? HostAndLabels_.Host
            : HostAndLabels_.IpAddress->ToString(false);
    }

    TFetcherUrlBase::TFetcherUrlBase(
            TFetcherShard shard,
            THostAndLabels hostAndLabels,
            NAuth::NTvm::ITicketProvider* ticketProvider,
            NCloud::ITokenProviderPtr iamTokenProvider,
            const TClusterInfo& clusterInfo,
            const ISourceIdFactory& factory)
        : HostAndLabels_{std::move(hostAndLabels)}
        , Shard_{std::move(shard)}
        , TicketProvider_{ticketProvider}
        , IamTokenProvider_{std::move(iamTokenProvider)}
        , ClusterInfo_{clusterInfo}
        , SourceIdFactory_{factory}
    {
        IpAddress_ = HostAndLabels_.IpAddress
            ? *HostAndLabels_.IpAddress
            : TIpv6Address{};

        Status_.Url = MakeUrl(Shard_, HostAndLabels_);
        Status_.Host = Fqdn();

        if (auto dcLabel = HostAndLabels_.Labels.Find(DC_LABEL_NAME)) {
            if (dcLabel->Value() == DC_AUTO) {
                // do not keep label 'DC=auto' to avoid further checks
                HostAndLabels_.Labels.Extract(DC_LABEL_NAME);
            } else {
                Status_.Dc = FromDcLabel(dcLabel->Value());
                ForcedDc_ = true; // do not try to change DC further
            }
        }

        HostLabel_ = HostLabel();
        FetchState_.Jitter = 50ms + TFullJitter{}(Shard_.FetchInterval() - 500ms);
    }

    void TFetcherUrlBase::SetFetcherShard(const TFetcherShard& shard) {
        if (Shard_.FetchInterval() != shard.FetchInterval()) {
            FetchState_.Jitter = 50ms + TFullJitter{}(Shard_.FetchInterval() - 500ms);
        }

        Shard_ = shard;
        Status_.Url = MakeUrl(Shard_, HostAndLabels_);
        HostLabel_ = HostLabel();
        UpdateSourceId();
    }

    TString TFetcherUrlBase::HostLabel() const {
        return GetHostLabel(HostAndLabels_, Shard_.IsUseFqdn());
    }

    TString TFetcherUrlBase::EffectiveUrl() const {
        return MakeUrl(Shard_, HostAndLabels_, IpAddress_);
    }

    TString TFetcherUrlBase::DisplayUrl() const {
        return Status_.Url;
    }

    void TFetcherUrlBase::SetIpAddress(TIpv6Address addr, EDc dc) {
        IpAddress_ = addr;

        if (!ForcedDc_) {
            // update DC label only if it's not forced by the user's config
            HostAndLabels_.Labels.Extract(DC_LABEL_NAME);
            HostAndLabels_.Labels.Add(DC_LABEL_NAME, ToDcLabel(dc));
            Status_.Dc = dc;
        }

        UpdateSourceId();
    }

    void TFetcherUrlBase::UpdateSourceId() {
        SourceId_ = SourceIdFactory_.Create(IpAddress_, EffectivePort(Shard_, HostAndLabels_), DisplayUrl());
    }

    TMaybeIp TFetcherUrlBase::IpAddress() const {
        return IpAddress_.IsValid()
            ? TMaybeIp{IpAddress_}
            : Nothing();
    }

    const TUrlStatus& TFetcherUrlBase::Status() const {
        return Status_;
    }

    TUrlState TFetcherUrlBase::State() const {
        TUrlState urlState;
        urlState.DisplayUrl = DisplayUrl();
        urlState.EffectiveUrl = EffectiveUrl();

        urlState.Status = Status();
        urlState.IpAddress = IpAddress_;
        urlState.SourceId = SourceId_;
        urlState.LastResponse = LastResponse();

        return urlState;
    }


    TErrorOr<bool, TGenericError> TFetcherUrlBase::AddTvmTokenToHeaders(THeaders& headers) const {
        auto tvmResult = GetTvmTicket();

        if (tvmResult.Fail()) {
            return tvmResult.PassError<bool>();
        }

        auto tvmToken = tvmResult.Extract();
        if (tvmToken) {
            headers[NAME_TVM_TICKET] = *tvmToken;
            return true;
        }
        return false;
    }

    TErrorOr<bool, TGenericError> TFetcherUrlBase::AddIamTokenToHeaders(THeaders& headers) const {
        if (Shard_.Protocol() != NDb::NModel::EPullProtocol::HTTPS) {
            return false;
        }

        auto iamResult = GetIamToken();

        if (iamResult.Fail()) {
            return iamResult.PassError<bool>();
        }

        auto iamToken = iamResult.Extract();
        if (iamToken) {
            headers[NAME_AUTHORIZATION] = VALUE_IAM_TOKEN_KEY + " " + *iamToken;
            return true;
        }
        return false;
    }

    TErrorOr<bool, TGenericError> TFetcherUrlBase::AddSecurityHeaders(THeaders& headers) const {
        auto tvmResult = AddTvmTokenToHeaders(headers);
        if (tvmResult.Fail() || tvmResult.Value()) {
            return tvmResult;
        }

        return AddIamTokenToHeaders(headers);
    }

    TErrorOr<TString, TGenericError> TFetcherUrlBase::MakeCgiParams() const {
        if (Shard_.IsAddTsArgs()) {
            const auto roundedNow = RoundToPrev(TInstant::Now(), Shard_.FetchInterval());

            auto url = EffectiveUrl();
            // :(
            const char delim = url.find('?') != TString::npos ? '&' : '?';
            return TErrorOr<TString, TGenericError>::FromValue(
                TStringBuilder() << delim << MakeTsParams(Shard_.FetchInterval(), FetchState_.StartRounded, roundedNow)
            );
        }

        return TErrorOr<TString, TGenericError>::FromValue(TString{});
    }

    TErrorOr<TString, TGenericError> TFetcherUrlBase::MakeRequestBody() const {
        return TErrorOr<TString, TGenericError>::FromValue(TString{});
    }

    TErrorOr<TFetchRequest, TGenericError> TFetcherUrlBase::PrepareRequest() const {
        TFetchRequest req;
        req.Url = EffectiveUrl();

        auto params = MakeCgiParams();
        if (!params.Success()) {
            return params.PassError<TFetchRequest>();
        }

        req.CgiParams = params.Extract();

        auto headers = MakeHeaders();
        if (!headers.Success()) {
            return headers.PassError<TFetchRequest>();
        }

        req.Headers = headers.Extract();
        auto body = MakeRequestBody();
        if (!body.Success()) {
            return body.PassError<TFetchRequest>();
        }

        req.Body = body.Extract();
        req.Method = HttpMethod();
        req.MaxResponseSizeBytes = Shard_.MaxResponseSizeBytes();
        return req;
    }

    void TFetcherUrlBase::SetFetchDuration(TDuration fetchDuration) {
        Status_.FetchDuration = fetchDuration;
        Status_.FetchTime = FetchState_.StartPrecise;
    }

    bool TFetcherUrlBase::DoesHaveDownloadErrors(const IHttpClient::TResult& fetchResultOrErr) {
        const bool isTimeout = (
                !fetchResultOrErr.Success()
                        && fetchResultOrErr.Error().Type() == TRequestError::EType::ReadTimeout);

        std::optional<TFetchError> err;

        if (isTimeout && (Status_.FetchDuration >= Shard_.FetchInterval())) {
            err = TFetchError(SKIP_TOO_LONG);
        } else if (isTimeout) {
            err = TFetchError(CONNECT_FAILURE);
        } else if (!fetchResultOrErr.Success()) {
            const auto& originalError = fetchResultOrErr.Error();
            TString message = TStringBuilder() << '[' << originalError.Type() << "] " << originalError.Message();

            switch (fetchResultOrErr.Error().Type()) {
                case TRequestError::EType::ConnectFailed:
                case TRequestError::EType::SocketError:
                    err = TFetchError(CONNECT_FAILURE, message);
                    break;
                case TRequestError::EType::DnsFailure:
                    err = TFetchError(UNKNOWN_HOST, message);
                    break;
                case TRequestError::EType::ResponseTooLarge:
                    err = TFetchError(RESPONSE_TOO_LARGE, message);
                    break;
                default: {
                    err = TFetchError(UNKNOWN_ERROR, message);
                    break;
                }
            }
        } else {
            return false;
        }

        Status_.UrlStatus = err->Status;
        Status_.UrlError = err->Message;

        return true;
    }

    void TFetcherUrlBase::SetResponseSize(size_t responseBytes) {
        Status_.ResponseBytes = responseBytes;
    }

    bool TFetcherUrlBase::DoesHaveContentErrors(IResponse& resp) {
        std::optional<TFetchError> err;

        switch (resp.Code()) {
            case HTTP_OK:
                return false;
            case HTTP_NO_CONTENT:
                err = TFetchError(OK);
                break;
            default: {
                TString msg = TStringBuilder() << ToString(static_cast<int>(resp.Code())) << ": " << resp.ExtractData();
                err = TFetchError(NOT_200, std::move(msg));
                break;
            }
        }

        Status_.UrlStatus = err->Status;
        Status_.UrlError = err->Message;

        return true;
    }

    void TFetcherUrlBase::ReportStatus(yandex::solomon::common::UrlStatusType status, TString errorMessage) {
        Status_.UrlStatus = status;
        Status_.UrlError = std::move(errorMessage);
    }

    bool TFetcherUrlBase::IsLocal() const {
        if (ClusterInfo_.OperationMode() == EOperationMode::Global) {
            return true;
        }

        auto dc = ClusterInfo_.Dc();
        return dc != EDc::UNKNOWN && dc == Status().Dc;
    }

    void TFetcherUrlBase::BecomeDefunct(UrlStatusType status, TString errorMessage) {
        Status_.UrlStatus = status;
        Status_.UrlError = std::move(errorMessage);
    }

    void TFetcherUrlBase::BecomeIdle() {
    }

    void TFetcherUrlBase::ClearError() {
        Status_.UrlStatus = UrlStatusType::OK;
        Status_.UrlError = {};
    }

    void TFetcherUrlBase::SetDownloadTime(TInstant preciseStart, TInstant roundedStart) {
        FetchState_.StartRounded = roundedStart;
        FetchState_.StartPrecise = preciseStart;
    }

    void TFetcherUrlBase::SetContentType(NMonitoring::EFormat format, NMonitoring::ECompression compression) {
        if (format == EFormat::JSON) {
            Status_.ContentType = JSON;
        } else if (format == EFormat::SPACK) {
            switch (compression) {
            case ECompression::UNKNOWN:
            case ECompression::IDENTITY:
                Status_.ContentType = SPACK;
                break;
            case ECompression::ZLIB:
                Status_.ContentType = SPACK_ZLIB;
                break;
            case ECompression::LZ4:
                Status_.ContentType = SPACK_LZ4;
                break;
            case ECompression::ZSTD:
                Status_.ContentType = SPACK_ZSTD;
                break;
            }
        } else {
            Status_.ContentType = UNKNOWN_CONTENT_TYPE;
        }
    }

} // namespace NSolomon::NFetcher
