#pragma once

#include "interval.h"
#include "url_state.h"
#include "parser.h"

#include <solomon/services/fetcher/lib/sink/sink.h>
#include <solomon/services/fetcher/lib/source_id/factory.h>
#include <solomon/services/fetcher/lib/download/download.h>
#include <solomon/services/fetcher/lib/fetcher_shard.h>
#include <solomon/services/fetcher/lib/cluster/cluster.h>
#include <solomon/services/fetcher/lib/host_groups/host_and_labels.h>

#include <solomon/services/fetcher/lib/shard_manager/counters.h>
#include <solomon/services/fetcher/lib/yasm/itype_white_list.h>

#include <solomon/libs/cpp/auth/tvm/tvm.h>
#include <solomon/libs/cpp/cloud/iam/iam.h>
#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/shard_key/shard_key.h>

#include <solomon/protos/common/url_info.pb.h>

namespace NSolomon::NFetcher {
    struct TFetchRequest {
        enum EMethod {
            GET,
            POST,
        };

        TString Url;
        TString CgiParams;
        THeaders Headers;
        TString Body;
        EMethod Method{GET};

        ui64 MaxResponseSizeBytes{0};
    };

    struct TFetchError {
        explicit TFetchError(yandex::solomon::common::UrlStatusType status, TString msg = {}) noexcept
            : Status{status}
            , Message{std::move(msg)}
        {
        }

        yandex::solomon::common::UrlStatusType Status;
        TString Message;
    };

    class TFetcherUrlBase {
    public:
        TFetcherUrlBase(
                TFetcherShard shard,
                THostAndLabels hostAndLabels,
                NAuth::NTvm::ITicketProvider*,
                NCloud::ITokenProviderPtr iamTokenProvider,
                const TClusterInfo& clusterInfo,
                const ISourceIdFactory& sourceIdFactory);
        virtual ~TFetcherUrlBase() = default;

        void SetFetcherShard(const TFetcherShard&);

        // this is a string representation built from the hostname (if any) and the path taken from service config
        TString DisplayUrl() const;

        // we avoid excess DNS queries by caching IP addresses, so this one contains the address instead of the hostname
        TString EffectiveUrl() const;

        TString HostLabel() const;

        virtual void SetIpAddress(TIpv6Address addr, EDc dc);
        TMaybeIp IpAddress() const;
        TString Fqdn() const;

        TUrlState State() const;
        const TUrlStatus& Status() const;
        void SetDownloadTime(TInstant preciseStart, TInstant roundedStart);
        void SetContentType(NMonitoring::EFormat format, NMonitoring::ECompression compression);

        void ReportStatus(yandex::solomon::common::UrlStatusType status, TString errorMessage = {});
        void BecomeDefunct(yandex::solomon::common::UrlStatusType status, TString errorMessage = {});
        void BecomeIdle();
        void ClearError();

        TErrorOr<TFetchRequest, TGenericError> PrepareRequest() const;
        void SetFetchDuration(TDuration fetchDuration);
        bool DoesHaveDownloadErrors(const IHttpClient::TResult& fetchResultOrErr);
        void SetResponseSize(size_t responseBytes);
        bool DoesHaveContentErrors(IResponse& fetchResult);

        TInstant LastDownloadStart() const {
            return FetchState_.StartRounded;
        }

        const NMonitoring::TLabels& HostLabels() const;

        TInstant LastDownloadStartPrecise() const {
            return FetchState_.StartPrecise;
        }

        TInstant NextDownloadStart(TInstant now) const {
            const auto interval = Shard_.FetchInterval();
            auto lastStart = FetchState_.StartRounded == TInstant::Zero()
                ? RoundToPrev(now, interval)
                : FetchState_.StartRounded;

            const auto nextExpected = lastStart + interval + FetchState_.Jitter;

            if (nextExpected <= now) {
                lastStart = RoundToPrev(now, interval);
            }

            return lastStart + interval + FetchState_.Jitter;
        }

        /**
         * Determines whether the url should be served by this fetcher instance
         */
        bool IsLocal() const;

        virtual void SetNextToken(std::variant<ui64, TString> nextToken) = 0;
        virtual void ToShardData(std::vector<TParsedData> data, std::function<void(TShardData)> consumer) = 0;

    protected:
        TErrorOr<std::optional<TString>, TGenericError> GetTvmTicket() const;
        TErrorOr<std::optional<TString>, TGenericError> GetIamToken() const;

        TErrorOr<THeaders, TGenericError> MakeHeaders() const;
        virtual TErrorOr<void, TGenericError> MakeHeadersImpl(THeaders& headers) const = 0;
        virtual TErrorOr<TString, TGenericError> MakeCgiParams() const;
        virtual TErrorOr<TString, TGenericError> MakeRequestBody() const;
        virtual TFetchRequest::EMethod HttpMethod() const { return TFetchRequest::GET; }
        TErrorOr<bool, TGenericError> AddTvmTokenToHeaders(THeaders& headers) const;
        TErrorOr<bool, TGenericError> AddIamTokenToHeaders(THeaders& headers) const;
        TErrorOr<bool, TGenericError> AddSecurityHeaders(THeaders& headers) const;

        virtual THashMap<TShardKey, TString> LastResponse() const = 0;

        void UpdateSourceId();

    protected:
        THostAndLabels HostAndLabels_;
        TIpv6Address IpAddress_;

        TUrlStatus Status_;

        TFetcherShard Shard_;
        TString HostLabel_;
        NAuth::NTvm::ITicketProvider* TicketProvider_{nullptr};
        NCloud::ITokenProviderPtr IamTokenProvider_;

        struct {
            TInstant StartRounded;
            TInstant StartPrecise;
            TDuration Jitter;
        } FetchState_;

        bool ForcedDc_{false};
        const TClusterInfo& ClusterInfo_;
        TSourceId SourceId_;
        const ISourceIdFactory& SourceIdFactory_;
    };

} // namespace NSolomon::NFetcher
