#include "fetcher_url_factory.h"
#include "fetcher_url.h"

#include <solomon/services/fetcher/lib/host_groups/host_and_labels.h>
#include <solomon/services/fetcher/lib/source_id/factory.h>

#include <library/cpp/actors/core/actor.h>

using namespace NSolomon::NAuth::NTvm;

namespace NSolomon::NFetcher {

// XXX
std::unique_ptr<TFetcherUrlBase> CreateSingleShardUrl(
        TFetcherShard,
        THostAndLabels,
        ITicketProvider*,
        NCloud::ITokenProviderPtr iamTokenProvider,
        const TClusterInfo&,
        const ISourceIdFactory&);

std::unique_ptr<TFetcherUrlBase> CreateMultiShardUrl(
        TFetcherShard,
        THostAndLabels,
        ITicketProvider*,
        NCloud::ITokenProviderPtr iamTokenProvider,
        const TClusterInfo&,
        const ISourceIdFactory&);

std::unique_ptr<TFetcherUrlBase> CreateYasmAgentUrl(
        TFetcherShard,
        THostAndLabels,
        ITicketProvider*,
        NCloud::ITokenProviderPtr iamTokenProvider,
        const TClusterInfo&,
        const ISourceIdFactory&);

std::unique_ptr<NActors::IActor> CreateSingleShardUrlParser();
std::unique_ptr<NActors::IActor> CreateMultiShardUrlParser(IShardMetricsPtr);
std::unique_ptr<NActors::IActor> CreateYasmAgentUrlParser(
        TString host,
        TString yasmPrefix,
        IYasmItypeWhiteListPtr whiteList,
        IShardMetricsPtr metrics);

namespace {

class TFetcherUrlFactory: public IFetcherUrlFactory {
public:
    TFetcherUrlFactory(
            ITicketProviderPtr ticketProvider,
            NCloud::ITokenProviderPtr iamTokenProvider,
            TClusterInfo clusterInfo,
            IYasmItypeWhiteListPtr yasmWhiteList,
            TStringBuf yasmPrefix)
        : TicketProvider_{std::move(ticketProvider)}
        , IamTokenProvider_{std::move(iamTokenProvider)}
        , ClusterInfo_{std::move(clusterInfo)}
        , YasmWhiteList_(std::move(yasmWhiteList))
        , YasmPrefix_(yasmPrefix)
    {
    }

    std::unique_ptr<NActors::IActor> CreateUrlParser(const TFetcherShard& shard, TString host, IShardMetricsPtr metrics) override {
        switch (shard.Type()) {
            case EFetcherShardType::Simple:
                return CreateSingleShardUrlParser();
            case EFetcherShardType::Agent:
                return CreateMultiShardUrlParser(std::move(metrics));
            case EFetcherShardType::YasmAgent:
                return CreateYasmAgentUrlParser(std::move(host), YasmPrefix_, YasmWhiteList_, std::move(metrics));
        }
    }

    std::unique_ptr<TFetcherUrlBase> CreateUrl(TFetcherShard shard, THostAndLabels hostAndLabels) override {
        switch (shard.Type()) {
            case EFetcherShardType::Simple:
                return CreateSingleShardUrl(
                        std::move(shard),
                        std::move(hostAndLabels),
                        TicketProvider_.Get(),
                        IamTokenProvider_,
                        ClusterInfo_,
                        *SourceIdFactory_);
            case EFetcherShardType::Agent:
                return CreateMultiShardUrl(
                        std::move(shard),
                        std::move(hostAndLabels),
                        TicketProvider_.Get(),
                        IamTokenProvider_,
                        ClusterInfo_,
                        *SourceIdFactory_);
            case EFetcherShardType::YasmAgent:
                return CreateYasmAgentUrl(
                        std::move(shard),
                        std::move(hostAndLabels),
                        TicketProvider_.Get(),
                        IamTokenProvider_,
                        ClusterInfo_,
                        *SourceIdFactory_);
        }
    }

private:
    ITicketProviderPtr TicketProvider_;
    NCloud::ITokenProviderPtr IamTokenProvider_;
    TClusterInfo ClusterInfo_;
    IYasmItypeWhiteListPtr YasmWhiteList_{nullptr};
    TString YasmPrefix_;
    ISourceIdFactoryPtr SourceIdFactory_ = DefaultSourceIdFactory();
};

} // namespace

std::shared_ptr<IFetcherUrlFactory> CreateFetcherUrlFactory(
        ITicketProviderPtr ticketProvider,
        NCloud::ITokenProviderPtr iamTokenProvider,
        TClusterInfo clusterInfo,
        IYasmItypeWhiteListPtr yasmWhiteList,
        TStringBuf yasmPrefix)
{
    return std::make_shared<TFetcherUrlFactory>(
            std::move(ticketProvider),
            std::move(iamTokenProvider),
            std::move(clusterInfo),
            std::move(yasmWhiteList),
            yasmPrefix);
}

} // namespace NSolomon::NFetcher
