#include "fetcher_url.h"
#include "shard_state.h"
#include "headers.h"

#include <solomon/libs/cpp/multi_shard/proto/multi_shard.pb.h>

using namespace NSolomon::NAuth::NTvm;

namespace NSolomon::NFetcher {
namespace {

class TMultiShardUrl final: public TFetcherUrlBase {
public:
    using TFetcherUrlBase::TFetcherUrlBase;

    TMultiShardUrl(
            TFetcherShard shard,
            THostAndLabels hostAndLabels,
            ITicketProvider* ticketProvider,
            NCloud::ITokenProviderPtr iamTokenProvider,
            const TClusterInfo& clusterInfo,
            const ISourceIdFactory& sourceIdFactory)
        : TFetcherUrlBase{std::move(shard), std::move(hostAndLabels), ticketProvider, std::move(iamTokenProvider), clusterInfo, sourceIdFactory}
    {
    }

    void SetNextToken(std::variant<ui64, TString> nextToken) override {
        if (std::holds_alternative<TString>(nextToken)) {
            NextToken_ = std::get<TString>(nextToken);
        }
    }

    void ToShardData(std::vector<TParsedData> data, std::function<void(TShardData)> consumer) override {
        const auto url = DisplayUrl();
        const auto& cluster = Shard_.Cluster()->Name();
        const auto fetchInterval = Shard_.FetchInterval();

        for (auto& r: data) {
            auto& state = ShardStates_[r.Key];
            if (r.Format != state.Format && state.Format != NMonitoring::EFormat::UNKNOWN) {
                state.Data = {};
            }

            TShardState prevState{std::move(state)};
            state = {};
            state.Format = r.Format;
            state.Instant = FetchState_.StartRounded;
            state.Data = std::move(r.Data);

            TShardData sample{
                .ProjectId = std::move(r.Key.ProjectId),
                .ServiceName = std::move(r.Key.ServiceName),
                .ClusterName = r.Key.ClusterName.empty() ? cluster : r.Key.ClusterName,

                .Host = HostLabel_,
                .Url = url,
                .Interval = fetchInterval,
                .Format = state.Format,
                .Labels = HostAndLabels_.Labels,

                .Data = state.Data,
                .Instant = state.Instant,

                .PrevData = std::move(prevState.Data),
                .PrevInstant = prevState.Instant,

                .SourceId = SourceId_,
            };
            sample.ShardId = {sample.ProjectId + "_" + sample.ClusterName + "_" + sample.ServiceName, 0};

            consumer(std::move(sample));
        }
    }

    TFetchRequest::EMethod HttpMethod() const override {
        return TFetchRequest::POST;
    }

    TErrorOr<void, TGenericError> MakeHeadersImpl(THeaders& headers) const override {
        headers[NAME_SHARD_LIMIT] = ToString(Shard_.MaxMetricsPerUrl());

        if (i32 gridSec = Shard_.GridSec(); gridSec > 0) {
            headers[NAME_GRID_SECONDS] = ToString(gridSec);
        }

        return TErrorOr<void, TGenericError>::FromValue();
    }

    TErrorOr<TString, TGenericError> MakeRequestBody() const override {
        TMultiShardRequest req;
        if (!NextToken_.Empty()) {
            req.SetContinuationToken(NextToken_);
        }

        TString body;
        const auto ok = req.SerializeToString(&body);
        if (!ok) {
            return TErrorOr<TString, TGenericError>::FromError("Error while serializing protobuf");
        }

        return TErrorOr<TString, TGenericError>::FromValue(body);
    }

    THashMap<TShardKey, TString> LastResponse() const override {
        THashMap<TShardKey, TString> result;
        for (const auto& state: ShardStates_) {
            result.emplace(state.first, state.second.Data);
        }
        return result;
    }

private:
    THashMap<TShardKey, TShardState> ShardStates_;
    IShardMetricsPtr Metrics_;
    TString NextToken_;
};

} // namespace

std::unique_ptr<TFetcherUrlBase> CreateMultiShardUrl(
        TFetcherShard fetcherShard,
        THostAndLabels hostAndLabels,
        ITicketProvider* ticketProvider,
        NCloud::ITokenProviderPtr iamTokenProvider,
        const TClusterInfo& clusterInfo,
        const ISourceIdFactory& sourceIdFactory)
{
    return std::make_unique<TMultiShardUrl>(
            std::move(fetcherShard),
            std::move(hostAndLabels),
            ticketProvider,
            std::move(iamTokenProvider),
            clusterInfo,
            sourceIdFactory);
}

} // namespace NSolomon::NFetcher
