#include "fetcher_url.h"
#include "shard_state.h"
#include "headers.h"

using namespace NSolomon::NAuth::NTvm;

namespace NSolomon::NFetcher {
namespace {

class TSingleShardUrl final: public TFetcherUrlBase {
public:
    using TFetcherUrlBase::TFetcherUrlBase;

    void SetNextToken(std::variant<ui64, TString> nextToken) override {
        if (std::holds_alternative<ui64>(nextToken)) {
            State_.SeqNo = std::get<ui64>(nextToken);
        }
    }

    void ToShardData(std::vector<TParsedData> data, std::function<void(TShardData)> consumer) override {
        Y_ENSURE(data.size() == 1, "unexpected size (" << data.size() << ") of parsed data");

        auto& r = data[0];
        if (r.Format != State_.Format && State_.Format != NMonitoring::EFormat::UNKNOWN) {
            State_.Data = {};
        }

        TShardState prevState{std::move(State_)};
        State_.Instant = FetchState_.StartRounded;
        State_.Data = std::move(r.Data);
        State_.Format = r.Format;

        TShardData sample{
            .ProjectId = ToString(Shard_.ProjectId()),
            .ShardId = Shard_.Id(),

            .Host = HostLabel_,
            .Url = DisplayUrl(),
            .Interval = Shard_.FetchInterval(),

            .Format = State_.Format,
            .Labels = HostAndLabels_.Labels,

            .Data = State_.Data,
            .Instant = State_.Instant,

            .PrevData = prevState.Data,
            .PrevInstant = prevState.Instant,

            .SourceId = SourceId_,
        };

        consumer(std::move(sample));
    }

    THashMap<TShardKey, TString> LastResponse() const override {
        return {{
            TShardKey{TString{Shard_.ProjectId()}, TString{Shard_.Cluster()->Name()}, TString{Shard_.ServiceName()}},
            State_.Data
        }};
    }

    TErrorOr<void, TGenericError> MakeHeadersImpl(THeaders& headers) const override {
        headers[NAME_SEQUENCE_NUMBER] = ToString(State_.SeqNo);
        headers[NAME_SHARD_LIMIT] = ToString(Shard_.MaxMetricsPerUrl());

        if (i32 gridSec = Shard_.GridSec(); gridSec > 0) {
            headers[NAME_GRID_SECONDS] = ToString(gridSec);
        }

        return TErrorOr<void, TGenericError>::FromValue();
    }

private:
    TShardState State_;
};

} // namespace

std::unique_ptr<TFetcherUrlBase> CreateSingleShardUrl(
        TFetcherShard fetcherShard,
        THostAndLabels hostAndLabels,
        ITicketProvider* ticketProvider,
        NCloud::ITokenProviderPtr iamTokenProvider,
        const TClusterInfo& clusterInfo,
        const ISourceIdFactory& sourceIdFactory)
{
    return std::make_unique<TSingleShardUrl>(
            std::move(fetcherShard),
            std::move(hostAndLabels),
            ticketProvider,
            std::move(iamTokenProvider),
            clusterInfo,
            sourceIdFactory);
}

} // namespace NSolomon::NFetcher
