#include "fetcher_url.h"
#include "parser.h"

#include <solomon/services/fetcher/lib/dc_label/dc_label.h>
#include <solomon/services/fetcher/lib/yasm/yasm_decoder.h>

#include <solomon/libs/cpp/shard_key/shard_key.h>

#include <library/cpp/http/misc/http_headers.h>

using namespace NMonitoring;
using namespace NSolomon::NAuth::NTvm;

namespace NSolomon::NFetcher {
namespace {

    bool IsGroupCluster(const TString& cluster) {
        return cluster.StartsWith(TStringBuf{"group"});
    }

    class TFetcherUrlYasm: public TFetcherUrlBase {
    public:
        TFetcherUrlYasm(
                TFetcherShard shard,
                THostAndLabels hostAndLabels,
                ITicketProvider* ticketProvider,
                NCloud::ITokenProviderPtr iamTokenProvider,
                const TClusterInfo& clusterInfo,
                const ISourceIdFactory& sourceIdFactory)
            : TFetcherUrlBase{std::move(shard), std::move(hostAndLabels), ticketProvider, std::move(iamTokenProvider), clusterInfo, sourceIdFactory}
        {
        }

        void SetIpAddress(TIpv6Address addr, EDc dc) override {
            IpAddress_ = addr;
            Status_.Dc = dc;

            auto dcLabel = HostAndLabels_.Labels.Find(DC_LABEL_NAME);
            if (dc == EDc::UNKNOWN) {
                if (dcLabel && dcLabel->Value() != DC_AUTO) {
                    Status_.Dc = FromDcLabel(dcLabel->Value());
                }
            } else if (!ForcedDc_ && dcLabel && FromDcLabel(dcLabel->Value()) != dc) {
                HostAndLabels_.Labels.Extract(DC_LABEL_NAME);
                HostAndLabels_.Labels.Add(DC_LABEL_NAME, ToDcLabel(dc));
            }

            UpdateSourceId();
        }

        void SetNextToken(std::variant<ui64, TString> nextToken) override {
            Y_UNUSED(nextToken);
        }

        void ToShardData(std::vector<TParsedData> data, std::function<void(TShardData)> consumer) override {
            const auto url = DisplayUrl();

            for (auto& r: data) {
                TShardData sample;

                sample.Format = EFormat::SPACK;
                sample.Data = std::move(r.Data);
                sample.Instant = FetchState_.StartRounded;

                sample.ProjectId = std::move(r.Key.ProjectId);
                sample.ClusterName = std::move(r.Key.ClusterName);
                sample.ServiceName = std::move(r.Key.ServiceName);
                sample.Interval = Shard_.FetchInterval();
                sample.Host = HostLabel_;
                sample.Url = url;
                sample.SourceId = SourceId_;

                if (IsGroupCluster(sample.ClusterName)) {
                    sample.Labels = HostAndLabels_.Labels;
                }

                consumer(std::move(sample));
            }
        }

        TErrorOr<void, TGenericError> MakeHeadersImpl(THeaders& headers) const override {
            headers[NHttpHeaders::ACCEPT_ENCODING] = TStringBuf{"z-snappy"};
            headers[NHttpHeaders::ACCEPT] = TStringBuf("application/x-protobuf");
            headers[NHttpHeaders::CONTENT_TYPE] = TStringBuf("application/json");

            return TErrorOr<void, TGenericError>::FromValue();
        }

        TErrorOr<TString, TGenericError> MakeRequestBody() const override {
            return TString{
                R"({"aggr": []})"
            };
        }

        TFetchRequest::EMethod HttpMethod() const override {
            return TFetchRequest::POST;
        }

    private:
        THashMap<TShardKey, TString> LastResponse() const override {
            return {};
        }
    };

} // namespace

    std::unique_ptr<TFetcherUrlBase> CreateYasmAgentUrl(
        TFetcherShard fetcherShard,
        THostAndLabels hostAndLabels,
        ITicketProvider* ticketProvider,
        NCloud::ITokenProviderPtr iamTokenProvider,
        const TClusterInfo& clusterInfo,
        const ISourceIdFactory& sourceIdFactory)
    {
        return std::make_unique<TFetcherUrlYasm>(
            std::move(fetcherShard),
            std::move(hostAndLabels),
            ticketProvider,
            std::move(iamTokenProvider),
            clusterInfo,
            sourceIdFactory
        );
    }
} // namespace NSolomon::NFetcher
