#include "parser.h"
#include "headers.h"

#include <solomon/services/fetcher/lib/shard_manager/counters.h>

#include <solomon/libs/cpp/multi_shard/multi_shard.h>
#include <solomon/libs/cpp/shard_key/shard_key.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;
using namespace yandex::solomon::common;

namespace NSolomon::NFetcher {
namespace {

struct TMessageHandler: public NMultiShard::IMessageHandler {
    NMonitoring::EFormat Format;
    TString Token;
    TString Error;
    std::vector<TParsedData> Data;

    TMessageHandler(NMonitoring::EFormat format)
        : Format{format}
    {
    }

    TString MakeShardId(const TString& project, const TString& service) const {
        return TString::Join(project, "_", service);
    }

    bool OnHeader(NMultiShard::THeader header) override {
        switch (header.FormatVersion) {
            case NMultiShard::EFormatVersion::V1:
                break;
            case NMultiShard::EFormatVersion::Min:
            case NMultiShard::EFormatVersion::Max:
                Error = TStringBuilder() << "Invalid version format value: " << ui32(header.FormatVersion);
                return false;
        }

        Token = std::move(header.ContinuationToken);
        return true;
    }

    bool OnShardData(TString project, TString cluster, TString service, TString data) override {
        Data.emplace_back(TParsedData{
            .Format = Format,
            .Key = TShardKey{
                std::move(project),
                std::move(cluster),
                std::move(service),
            },
            .Data = std::move(data),
        });
        return true;
    }

    void OnError(TString msg) override {
        Error = std::move(msg);
    }

    void OnStreamEnd() override {
    }
};

class TMultiShardUrlParser: public TActor<TMultiShardUrlParser> {
public:
    explicit TMultiShardUrlParser(IShardMetricsPtr metrics) noexcept
        : TActor<TMultiShardUrlParser>(&TThis::StateFunc)
        , Metrics_{std::move(metrics)}
    {
    }

    STATEFN(StateFunc) {
        switch (auto type = ev->GetTypeRewrite()) {
            hFunc(TUrlParserEvents::TParse, OnParse)
            hFunc(TEvents::TEvPoison, OnPoison)
            default:
                ythrow yexception() << "unsupported event: " << type;
        }
    }

private:
    void OnParse(TUrlParserEvents::TParse::TPtr& ev) {
        TInstant start;
        if (Metrics_) {
            Metrics_->IncDecodeInflightMultishard();
            start = TInstant::Now();
        }

        Y_DEFER {
            if (Metrics_) {
                Metrics_->DecDecodeInflightMultishard();
                Metrics_->DecodeTimeMultishard(TInstant::Now() - start);
            }
        };

        auto& response = ev->Get()->Response;

        bool hasMore{false};
        if (auto maybeHasMore = response->Headers().Find(NAME_HAS_MORE)) {
            hasMore = ParseHasMoreValue(*maybeHasMore);
        }

        NMonitoring::EFormat format;
        if (auto contentType = response->Headers().Find("content-type")) {
            if (*contentType == NMultiShard::SPACK_CONTENT_TYPE) {
                format = NMonitoring::EFormat::SPACK;
            } else if (*contentType == NMultiShard::JSON_CONTENT_TYPE) {
                format = NMonitoring::EFormat::JSON;
            } else {
                auto msg = TStringBuilder() << "Unsupported content-type: " << contentType;
                Send(ev->Sender, new TUrlParserEvents::TParseError{UNKNOWN_ERROR, std::move(msg)}, 0, ev->Cookie);
                return;
            }
        } else {
            Send(ev->Sender, new TUrlParserEvents::TParseError{UNKNOWN_ERROR, "no content-type header"}, 0, ev->Cookie);
            return;
        }

        TMessageHandler handler{format};
        auto decoder = CreateMultiShardContinuousChunkDecoder(handler);
        decoder->Decode(response->Data());

        if (handler.Error) {
            Send(ev->Sender, new TUrlParserEvents::TParseError{JSON_ERROR, std::move(handler.Error)}, 0, ev->Cookie);
            return;
        }

        auto resultEvent = std::make_unique<TUrlParserEvents::TParseResult>(
            format,
            NMonitoring::ECompression::IDENTITY,
            hasMore,
            handler.Token
        );
        resultEvent->Data = std::move(handler.Data);

        Send(ev->Sender, resultEvent.release(), 0, ev->Cookie);
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    IShardMetricsPtr Metrics_;
};

} // namespace

std::unique_ptr<IActor> CreateMultiShardUrlParser(IShardMetricsPtr metrics) {
    return std::make_unique<TMultiShardUrlParser>(std::move(metrics));
}

} // namespace NSolomon::NFetcher
