#include "parser.h"
#include "headers.h"

#include <solomon/services/fetcher/lib/codec/codec.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/http/misc/http_headers.h>

using namespace NActors;

namespace NSolomon::NFetcher {
namespace {

ICodec::TDecodeResult DecodeResponse(NMonitoring::EFormat format, IResponsePtr response) {
    if (format == NMonitoring::EFormat::SPACK) {
        // do nothing, because SPACK has it own compression
        return response->ExtractData();
    }

    const auto encoding = response->Headers().Find(NHttpHeaders::CONTENT_ENCODING);
    if (encoding.Empty() || *encoding == TStringBuf{"identity"}) {
        // do nothing
        return response->ExtractData();
    }

    if (auto* codec = CodecFactory()->Codec(to_lower(TString{*encoding}))) {
        return codec->Decompress(response->Data());
    }

    // unknown encoding, return response as-is
    return response->ExtractData();
}

class TSingleShardUrlParser: public TActor<TSingleShardUrlParser> {
public:
    TSingleShardUrlParser()
        : TActor<TSingleShardUrlParser>(&TThis::StateFunc)
    {
    }

    STATEFN(StateFunc) {
        switch (auto type = ev->GetTypeRewrite()) {
            hFunc(TUrlParserEvents::TParse, OnParse)
            hFunc(TEvents::TEvPoison, OnPoison)
            default:
                ythrow yexception() << "unsupported event: " << type;
        }
    }

private:
    void OnParse(TUrlParserEvents::TParse::TPtr& ev) {
        auto& response = ev->Get()->Response;

        bool hasMore{false};
        if (auto maybeHasMore = response->Headers().Find(NAME_HAS_MORE)) {
            hasMore = ParseHasMoreValue(*maybeHasMore);
        }

        auto format = NMonitoring::EFormat::JSON;
        if (auto contentType = response->Headers().Find(NHttpHeaders::CONTENT_TYPE)) {
            format = NMonitoring::FormatFromContentType(*contentType);
            if (format != NMonitoring::EFormat::JSON && format != NMonitoring::EFormat::SPACK) {
                // there are users, who sending JSON with invalid Content-Type header,
                // for all of them we will do our best and treat their data as JSON
                format = NMonitoring::EFormat::JSON;
            }
        }

        auto compression = NMonitoring::ECompression::UNKNOWN;
        if (format == NMonitoring::EFormat::SPACK) {
            if (auto contentEncoding = response->Headers().Find(NHttpHeaders::CONTENT_ENCODING)) {
                compression = NMonitoring::CompressionFromAcceptEncodingHeader(*contentEncoding);
            }
        }

        ui64 seqNo = 0;
        if (auto seqNoParam = response->Headers().Find(NAME_NEXT_SEQUENCE_NUMBER)) {
            if (!TryFromString<ui64>(*seqNoParam, seqNo)) {
                auto msg = TStringBuilder{} << "Cannot parse seqNo " << *seqNoParam;
                auto* event = new TUrlParserEvents::TParseError{yandex::solomon::common::PARSE_ERROR, std::move(msg)};
                Send(ev->Sender, event, 0, ev->Cookie);
                return;
            }
        }

        auto decodeResult = DecodeResponse(format, std::move(response));
        if (decodeResult.Fail()) {
            auto msg = TStringBuilder{} << "Cannot decompress response " << decodeResult.Error().Message;
            auto* event = new TUrlParserEvents::TParseError{yandex::solomon::common::PARSE_ERROR, std::move(msg)};
            Send(ev->Sender, event, 0, ev->Cookie);
            return;
        }

        auto result = std::make_unique<TUrlParserEvents::TParseResult>(format, compression, hasMore, seqNo);
        result->Data.emplace_back(TParsedData{
            .Format = format,
            .Data = decodeResult.Extract()
        });

        Send(ev->Sender, result.release(), 0, ev->Cookie);
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        PassAway();
    }
};

} // namespace

std::unique_ptr<IActor> CreateSingleShardUrlParser() {
    return std::make_unique<TSingleShardUrlParser>();
}

} // namespace NSolomon::NFetcher
