#include "parser.h"
#include "headers.h"

#include <solomon/services/fetcher/lib/shard_manager/counters.h>
#include <solomon/services/fetcher/lib/yasm/yasm_decoder.h>

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/steady_timer/steady_timer.h>

#include <infra/yasm/interfaces/internal/agent.pb.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/blockcodecs/codecs.h>
#include <library/cpp/blockcodecs/stream.h>
#include <library/cpp/http/misc/http_headers.h>

#include <util/stream/mem.h>

using namespace NActors;
using namespace NYasm::NInterfaces::NInternal;

namespace NSolomon::NFetcher {
namespace {

const TString YASM_SERVICE{"yasm"};

TErrorOr<TAgentResponse*, TGenericError> ParseAgentResponse(google::protobuf::Arena* arena, IResponsePtr response) {
    // yasm-agent supports only snappy encoding
    static const NBlockCodecs::ICodec* snappyCodec = NBlockCodecs::Codec("snappy");

    auto* proto = NProtoBuf::Arena::CreateMessage<TAgentResponse>(arena);
    TStringBuf responseData = response->Data();

    auto encoding = response->Headers().Find(NHttpHeaders::CONTENT_ENCODING);
    if (encoding && *encoding == TStringBuf{"z-snappy"}) {
        // process compressed response
        TMemoryInput data{responseData};
        NBlockCodecs::TDecodedInput decoded(&data, snappyCodec);
        if (!proto->ParseFromArcadiaStream(&decoded)) {
            return TGenericError{"cannot parse TAgentResponse from compressed response"};
        }
        return proto;
    }

    // process plain response
    if (!proto->ParseFromArray(responseData.data(), static_cast<int>(responseData.size()))) {
        return TGenericError{"cannot parse TAgentResponse from plain response"};
    }

    return proto;
}

class TResultConsumer: public NYasm::IDataConsumer {
public:
    TResultConsumer(TStringBuf yasmPrefix, std::vector<TParsedData>* result) noexcept
        : YasmPrefix_{yasmPrefix}
        , Result_{result}
    {
    }

    void OnShardData(NYasm::TYasmShardKey key, TString data) override {
        Result_->emplace_back(TParsedData{
            .Format = NMonitoring::EFormat::SPACK,
            .Key = TShardKey{
                TString::Join(YasmPrefix_, key.GetItype()),
                TString{key.GetCluster()},
                YASM_SERVICE
            },
            .Data = std::move(data),
        });
    }

private:
    TStringBuf YasmPrefix_;
    std::vector<TParsedData>* Result_;
};

class TYasmAgentUrlParser: public TActor<TYasmAgentUrlParser> {
public:
    TYasmAgentUrlParser(TString host, TString yasmPrefix, IYasmItypeWhiteListPtr whiteList, IShardMetricsPtr metrics)
        : TActor<TYasmAgentUrlParser>(&TThis::StateFunc)
        , YasmPrefix_{std::move(yasmPrefix)}
        , WhiteList_{std::move(whiteList)}
        , Decoder_{NYasm::CreateYasmAgentDecoder(host, NYasm::CreateMultiShardEncoder(NMonitoring::ECompression::IDENTITY), WhiteList_.Get())}
        , Metrics_{std::move(metrics)}
        , Host_{std::move(host)}
    {
    }

    STATEFN(StateFunc) {
        switch (auto type = ev->GetTypeRewrite()) {
            hFunc(TUrlParserEvents::TParse, OnParse)
            hFunc(TEvents::TEvPoison, OnPoison)
            default:
                ythrow yexception() << "unsupported event: " << type;
        }
    }

private:
    void OnParse(TUrlParserEvents::TParse::TPtr& ev) {
        TSteadyTimer timer;
        if (Metrics_) {
            Metrics_->IncDecodeInflight();
        }

        Y_DEFER {
            TDuration duration = timer.Step();
            if (duration >= TDuration::Seconds(1)) {
                MON_INFO(UrlParser, "heavy response from " << Host_ << ", took " << duration << " to parse");
            }
            if (Metrics_) {
                Metrics_->DecDecodeInflight();
                Metrics_->DecodeTime(duration);
            }
        };

        NProtoBuf::Arena arena;
        auto agentResponse = ParseAgentResponse(&arena, std::move(ev->Get()->Response));
        if (agentResponse.Fail()) {
            Send(ev->Sender, new TUrlParserEvents::TParseError{
                    yandex::solomon::common::PARSE_ERROR,
                    agentResponse.Error().MessageString()});
            return;
        }

        try {
            auto resultEvent = std::make_unique<TUrlParserEvents::TParseResult>(
                    NMonitoring::EFormat::SPACK,
                    NMonitoring::ECompression::IDENTITY,
                    false,
                    ui64(0));

            TResultConsumer consumer{YasmPrefix_, &resultEvent->Data};
            Decoder_->Decode(
                    agentResponse.Value()->GetPerInstanceRecords(),
                    agentResponse.Value()->GetAggregatedRecords(),
                    &consumer);

            Send(ev->Sender, resultEvent.release(), 0, ev->Cookie);
        } catch (...) {
            auto* error = new TUrlParserEvents::TParseError{
                    yandex::solomon::common::UrlStatusType::PARSE_ERROR,
                    CurrentExceptionMessage()};
            Send(ev->Sender, error, 0, ev->Cookie);
        }
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    const TString YasmPrefix_;
    IYasmItypeWhiteListPtr WhiteList_;
    TIntrusivePtr<NYasm::IYasmAgentDecoder> Decoder_;
    IShardMetricsPtr Metrics_;
    TString Host_;
};

} // namespace

std::unique_ptr<IActor> CreateYasmAgentUrlParser(
        TString host,
        TString yasmPrefix,
        IYasmItypeWhiteListPtr whiteList,
        IShardMetricsPtr metrics)
{
    return std::make_unique<TYasmAgentUrlParser>(
            std::move(host),
            std::move(yasmPrefix),
            std::move(whiteList),
            std::move(metrics));
}

} // namespace NSolomon::NFetcher
