#include "url.h"
#include "interval.h"
#include "fetcher_url.h"
#include "parser.h"

#include <solomon/libs/cpp/auth/core/authenticator.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <solomon/services/fetcher/lib/app_data.h>
#include <solomon/services/fetcher/lib/auth/auth_token.h>
#include <solomon/services/fetcher/lib/dns/continuous_resolver.h>
#include <solomon/services/fetcher/lib/multishard_pulling/auth_gatekeeper.h>
#include <solomon/services/fetcher/lib/shard_manager/events.h>
#include <solomon/services/fetcher/lib/shard_manager/local_events.h>
#include <solomon/services/fetcher/lib/shard_manager/shard_stat.h>
#include <solomon/services/fetcher/lib/sink/sink.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/encode/text/text.h>

#include <util/generic/scope.h>
#include <util/stream/str.h>

using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon::NAuth;
using namespace yandex::solomon::common;

TString DecodeData(yandex::solomon::common::UrlContentType format, const TString& encoded) try {
    TMemoryInput in{encoded};
    TStringStream out;

    auto encoder = NMonitoring::EncoderText(&out);
    switch (format) {
        case JSON:
            NMonitoring::DecodeJson(encoded, encoder.Get());
            break;
        case SPACK:
            NMonitoring::DecodeSpackV1(&in, encoder.Get());
            break;
        default:
            return TString{"Unsupported data type "} + UrlContentType_Name(format);
    };

    return out.Str();
} catch (...) {
    return TString{"Error decoding data: "} + CurrentExceptionMessage();
}

IOutputStream& operator<<(IOutputStream& os, const NSolomon::NFetcher::TUrlStatus& status) {
    os << "Status: " << UrlStatusType_Name(status.UrlStatus) << "\n"
        << "Last fetch time: " << status.FetchTime << "\n"
        << "Reponse bytes: " << status.ResponseBytes << "\n"
        << "Error: " << status.UrlError;

    return os;
}

#define TRACING_OUTPUT(msg) \
    do { \
        if (Y_UNLIKELY(Tracing_)) { \
            MON_INFO(Tracing, msg); \
        } \
    } while (0)

#define URL_WARN(s) MON_WARN(Downloader, "[URL] " << ShardConf_.Id() << ' ' << UrlHandler_->DisplayUrl() << ' ' << s)
#define URL_DEBUG(s) MON_DEBUG(Downloader, "[URL] " << ShardConf_.Id() << ' ' << UrlHandler_->DisplayUrl() << ' ' << s)
#define URL_INFO(s) MON_INFO(Downloader, "[URL] " << ShardConf_.Id() << ' ' << UrlHandler_->DisplayUrl() << ' ' << s)
#define URL_ERR(s) MON_ERROR(Downloader, "[URL] " << ShardConf_.Id() << ' ' << UrlHandler_->DisplayUrl() << ' ' << s)

namespace NSolomon::NFetcher {
namespace {

    class TUrlActor: public TActorBootstrapped<TUrlActor> {
    public:
        TUrlActor(
            THostAndLabels hostAndLabels,
            TFetcherShard shardConf, IFetcherUrlFactory& urlFactory,
            TActorId parserId,
            TActorId dnsResolverId, TActorId dataSinkId, TActorId statCollectorId, TActorId authGatekeeper,
            IShardMetricsPtr shardCounters, ILimiterPtr limiter, IHttpClientPtr httpClient
        )
            : ShardConf_{std::move(shardConf)}
            , ParserId_{parserId}
            , DataSinkId_{dataSinkId}
            , StatCollectorId_{statCollectorId}
            , AuthGatekeeper_{authGatekeeper}
            , ContinuousResolverId_{dnsResolverId}
            , ShardCounters_{std::move(shardCounters)}
            , Limiter_{std::move(limiter)}
            , HttpClient_{std::move(httpClient)}
        {
            AuthHeaderParser_ = CreateAuthenticatorMultiplexer({{
                CreateFakeAuthenticator(NAuth::EAuthType::Iam),
                CreateFakeAuthenticator(NAuth::EAuthType::TvmService),
            }});
            UrlHandler_ = urlFactory.CreateUrl(ShardConf_, std::move(hostAndLabels));
        }

        void Bootstrap() {
            AppData_ = &GetAppData();
            DoSkipAuth_ = AppData_->FetchConfig().DoSkipAuth;
            SkipProviderDataWithWrongService_ = AppData_->FetchConfig().SkipProviderDataWithWrongService;
            ShardCounters_->AddUrlCount(1);

            if (!UrlHandler_->IpAddress()) {
                Become(&TThis::DefunctStateFunc);
                auto fqdn = UrlHandler_->Fqdn();
                Y_VERIFY_DEBUG(!fqdn.empty());
                if (fqdn.empty()) {
                    return;
                }

                Send(ContinuousResolverId_, new TEvStartResolving{fqdn});
            } else if (UrlHandler_->IsLocal()) {
                BecomeWorking();
            } else {
                BecomeIdle();
            }
        }

    private:
        void BecomeDefunct(UrlStatusType newStatus) {
            if (State_ == EUrlActorState::Defunct && newStatus == UrlHandler_->Status().UrlStatus) {
                return;
            }

            UrlHandler_->BecomeDefunct(newStatus);
            ResetInflight();
            ReportUrlStatus();
            State_ = EUrlActorState::Defunct;
            Become(&TThis::DefunctStateFunc);
            Schedule(ShardConf_.FetchInterval(), new TEvents::TEvWakeup{DEFUNCT_WAKEUP_TAG});
        }

        void BecomeIdle() {
            ShardCounters_->AddIdle(1);
            ResetInflight();
            State_ = EUrlActorState::Idle;
            UrlHandler_->BecomeIdle();
            URL_INFO(UrlHandler_->Fqdn() << " will now become idle");
            Become(&TThis::IdleStateFunc);
        }

        void BecomeWorking() {
            Y_VERIFY_DEBUG(State_ != EUrlActorState::Working);
            if (State_ == EUrlActorState::Working) {
                return;
            }

            ScheduleDownload();
            ReportUrlStatus();
            State_ = EUrlActorState::Working;
            Become(&TThis::WorkingStateFunc);
        }

        STFUNC(DefunctStateFunc) {
            switch (ev->GetTypeRewrite()) {
                CFunc(TEvents::TSystem::PoisonPill, OnPoisonPill);
                hFunc(TEvHostResolveOk, OnInitialResolveOk);
                cFunc(TShardEvents::EvStartTracing, OnStartTracing);
                cFunc(TShardEvents::EvStopTracing, OnStopTracing);
                hFunc(TEvShardChanged, OnShardChanged);
                cFunc(TEvents::TSystem::Wakeup, OnWakeupDefunct);
                hFunc(NSelfMon::TEvPageDataReq, OnSelfMon);
            }
        }

        STFUNC(WorkingStateFunc) {
            switch (ev->GetTypeRewrite()) {
                CFunc(TEvents::TSystem::PoisonPill, OnPoisonPill);
                hFunc(TEvHostResolveOk, OnResolveOk);
                hFunc(TEvHostResolveFail, OnResolveFail);
                hFunc(TEvMetricDataWritten, OnWriteResult);
                cFunc(TShardEvents::EvStartTracing, OnStartTracing);
                cFunc(TShardEvents::EvStopTracing, OnStopTracing);
                hFunc(TEvShardChanged, OnShardChanged);
                hFunc(TEvents::TEvWakeup, OnDownload);
                hFunc(TEvDownloadCompleted, OnDownloadCompleted);
                hFunc(TUrlParserEvents::TParseResult, OnParseResult);
                hFunc(TUrlParserEvents::TParseError, OnParseError);
                hFunc(NSelfMon::TEvPageDataReq, OnSelfMon);
            }
        }

        STFUNC(IdleStateFunc) {
            switch (ev->GetTypeRewrite()) {
                CFunc(TEvents::TSystem::PoisonPill, OnPoisonPill);
                hFunc(TEvHostResolveOk, OnInitialResolveOk);
                cFunc(TShardEvents::EvStartTracing, OnStartTracing);
                cFunc(TShardEvents::EvStopTracing, OnStopTracing);
                hFunc(NSelfMon::TEvPageDataReq, OnSelfMon);
            }
        }

        void OnWakeupDefunct() {
            ReportUrlStatus();
            Schedule(ShardConf_.FetchInterval(), new TEvents::TEvWakeup{DEFUNCT_WAKEUP_TAG});
        }

        void OnDownload(const TEvents::TEvWakeup::TPtr& ev) {
            ++Epoch_;

            if (ev->Get()->Tag == DEFUNCT_WAKEUP_TAG) {
                return;
            }

            const auto now = TActivationContext::Now();
            TInstant roundedStart = RoundToPrev(now, ShardConf_.FetchInterval());

            if (roundedStart <= UrlHandler_->LastDownloadStart()) {
                MON_WARN(Downloader,
                    "previous start: " << UrlHandler_->LastDownloadStart() << " is >= than the current start "
                    << roundedStart << " for " << UrlHandler_->DisplayUrl());
                Schedule(TDuration::MilliSeconds(RandomNumber(20u)), new TEvents::TEvWakeup);
                return;
            }

            auto result = UrlHandler_->PrepareRequest();
            if (!result.Success()) {
                MON_ERROR(Downloader, "Failed to prepare request to "
                        << UrlHandler_->DisplayUrl() << ": " << result.Error().Message());

                ReportUrlStatus();
                // XXX distinguish between permanent and intermittent errors?
                ScheduleDownload();
                return;
            }

            if (!Limiter_->TryInc()) {
                ShardCounters_->Postponed();
                Schedule(TDuration::MilliSeconds(RandomNumber(200u)), new TEvents::TEvWakeup);
                return;
            }

            RequestInFlight_ = true;
            ShardCounters_->IncInflight();

            auto req = result.Extract();
            UrlHandler_->SetDownloadTime(now, roundedStart);

            TDownloadRequest r{
                std::move(req.Url),
                std::move(req.CgiParams),
                std::move(req.Headers),
                req.MaxResponseSizeBytes,
                roundedStart,
                std::move(req.Body),
                req.Method == TFetchRequest::POST,
            };

            Register(
                CreateHttpDownloadActor(
                    {HttpClient_, ShardConf_.FetchInterval(), ConnectTimeout_},
                    std::move(r),
                    SelfId()
                ),
                TMailboxType::Simple);
        }

        bool IsProviderData() const {
            return ShardConf_.Type() == EFetcherShardType::Agent;
        }

        void OnDownloadCompleted(const TEvDownloadCompleted::TPtr& evPtr) {
            Y_DEFER {
                ScheduleDownload();
            };

            Limiter_->Dec();
            RequestInFlight_ = false;
            ShardCounters_->DecInflight();

            auto& ev = *evPtr->Get();
            RecordStats(ev.Result, ev.Duration);

            if (ev.Result.Success()) {
                URL_DEBUG("download ok: " << HttpCodeStrEx(ev.Result.Value()->Code()));
            } else {
                const auto& err = ev.Result.Error();
                URL_INFO("download error (" << err.Type() << "): " << err.Message());
            }

            UrlHandler_->SetFetchDuration(ev.Duration);

            if (UrlHandler_->DoesHaveDownloadErrors(ev.Result)) {
                URL_DEBUG(UrlHandler_->Status().UrlError);

                UrlHandler_->SetResponseSize(0);
                ReportUrlStatus();
                return;
            }

            auto& response = *ev.Result.Value();
            UrlHandler_->SetResponseSize(response.Data().size());

            if (UrlHandler_->DoesHaveContentErrors(response)) {
                URL_DEBUG(UrlHandler_->Status().UrlError);
                ReportUrlStatus();
                return;
            }

            ParseResponse(evPtr->Get()->Result);
        }

        void ParseResponse(IHttpClient::TResult& httpResult) {
            auto fetchResponse = httpResult.Extract();

            if (IsProviderData() && !DoSkipAuth_) {
                auto tokenOrErr = AuthHeaderParser_->GetToken(&fetchResponse->Headers());
                if (tokenOrErr.Fail()) {
                    URL_WARN(tokenOrErr.Error().GetMessage());
                    UrlHandler_->BecomeDefunct(AUTH_ERROR, tokenOrErr.Error().GetMessage());
                    ReportUrlStatus();
                    return;
                }

                if (tokenOrErr.Value().Type != NAuth::EAuthType::Iam &&
                    tokenOrErr.Value().Type != NAuth::EAuthType::TvmService)
                {
                    auto errMsg = TStringBuilder()
                            << "unknown auth type. expected IAM or TvmService; got: " << tokenOrErr.Value().Type;
                    URL_WARN(errMsg);
                    UrlHandler_->BecomeDefunct(AUTH_ERROR, std::move(errMsg));
                    ReportUrlStatus();
                    return;
                }

                // TODO(ivanzhukov): store a type as well and send a token to the appropriate AuthGatekeeper depending on a type (IAM or TVM)
                AuthToken_ = tokenOrErr.Value().Value;
            }

            Send(ParserId_, new TUrlParserEvents::TParse{std::move(fetchResponse)});
        }

        void OnParseResult(TUrlParserEvents::TParseResult::TPtr& ev) {
            UrlHandler_->ClearError();
            UrlHandler_->SetContentType(ev->Get()->Format, ev->Get()->Compression);
            UrlHandler_->SetNextToken(ev->Get()->NextToken);

            // TODO(msherbakov): remove once SOLOMON-4481 is closed
            // for now this value is saved to be able to indicate metric overflow
            UrlHasMore_ = ev->Get()->HasMore;
            ShardCounters_->ReportFormat(ev->Get()->Format, ev->Get()->Compression);

            if (UrlHasMore_) {
                UrlHandler_->ReportStatus(SENSOR_OVERFLOW);
            } else {
                UrlHandler_->ReportStatus(OK);
            }

            ReportUrlStatus();

            if (IsProviderData() && !DoSkipAuth_) {
                const auto& providerId = ShardConf_.Cluster()->Id().ProjectId();

                UrlHandler_->ToShardData(std::move(ev->Get()->Data), [this, &providerId](TShardData shardData) {
                    const auto& service = shardData.ServiceName;

                    if (!service.empty() && service != providerId) {
                        TStringBuilder errMsg;
                        errMsg << "service provider \"" << providerId << "\" writes data to service \"" << service << "\"";
                        URL_WARN(errMsg);

                        ui64 metricsParsed = 0;
                        ReportShardStatus(shardData.ShardId.StrId(), AUTH_ERROR, metricsParsed, std::move(errMsg));

                        if (SkipProviderDataWithWrongService_) {
                            return;
                        }
                    }

                    Send(AuthGatekeeper_, new TAuthGatekeeperEvents::TExamine{
                        std::move(shardData),
                        providerId,
                        AuthToken_,
                        UrlHandler_->DisplayUrl()
                    });
                });
            } else {
                UrlHandler_->ToShardData(std::move(ev->Get()->Data), [this](auto shardData) {
                    Send(DataSinkId_, new TEvSinkWrite{std::move(shardData)});
                });
            }
        }

        void OnParseError(TUrlParserEvents::TParseError::TPtr& ev) {
            const auto& err = *ev->Get();
            URL_WARN("unable to parse data: " << UrlStatusType_Name(err.Status) << ' ' << err.Message);
            UrlHandler_->BecomeDefunct(err.Status, err.Message);
            ReportUrlStatus();
        }

        void OnStartTracing() {
            Tracing_ = true;
            TStringStream os;

            auto fqdn = UrlHandler_->Fqdn();
            os << "Status for " << (!fqdn.empty() ? fqdn : UrlHandler_->IpAddress()->ToString()) << ": " << UrlHandler_->Status();
            TRACING_OUTPUT(os.Str());
        }

        void OnStopTracing() {
            Tracing_ = false;
        }

        void OnInitialResolveOk(TEvHostResolveOk::TPtr& evPtr) {
            auto& ev = *evPtr->Get();
            URL_INFO("got DNS record for " << UrlHandler_->Fqdn() << ' ' << ev.Address.ToString(false) << ' ' << ev.Dc);
            UrlHandler_->SetIpAddress(ev.Address, ev.Dc);

            if (UrlHandler_->IsLocal()) {
                BecomeWorking();
            } else {
                BecomeIdle();
            }
        }

        void OnResolveOk(TEvHostResolveOk::TPtr& evPtr) {
            auto& ev = *evPtr->Get();
            TRACING_OUTPUT("DNS record for " << UrlHandler_->Fqdn() << " changed from "
                << UrlHandler_->IpAddress()->ToString(false) << " to " << ev.Address.ToString(false)
                << ", dc=" << ev.Dc);

            URL_INFO("DNS record for " << UrlHandler_->Fqdn() << " changed from "
                << UrlHandler_->IpAddress()->ToString(false) << " to " << ev.Address.ToString(false)
                << ", dc=" << ev.Dc);

            UrlHandler_->SetIpAddress(ev.Address, ev.Dc);

            if (!UrlHandler_->IsLocal()) {
                BecomeIdle();
            } else if (State_ == EUrlActorState::Defunct) {
                // activate the URL actor again if there were some DNS errors before
                BecomeWorking();
            }
        }

        void OnResolveFail(TEvHostResolveFail::TPtr& evPtr) {
            auto& ev = *evPtr->Get();
            if (ev.Type == EErrorType::Timeout) {
                URL_WARN("DNS timed out for " << UrlHandler_->Fqdn());
            } else {
                URL_WARN("DNS request failed for " << UrlHandler_->Fqdn() << ": " << ev.Reason);
                BecomeDefunct(UrlStatusType::UNKNOWN_HOST);
            }
        }

        void OnWriteResult(const TEvMetricDataWritten::TPtr& evPtr) {
            TRACING_OUTPUT("Write to the sink done for " << UrlHandler_->EffectiveUrl() << " with status "
                << UrlStatusType_Name(evPtr->Get()->Status) << " written " << evPtr->Get()->MetricsParsed << " metrics");

            auto&& ev = *evPtr->Get();
            ReportShardStatus(std::move(ev.ShardId), ev.Status, ev.MetricsParsed, std::move(ev.Error));
        }

        void OnPoisonPill(const TActorContext&) {
            auto resolverId = MakeDnsResolverId();
            Send(resolverId, new TEvStopResolving{UrlHandler_->Fqdn()});
            RemoveStatus(UrlHandler_->DisplayUrl());
            Send(ParserId_, new TEvents::TEvPoison);

            if (RequestInFlight_) {
                Limiter_->Dec();
                RequestInFlight_ = false;
            }

            if (IsIdle()) {
                ShardCounters_->AddIdle(-1);
            }

            ShardCounters_->AddUrlCount(-1);
            ShardCounters_.reset();
            PassAway();
        }

        void OnSelfMon(NSelfMon::TEvPageDataReq::TPtr& ev) {
            TUrlState state = UrlHandler_->State();

            ::yandex::monitoring::selfmon::Page page;
            page.set_title(TStringBuilder{} << "URL: " << state.DisplayUrl);
            auto* grid = page.mutable_grid();

            auto* panelBodyObject = grid->add_rows()->add_columns()->mutable_component()->mutable_object();
            auto addField = [panelBodyObject](TString name, TString value) {
                auto* field = panelBodyObject->add_fields();
                field->set_name(std::move(name));
                auto* fieldValue = field->mutable_value();
                fieldValue->set_string(std::move(value));
            };

            switch (State_) {
                case EUrlActorState::Defunct:
                    addField("URL State", "Defunct");
                    break;
                case EUrlActorState::Working:
                    addField("URL State", "Working");
                    break;
                case EUrlActorState::Idle:
                    addField("URL State", "Idle");
                    break;
            }

            addField("Datacenter", ToString(state.Status.Dc));
            addField("Display URL", state.DisplayUrl);
            addField("Effective URL", state.EffectiveUrl);
            addField("Fetch Status", UrlStatusType_Name(state.Status.UrlStatus));
            addField("IP Address", state.IpAddress.ToString());
            addField("SourceId", ToString(state.SourceId));
            addField("Format", UrlContentType_Name(state.Status.ContentType));
            addField("Last fetch time", ToString(state.Status.FetchTime));

            for (auto&& [key, data]: state.LastResponse) {
                auto* panelGrid = grid->add_rows()->add_columns()->mutable_grid();

                auto* panelKey = panelGrid->add_rows()->add_columns()->mutable_component()->mutable_code();
                panelKey->set_content((TStringBuilder() << key));

                auto* panelWell = panelGrid->add_rows()->add_columns()->mutable_component()->mutable_code();
                panelWell->set_content((TStringBuilder() << DecodeData(state.Status.ContentType, data)));
            }

            Send(ev->Sender, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        }

        void OnShardChanged(const TEvShardChanged::TPtr& ev) {
            ShardConf_ = std::move(ev->Get()->Shard);
            auto oldUrl = UrlHandler_->DisplayUrl();
            UrlHandler_->SetFetcherShard(ShardConf_);
            auto newUrl = UrlHandler_->DisplayUrl();

            if (oldUrl != newUrl) {
                RemoveStatus(std::move(oldUrl));
                ReportUrlStatus();
            }
        }

    private:
        void ReportUrlStatus() {
            auto s = UrlHandler_->Status();
            ShardCounters_->ReportStatus(s.UrlStatus);
            Send(StatCollectorId_, new TEvReportUrlStatus{Epoch_, std::move(s)});
        }

        void ReportShardStatus(
                TString shardId,
                yandex::solomon::common::UrlStatusType status,
                ui64 metricsParsed,
                TString&& error)
        {
            ShardCounters_->ReportMetricsParsed(metricsParsed, UrlHandler_->Status().Dc);

            TInstant lastOverflow = (status == IPC_QUEUE_OVERFLOW) ? TActivationContext::Now() : TInstant::Zero();
            ShardCounters_->ReportStatus(status);
            Send(StatCollectorId_,
                 new TEvReportShardStatus{
                         UrlHandler_->Status().Url,
                         TShardStatus{
                                 .ShardId = std::move(shardId),
                                 .Status = status,
                                 .Error = std::move(error),
                                 .MetricsParsed = metricsParsed,
                                 .LastOverflow = lastOverflow,
                         }});
        }

        void ResetInflight() {
            if (RequestInFlight_) {
                Limiter_->Dec();
                RequestInFlight_ = false;
            }
        }

        void RemoveStatus(TString url) {
            Send(StatCollectorId_, new TEvRemoveStats{std::move(url)});
        }

        void ScheduleDownload() {
            const auto now = TActivationContext::Now();
            auto delay = UrlHandler_->NextDownloadStart(now) - now;
            auto shard = ShardConf_.Id();
            MON_DEBUG(Downloader, "Schedule " << UrlHandler_->EffectiveUrl()
                    << " host: " << UrlHandler_->Fqdn() << " shard: " << shard << " in " << delay);
            Schedule(delay, new TEvents::TEvWakeup{});
        }

        bool IsIdle() const {
            return State_ == EUrlActorState::Idle;
        }

    private:
        void RecordStats(const IHttpClient::TResult& result, TDuration duration) {
            if (result.Success() && result.Value()->Code() == HTTP_OK) {
                ShardCounters_->DownloadSuccess();
            } else if (!result.Success() && result.Error().Type() == TRequestError::EType::ReadTimeout) {
                ShardCounters_->DownloadFail();
                ShardCounters_->DownloadTimeout();
            } else {
                ShardCounters_->DownloadFail();
            }

            if (result.Success()) {
                ShardCounters_->ReportResponseSize(result.Value()->Data().size(), UrlHandler_->Status().Dc, ShardConf_.ProjectId());
            }

            ShardCounters_->RecordDownloadDuration(duration.MilliSeconds());
        }

    private:
        EUrlActorState State_{EUrlActorState::Defunct};
        bool RequestInFlight_{false};
        bool UrlHasMore_{false};
        TString AuthToken_;

        std::unique_ptr<TFetcherUrlBase> UrlHandler_;
        ui64 Epoch_{0};
        TFetcherShard ShardConf_;

        const TAppData* AppData_;

        TActorId ParserId_;
        TActorId DataSinkId_;
        TActorId StatCollectorId_;
        TActorId AuthGatekeeper_;
        TActorId ContinuousResolverId_;

        IShardMetricsPtr ShardCounters_;
        ILimiterPtr Limiter_;

        bool Tracing_{false};

        IHttpClientPtr HttpClient_;
        IAuthenticatorPtr AuthHeaderParser_;
        TDuration ConnectTimeout_{TDuration::Seconds(5)};

        bool DoSkipAuth_{false};
        bool SkipProviderDataWithWrongService_{false};
    };

} // namespace

    IActor* CreateUrlActor(
            THostAndLabels hostAndLabels, TFetcherShard shardConf, IFetcherUrlFactory& urlFactory,
            TActorId parserId,
            TActorId dnsResolverId, TActorId dataSinkId, TActorId statCollectorId, TActorId authGatekeeper,
            IShardMetricsPtr metrics, ILimiterPtr limiter, IHttpClientPtr httpClient)
    {
        return new TUrlActor{
            std::move(hostAndLabels),
            std::move(shardConf),
            urlFactory,
            parserId,
            dnsResolverId,
            dataSinkId,
            statCollectorId,
            authGatekeeper,
            std::move(metrics),
            std::move(limiter),
            std::move(httpClient),
        };
    }
} // namespace NSolomon::NFetcher
