#include "mock_matcher.h"

#include <solomon/services/fetcher/lib/dc_label/dc_label.h>
#include <solomon/services/fetcher/lib/url/fetcher_url.h>
#include <solomon/services/fetcher/lib/url/fetcher_url_factory.h>
#include <solomon/services/fetcher/lib/url/headers.h>
#include <solomon/services/fetcher/testlib/fetcher_shard.h>
#include <solomon/services/fetcher/testlib/http_client.h>
#include <solomon/services/fetcher/testlib/tvm.h>

#include <solomon/libs/cpp/multi_shard/multi_shard.h>
#include <solomon/libs/cpp/multi_shard/proto/multi_shard.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/serialized_enum.h>
#include <util/string/cast.h>

using namespace NMonitoring;
using namespace NSolomon;
using namespace NSolomon::NMultiShard;
using namespace NSolomon::NTesting;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NDb::NModel;
using namespace yandex::solomon::common;

using namespace testing;

namespace NSolomon {
    bool operator==(const TMultiShardRequest& lhs, const TMultiShardRequest& rhs) {
        return lhs.GetContinuationToken() == rhs.GetContinuationToken();
    }
} // namespace NSolomon

const TString IamToken = "fetcher-iam-test-token";

class TFetcherUrlTest: public ::testing::Test {
public:
    void SetUp() override {
        MockShard_ = ShardFactory_.MakeShard();
        Shard_.Reset(new TFetcherShard{MockShard_});
        auto iamTokenProvider = NSolomon::NCloud::CreateStaticTokenProvider(IamToken);
        UrlFactory_ = CreateFetcherUrlFactory(MakeIntrusive<TMockTicketProvider>(), iamTokenProvider, TClusterInfo{"cluster"});
    }

protected:
    TIntrusiveConstPtr<TMockShard> MockShard_;
    THolder<TFetcherShard> Shard_;
    THostAndLabels HostAndLabels_{"my.hostname"};
    TFetcherShardFactory ShardFactory_;
    std::shared_ptr<IFetcherUrlFactory> UrlFactory_;
};

TEST_F(TFetcherUrlTest, Init) {
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
    ASSERT_THAT(fetcherUrl->Fqdn(), StrEq("my.hostname"));
    ASSERT_THAT(fetcherUrl->HostLabel(), StrEq("my"));
    ASSERT_FALSE(fetcherUrl->IpAddress());
}

TEST_F(TFetcherUrlTest, ForcedDc) {
    THostAndLabels hl("localhost", {{"DC", "Sas"}});
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, hl);
    bool ok;

    ASSERT_EQ(fetcherUrl->Status().Dc, EDc::SAS);

    // VEG DC
    fetcherUrl->SetIpAddress(TIpv6Address::FromString("::0", ok), EDc::UNKNOWN);

    ASSERT_TRUE(fetcherUrl->HostLabels().Find("DC"));
    ASSERT_EQ(fetcherUrl->HostLabels().Find("DC")->Value(), "Sas");

    // UNKNOWN DC
    fetcherUrl->SetIpAddress(TIpv6Address::FromString("::4", ok), EDc::UNKNOWN);

    ASSERT_TRUE(fetcherUrl->HostLabels().Find("DC"));
    ASSERT_EQ(fetcherUrl->HostLabels().Find("DC")->Value(), "Sas");
}

TEST_F(TFetcherUrlTest, ChangeDc_WithDcAuto) {
    bool ok;
    TIpv6Address address = TIpv6Address::FromString("2a02:6b8:c03:35d:0:4c7d:b2a9:67d2", ok);

    THostAndLabels hl("localhost", {{DC_LABEL_NAME, "auto"}});
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, hl);

    {
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_FALSE(dcLabel);
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }

    // UNKNOWN -> UNKNOWN
    {
        fetcherUrl->SetIpAddress(address, EDc::UNKNOWN);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "dc-unknown");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }

    // UNKNOWN -> SAS
    {
        fetcherUrl->SetIpAddress(address, EDc::SAS);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "Sas");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::SAS);
    }

    // SAS -> MYT
    {
        fetcherUrl->SetIpAddress(address, EDc::MYT);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "Myt");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::MYT);
    }

    // MYT -> UNKNOWN
    {
        fetcherUrl->SetIpAddress(address, EDc::UNKNOWN);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "dc-unknown");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }
}

TEST_F(TFetcherUrlTest, ChangeDc_WithoutDc) {
    bool ok;
    TIpv6Address address = TIpv6Address::FromString("2a02:6b8:c03:35d:0:4c7d:b2a9:67d2", ok);

    THostAndLabels hl("localhost", {});
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, hl);

    {
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_FALSE(dcLabel);
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }

    // UNKNOWN -> UNKNOWN
    {
        fetcherUrl->SetIpAddress(address, EDc::UNKNOWN);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "dc-unknown");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }

    // UNKNOWN -> SAS
    {
        fetcherUrl->SetIpAddress(address, EDc::SAS);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "Sas");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::SAS);
    }

    // SAS -> VLA
    {
        fetcherUrl->SetIpAddress(address, EDc::MYT);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "Myt");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::MYT);
    }

    // MYT -> UNKNOWN
    {
        fetcherUrl->SetIpAddress(address, EDc::UNKNOWN);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "dc-unknown");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }
}

TEST_F(TFetcherUrlTest, ChangeDc_WithInvalidDc) {
    bool ok;
    TIpv6Address address = TIpv6Address::FromString("2a02:6b8:c03:35d:0:4c7d:b2a9:67d2", ok);

    THostAndLabels hl("localhost", {{DC_LABEL_NAME, "Prestable"}}); // real case from production
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, hl);

    {
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "Prestable");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }

    // UNKNOWN -> UNKNOWN
    {
        fetcherUrl->SetIpAddress(address, EDc::UNKNOWN);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "Prestable");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }

    // UNKNOWN -> SAS
    {
        fetcherUrl->SetIpAddress(address, EDc::SAS);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "Prestable");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }

    // UNKNOWN -> VLA
    {
        fetcherUrl->SetIpAddress(address, EDc::VLA);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "Prestable");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }

    // UNKNOWN -> UNKNOWN
    {
        fetcherUrl->SetIpAddress(address, EDc::UNKNOWN);
        auto dcLabel = fetcherUrl->HostLabels().Find(DC_LABEL_NAME);
        ASSERT_TRUE(dcLabel);
        ASSERT_EQ(dcLabel->Value(), "Prestable");
        ASSERT_EQ(fetcherUrl->Status().Dc, EDc::UNKNOWN);
    }
}

TEST_F(TFetcherUrlTest, EffectiveUrlChangesOnUrlChange) {
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
    bool ok;

    fetcherUrl->SetIpAddress(TIpv6Address::FromString("::2", ok), EDc::UNKNOWN);
    ASSERT_THAT(fetcherUrl->EffectiveUrl(), StrEq("http://[::2]:1337/something"));
    ASSERT_THAT(fetcherUrl->DisplayUrl(), StrEq("http://my.hostname:1337/something"));

    fetcherUrl->SetIpAddress(TIpv6Address::FromString("::3", ok), EDc::UNKNOWN);
    ASSERT_THAT(fetcherUrl->EffectiveUrl(), StrEq("http://[::3]:1337/something"));
}

TEST_F(TFetcherUrlTest, EffectiveUrlChangesOnShardChange) {
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
    ASSERT_THAT(fetcherUrl->DisplayUrl(), StrEq("http://my.hostname:1337/something"));

    TIntrusiveConstPtr<TMockShard> newShard = CloneShard(*MockShard_);
    newShard->MutService().ShardSettings.PullOrPush = TPullSettings{
        .Port = 7331,
        .Path = "/something_else"
    };
    fetcherUrl->SetFetcherShard(TFetcherShard{newShard});

    ASSERT_THAT(fetcherUrl->DisplayUrl(), StrEq("http://my.hostname:7331/something_else"));
}

TEST_F(TFetcherUrlTest, PreparedRequestHasTsArgs) {
    MockShard_->MutService().ShardSettings.PullOrPush = TPullSettings{.AddTsArgs = true};
    MockShard_->MutService().Interval = TDuration::Seconds(15);

    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
    fetcherUrl->SetIpAddress(DEFAULT_IP, EDc::VEG);
    auto req = fetcherUrl->PrepareRequest();
    ASSERT_TRUE(req.Success());
    ASSERT_THAT(req.Value().CgiParams, MatchesRegex(R"(\?now=.+&period=15s)"));
}

TEST_F(TFetcherUrlTest, PrepareRequestSimple) {
    MockShard_->MutService().GridSec = NDb::NModel::TServiceConfig::GRID_ABSENT;
    MockShard_->MutService().ShardSettings.PullOrPush = TPullSettings{
        .Port = 1337,
        .Path = "/something",
        .AddTsArgs = false,
    };

    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
    fetcherUrl->SetIpAddress(DEFAULT_IP, EDc::UNKNOWN);
    auto req = fetcherUrl->PrepareRequest();
    ASSERT_TRUE(req.Success());
    auto& headers = req.Value().Headers;
    ASSERT_FALSE(headers.contains(NAME_TVM_TICKET));
    ASSERT_FALSE(headers.contains(NAME_GRID_SECONDS));
    ASSERT_THAT(headers.at(NAME_SEQUENCE_NUMBER), StrEq(ToString(0)));
    ASSERT_THAT(headers.at(NAME_SHARD_LIMIT), StrEq(ToString(Shard_->MaxMetricsPerUrl())));
    ASSERT_THAT(headers.at(NAME_HOST), StrEq("my.hostname"));
    ASSERT_THAT(headers.at(NAME_FETCHER_ID), StrEq("cluster"));
    ASSERT_THAT(headers.at(NAME_CLUSTER_ID), StrEq("cluster"));
    ASSERT_THAT(req.Value().CgiParams, IsEmpty());
    ASSERT_THAT(req.Value().Url, StrEq("http://[::11]:1337/something"));
}

TEST_F(TFetcherUrlTest, PreparedRequestHasGridHeader) {
    MockShard_->MutService().GridSec = 60;

    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
    fetcherUrl->SetIpAddress(DEFAULT_IP, EDc::UNKNOWN);
    auto req = fetcherUrl->PrepareRequest();
    ASSERT_TRUE(req.Success());

    auto& headers = req.Value().Headers;
    ASSERT_TRUE(headers.contains(NAME_GRID_SECONDS));
    ASSERT_THAT(req.Value().Headers.at(NAME_GRID_SECONDS), StrEq("60"));
}

TEST_F(TFetcherUrlTest, PreparedRequestHasTvmHeader) {
    auto hl = HostAndLabels_;
    hl.TvmDestId = 1337;
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, hl);
    fetcherUrl->SetIpAddress(DEFAULT_IP, EDc::UNKNOWN);
    auto req = fetcherUrl->PrepareRequest();
    ASSERT_TRUE(req.Success());
    ASSERT_THAT(req.Value().Headers.at(NAME_TVM_TICKET), "ticket");
}

TEST_F(TFetcherUrlTest, PreparedRequestHasIamHeader) {
    MockShard_->MutService().ShardSettings.PullOrPush = TPullSettings{
            .Port = 1337,
            .Path = "/something",
            .Protocol = EPullProtocol::HTTPS // IAM token is put only with HTTPS
    };

    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
    fetcherUrl->SetIpAddress(DEFAULT_IP, EDc::UNKNOWN);
    auto req = fetcherUrl->PrepareRequest();
    ASSERT_TRUE(req.Success());
    auto& headers = req.Value().Headers;
    ASSERT_TRUE(headers.contains(NAME_AUTHORIZATION));
    ASSERT_EQ("Bearer " + IamToken, headers.at(NAME_AUTHORIZATION));
}

TEST_F(TFetcherUrlTest, PreparedRequestNoIamHeader) {
    MockShard_->MutService().ShardSettings.PullOrPush = TPullSettings{
            .Port = 1337,
            .Path = "/something",
            .Protocol = EPullProtocol::HTTP // IAM token is put only with HTTPS
    };

    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
    fetcherUrl->SetIpAddress(DEFAULT_IP, EDc::UNKNOWN);
    auto req = fetcherUrl->PrepareRequest();
    ASSERT_TRUE(req.Success());
    auto& headers = req.Value().Headers;
    ASSERT_FALSE(headers.contains(NAME_AUTHORIZATION));
}

TEST_F(TFetcherUrlTest, PreparedRequestHasTvmNoIamHeader) {
    MockShard_->MutService().ShardSettings.PullOrPush = TPullSettings{
            .Port = 1337,
            .Path = "/something",
            .Protocol = EPullProtocol::HTTPS // IAM token is put only with HTTPS
    };
    auto hl = HostAndLabels_;
    hl.TvmDestId = 1337;

    // IAM token will be omitted if TVM ticket is placed into headers
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, hl);
    fetcherUrl->SetIpAddress(DEFAULT_IP, EDc::UNKNOWN);
    auto req = fetcherUrl->PrepareRequest();
    ASSERT_TRUE(req.Success());
    auto& headers = req.Value().Headers;
    ASSERT_THAT(headers.at(NAME_TVM_TICKET), "ticket");
    ASSERT_FALSE(headers.contains(NAME_AUTHORIZATION));
}

TEST_F(TFetcherUrlTest, ReturnsErrorOnTvmFailure) {
    auto hl = HostAndLabels_;
    hl.TvmDestId = 1337;
    auto urlFactory = CreateFetcherUrlFactory(new TFailingTicketProvider, {}, TClusterInfo{"cluster"});
    auto fetcherUrl = urlFactory->CreateUrl(*Shard_, hl);
    fetcherUrl->SetIpAddress(DEFAULT_IP, EDc::VEG);
    auto req = fetcherUrl->PrepareRequest();
    ASSERT_FALSE(req.Success());
}

TEST_F(TFetcherUrlTest, StatusGetsUpdated) {
    {
        auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
        fetcherUrl->ReportStatus(IPC_QUEUE_OVERFLOW, "error!");
        fetcherUrl->SetResponseSize(123u);
        auto&& status = fetcherUrl->Status();
        ASSERT_THAT(status.UrlStatus, Eq(IPC_QUEUE_OVERFLOW));
        ASSERT_THAT(status.Host, Eq("my.hostname"));
        ASSERT_THAT(status.ResponseBytes, Eq(123u));
    }
    {
        auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
        fetcherUrl->BecomeDefunct(UNKNOWN_HOST);
        auto&& status = fetcherUrl->Status();
        ASSERT_THAT(status.UrlStatus, Eq(UNKNOWN_HOST));
        ASSERT_THAT(status.Host, Eq("my.hostname"));
    }
}

TEST_F(TFetcherUrlTest, StatusDcChangedOnIpChange) {
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
    ASSERT_THAT(fetcherUrl->Status().Dc, Eq(EDc::UNKNOWN));
    fetcherUrl->SetIpAddress(DEFAULT_IP, EDc::VEG);
    ASSERT_THAT(fetcherUrl->Status().Dc, Eq(EDc::VEG));
}

TEST_F(TFetcherUrlTest, UseFqdnChangesHostLabel) {
    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);
    ASSERT_THAT(fetcherUrl->HostLabel(), StrEq("my"));

    TIntrusiveConstPtr<TMockShard> newShard = CloneShard(*MockShard_);
    newShard->MutCluster().ShardSettings.PullOrPush = TPullSettings{.HostLabelPolicy = EHostLabelPolicy::FULL_HOSTNAME};
    fetcherUrl->SetFetcherShard(TFetcherShard{newShard});
    ASSERT_THAT(fetcherUrl->HostLabel(), StrEq("my.hostname"));

    fetcherUrl->SetFetcherShard(*Shard_);
    ASSERT_THAT(fetcherUrl->HostLabel(), StrEq("my"));
}

TEST_F(TFetcherUrlTest, IpAddressChangeToDifferentDc) {
    const TIpv6Address VLA_IP = DC_ADDRS[size_t(EDc::VLA)];
    const TIpv6Address SAS_IP = DC_ADDRS[size_t(EDc::SAS)];

    auto fetcherUrl = UrlFactory_->CreateUrl(*Shard_, HostAndLabels_);

    fetcherUrl->SetIpAddress(VLA_IP, EDc::VLA);
    ASSERT_THAT(fetcherUrl->Status().Dc, Eq(EDc::VLA));

    fetcherUrl->SetIpAddress(SAS_IP, EDc::SAS);
    ASSERT_THAT(fetcherUrl->Status().Dc, Eq(EDc::SAS));
}

class TMultiShardUrlTest: public TFetcherUrlTest {
public:
    void SetUp() override {
        TFetcherUrlTest::SetUp();

        NDb::NModel::TAgentConfig agent;
        agent.Provider = "provider";
        agent.Hostname = "my.host";
        agent.DataPort = 3443;

        Agents_ = {agent};

        HostAndLabels_.Host = agent.Hostname;
        HostAndLabels_.Port = agent.DataPort;

        AgentShard_ = MakeAgentShard();
        Cerr << "Fetch interval: " << AgentShard_->FetchInterval() << Endl;
    }

protected:
    THolder<TFetcherShard> MakeAgentShard() {
        return MakeHolder<TFetcherShard>(CreateAgentShard("projectId", TDuration::Seconds(15), Agents_));
    }

    TString MakeData(int sz = 5, bool overrideCluster = false) {
        TStringStream ss;
        auto encoder = NMultiShard::CreateMultiShardEncoder(NMonitoring::EFormat::SPACK, ss);

        encoder->SetHeader({"token"});
        for (auto i = 0; i < sz; ++i) {
            TString cluster;
            if (overrideCluster) {
                cluster = TStringBuilder() << "cluster" << i;
            }

            encoder->OnShardBegin(TStringBuilder() << "project" << i, TStringBuilder() << "service" << i, cluster);

            encoder->OnStreamBegin();
            encoder->OnMetricBegin(EMetricType::GAUGE);
            encoder->OnLabelsBegin();
            encoder->OnLabel("sensor", "foo");
            encoder->OnLabelsEnd();
            encoder->OnDouble(TInstant::Zero(), i);
            encoder->OnMetricEnd();
            encoder->OnStreamEnd();

            encoder->OnShardEnd();
        }

        encoder->Close();
        return ss.Str();
    }

    TVector<NDb::NModel::TAgentConfig> Agents_;
    THolder<TFetcherShard> AgentShard_;
};

TEST_F(TMultiShardUrlTest, Settings) {
    auto fetcherUrl = UrlFactory_->CreateUrl(*AgentShard_, HostAndLabels_);

    ASSERT_THAT(fetcherUrl->HostLabel(), StrEq("my.host"));
    ASSERT_THAT(fetcherUrl->Fqdn(), StrEq("my.host"));
    ASSERT_THAT(fetcherUrl->NextDownloadStart(TInstant::Seconds(1)), Gt(TInstant{} + AgentShard_->FetchInterval()));
    ASSERT_THAT(fetcherUrl->NextDownloadStart(TInstant::Seconds(1)), Lt(TInstant{} + 2 * AgentShard_->FetchInterval()));
}

TEST_F(TMultiShardUrlTest, PrepareRequestSimple) {
    auto fetcherUrl = UrlFactory_->CreateUrl(*AgentShard_, HostAndLabels_);
    auto result = fetcherUrl->PrepareRequest();
    fetcherUrl->SetIpAddress(DEFAULT_IP, EDc::VEG);
    auto req = fetcherUrl->PrepareRequest();
    ASSERT_TRUE(req.Success());
    auto& headers = req.Value().Headers;
    ASSERT_THAT(headers.at(NAME_SHARD_LIMIT), StrEq(ToString(AgentShard_->MaxMetricsPerUrl())));
    ASSERT_THAT(headers.at(NAME_HOST), StrEq("my.host"));
    ASSERT_THAT(headers.at(NAME_FETCHER_ID), StrEq("cluster"));
    ASSERT_THAT(headers.at(NAME_CLUSTER_ID), StrEq("cluster"));
    ASSERT_THAT(req.Value().CgiParams, IsEmpty());
    ASSERT_THAT(req.Value().Url, StrEq("http://[::11]:3443/storage/readAll"));

    TMultiShardRequest emptyMultiShardMessage;
    ASSERT_THAT(req.Value().Body, emptyMultiShardMessage.SerializeAsStringOrThrow());
}

class TUrlLocalityTest: public ::testing::Test {
public:
    void SetUp() override {
        MockShard_ = ShardFactory_.MakeShard();
        Shard_.Reset(new TFetcherShard{MockShard_});
    }

protected:
    TIntrusiveConstPtr<TMockShard> MockShard_;
    THolder<TFetcherShard> Shard_;
    TFetcherShardFactory ShardFactory_;
    THostAndLabels HostAndLabels_{"my.hostname"};
};

TEST_F(TUrlLocalityTest, AnyUrlIsLocalInGlobalMode) {
    auto urlFactory = CreateFetcherUrlFactory(new TFailingTicketProvider, {}, TClusterInfo{"cluster"});
    auto url = urlFactory->CreateUrl(*Shard_, HostAndLabels_);

    for (auto addr: DC_ADDRS) {
        url->SetIpAddress(addr, EDc::UNKNOWN);
        ASSERT_THAT(url->IsLocal(), true);
    }
}

TEST_F(TUrlLocalityTest, LocalMode) {
    auto urlFactory = CreateFetcherUrlFactory(
            new TFailingTicketProvider,
            {},
            TClusterInfo{"cluster", EOperationMode::Local, EDc::SAS});

    auto url = urlFactory->CreateUrl(*Shard_, HostAndLabels_);
    url->SetIpAddress(DC_ADDRS[size_t(EDc::SAS)], EDc::SAS);
    ASSERT_THAT(url->IsLocal(), true);
}
