#include <solomon/services/fetcher/lib/host_groups/host_and_labels.h>
#include <solomon/services/fetcher/lib/url/fetcher_url.h>
#include <solomon/services/fetcher/lib/url/fetcher_url_factory.h>
#include <solomon/services/fetcher/lib/url/headers.h>
#include <solomon/services/fetcher/lib/url/parser.h>
#include <solomon/services/fetcher/testlib/fetcher_shard.h>
#include <solomon/services/fetcher/testlib/http_client.h>
#include <solomon/services/fetcher/testlib/tvm.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>
#include <solomon/libs/cpp/multi_shard/multi_shard.h>
#include <solomon/libs/cpp/multi_shard/proto/multi_shard.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NFetcher;
using namespace NSolomon::NTesting;

using yandex::solomon::common::UrlStatusType;

TVector<NDb::NModel::TAgentConfig> CreateAgentsConfigs() {
    NDb::NModel::TAgentConfig agent;
    agent.Provider = "provider";
    agent.Hostname = "my.host";
    agent.DataPort = 3443;
    return {agent};
}

TString MakeData(int sz = 5, bool overrideCluster = false) {
    TStringStream ss;
    auto encoder = NMultiShard::CreateMultiShardEncoder(NMonitoring::EFormat::SPACK, ss);

    encoder->SetHeader({"token"});
    for (auto i = 0; i < sz; ++i) {
        TString cluster;
        if (overrideCluster) {
            cluster = TStringBuilder() << "cluster" << i;
        }

        encoder->OnShardBegin(TStringBuilder() << "project" << i, TStringBuilder() << "service" << i, cluster);

        encoder->OnStreamBegin();
        {
            encoder->OnMetricBegin(NMonitoring::EMetricType::GAUGE);
            {
                encoder->OnLabelsBegin();
                encoder->OnLabel("sensor", "foo");
                encoder->OnLabelsEnd();
            }
            encoder->OnDouble(TInstant::Zero(), i);
            encoder->OnMetricEnd();
        }
        encoder->OnStreamEnd();
        encoder->OnShardEnd();
    }

    encoder->Close();
    return ss.Str();
}

class TMultiShardUrlParserTest: public ::testing::Test {
public:
    TMultiShardUrlParserTest()
        : Shard_{CreateAgentShard("projectId", TDuration::Seconds(15), CreateAgentsConfigs())}
        , UrlFactory_{CreateFetcherUrlFactory(MakeIntrusive<TMockTicketProvider>(), {}, TClusterInfo{"cluster"})}
        , Runtime_{TTestActorRuntime::CreateInited()}
    {
    }

    void SetUp() override {
        ParserId_ = Runtime_->Register(UrlFactory_->CreateUrlParser(Shard_, "my-host", nullptr));
        EdgeId_ = Runtime_->AllocateEdgeActor();
    }

    void TearDown() override {
        Runtime_->Send(ParserId_, EdgeId_, std::make_unique<NActors::TEvents::TEvPoison>());
        Runtime_->GrabEdgeEvent<NActors::TEvents::TEvPoisonTaken>(EdgeId_);
    }

protected:
    TFetcherShard Shard_;
    std::shared_ptr<IFetcherUrlFactory> UrlFactory_;
    THolder<TTestActorRuntime> Runtime_;
    NActors::TActorId ParserId_;
    NActors::TActorId EdgeId_;
};

TEST_F(TMultiShardUrlParserTest, ParseBrokenResponse) {
    {
        auto resp = MakeHttpResponse(HTTP_OK, "somedata", {{"Content-Type", "application/json"}});
        ASSERT_TRUE(resp);

        Runtime_->Send(ParserId_, EdgeId_, std::make_unique<TUrlParserEvents::TParse>(std::move(resp)));
        auto parseResult = Runtime_->GrabEdgeEvent<TUrlParserEvents::TParseError>();

        ASSERT_TRUE(parseResult);
        EXPECT_EQ(parseResult->Status, UrlStatusType::UNKNOWN_ERROR);
        EXPECT_EQ(parseResult->Message, "Unsupported content-type: application/json");
    }

    {
        auto resp = MakeHttpResponse(
                HTTP_OK,
                "somedata",
                {{"content-type", ToString(NMultiShard::JSON_CONTENT_TYPE)}});

        Runtime_->Send(ParserId_, EdgeId_, std::make_unique<TUrlParserEvents::TParse>(std::move(resp)));
        auto parseResult = Runtime_->GrabEdgeEvent<TUrlParserEvents::TParseError>();

        ASSERT_TRUE(parseResult);
        EXPECT_EQ(parseResult->Status, UrlStatusType::JSON_ERROR);
        EXPECT_EQ(parseResult->Message, "Some data left in buffer on close");
    }
}

TEST_F(TMultiShardUrlParserTest, ParseResponseSimple) {
    const auto SHARD_COUNT = 5u;
    auto data = MakeData(SHARD_COUNT);

    auto resp = MakeHttpResponse(HTTP_OK, data, {{"content-type", ToString(NMultiShard::SPACK_CONTENT_TYPE)}});
    ASSERT_TRUE(resp);

    Runtime_->Send(ParserId_, EdgeId_, std::make_unique<TUrlParserEvents::TParse>(std::move(resp)));
    auto parseResult = Runtime_->GrabEdgeEvent<TUrlParserEvents::TParseResult>();

    ASSERT_TRUE(parseResult);
    EXPECT_EQ(parseResult->Format, NMonitoring::EFormat::SPACK);
    EXPECT_EQ(parseResult->Compression, NMonitoring::ECompression::IDENTITY);
    EXPECT_FALSE(parseResult->HasMore);

    auto& val = parseResult->Data;
    ASSERT_EQ(val.size(), SHARD_COUNT);

    for (auto i = 0u; i < SHARD_COUNT; ++i) {
        auto iStr = ToString(i);
        // empty cluster
        TShardKey key{"project" + iStr, "", "service" + iStr};

        ASSERT_EQ(val[i].Key, key);
        ASSERT_EQ(val[i].Format, NMonitoring::EFormat::SPACK);
        ASSERT_FALSE(val[i].Data.empty());
    }
}

TEST_F(TMultiShardUrlParserTest, ClusterOverride) {
    const auto SHARD_COUNT = 5u;
    auto data = MakeData(SHARD_COUNT, true);

    auto resp = MakeHttpResponse(HTTP_OK, data, {{"content-type", ToString(NMultiShard::SPACK_CONTENT_TYPE)}});
    ASSERT_TRUE(resp);

    Runtime_->Send(ParserId_, EdgeId_, std::make_unique<TUrlParserEvents::TParse>(std::move(resp)));
    auto parseResult = Runtime_->GrabEdgeEvent<TUrlParserEvents::TParseResult>();

    ASSERT_TRUE(parseResult);
    EXPECT_EQ(parseResult->Format, NMonitoring::EFormat::SPACK);
    EXPECT_EQ(parseResult->Compression, NMonitoring::ECompression::IDENTITY);
    EXPECT_FALSE(parseResult->HasMore);

    auto& val = parseResult->Data;
    ASSERT_EQ(val.size(), SHARD_COUNT);

    for (auto i = 0u; i < SHARD_COUNT; ++i) {
        auto iStr = ToString(i);
        // empty cluster
        TShardKey key{"project" + iStr, "cluster" + iStr, "service" + iStr};

        ASSERT_EQ(val[i].Key, key);
        ASSERT_EQ(val[i].Format, NMonitoring::EFormat::SPACK);
        ASSERT_FALSE(val[i].Data.empty());
    }
}
