#include <solomon/services/fetcher/lib/host_groups/host_and_labels.h>
#include <solomon/services/fetcher/lib/url/fetcher_url.h>
#include <solomon/services/fetcher/lib/url/fetcher_url_factory.h>
#include <solomon/services/fetcher/lib/url/headers.h>
#include <solomon/services/fetcher/lib/url/parser.h>
#include <solomon/services/fetcher/testlib/fetcher_shard.h>
#include <solomon/services/fetcher/testlib/http_client.h>
#include <solomon/services/fetcher/testlib/tvm.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NFetcher;
using namespace NSolomon::NTesting;

class TSingleShardUrlParserTest: public ::testing::Test {
public:
    TSingleShardUrlParserTest()
        : Shard_{TIntrusiveConstPtr<TMockShard>{TFetcherShardFactory{}.MakeShard()}}
        , UrlFactory_{CreateFetcherUrlFactory(MakeIntrusive<TMockTicketProvider>(), {}, TClusterInfo{"cluster"})}
        , Runtime_{TTestActorRuntime::CreateInited()}
    {
    }

    void SetUp() override {
        ParserId_ = Runtime_->Register(UrlFactory_->CreateUrlParser(Shard_, "my-host", nullptr));
        EdgeId_ = Runtime_->AllocateEdgeActor();
    }

    void TearDown() override {
        Runtime_->Send(ParserId_, EdgeId_, std::make_unique<NActors::TEvents::TEvPoison>());
        Runtime_->GrabEdgeEvent<NActors::TEvents::TEvPoisonTaken>(EdgeId_);
    }

protected:
    TFetcherShard Shard_;
    std::shared_ptr<IFetcherUrlFactory> UrlFactory_;
    THolder<TTestActorRuntime> Runtime_;
    NActors::TActorId ParserId_;
    NActors::TActorId EdgeId_;
};

TEST_F(TSingleShardUrlParserTest, ParseJson) {
    auto resp = MakeHttpResponse(HTTP_OK, "somedata", {{"Content-Type", "application/json"}});
    ASSERT_TRUE(resp);

    Runtime_->Send(ParserId_, EdgeId_, std::make_unique<TUrlParserEvents::TParse>(std::move(resp)));
    auto parseResult = Runtime_->GrabEdgeEvent<TUrlParserEvents::TParseResult>();

    ASSERT_TRUE(parseResult);
    EXPECT_EQ(parseResult->Format, NMonitoring::EFormat::JSON);
    EXPECT_EQ(parseResult->Compression, NMonitoring::ECompression::UNKNOWN);
    EXPECT_FALSE(parseResult->HasMore);

    auto& val = parseResult->Data;
    ASSERT_EQ(val.size(), 1u);
    auto& shardData = val[0];

    ASSERT_EQ(shardData.Data, "somedata");
    ASSERT_EQ(shardData.Format, NMonitoring::EFormat::JSON);
    ASSERT_EQ(shardData.Key, TShardKey{});
}

TEST_F(TSingleShardUrlParserTest, ParseSpack) {
    auto resp = MakeHttpResponse(HTTP_OK, "spack_data", {{"content-type", "application/x-solomon-spack"}});
    ASSERT_TRUE(resp);

    Runtime_->Send(ParserId_, EdgeId_, std::make_unique<TUrlParserEvents::TParse>(std::move(resp)));
    auto parseResult = Runtime_->GrabEdgeEvent<TUrlParserEvents::TParseResult>();

    ASSERT_TRUE(parseResult);
    EXPECT_EQ(parseResult->Format, NMonitoring::EFormat::SPACK);
    EXPECT_EQ(parseResult->Compression, NMonitoring::ECompression::UNKNOWN);
    EXPECT_FALSE(parseResult->HasMore);

    auto& val = parseResult->Data;
    ASSERT_EQ(val.size(), 1u);
    auto& shardData = val[0];

    ASSERT_EQ(shardData.Data, "spack_data");
    ASSERT_EQ(shardData.Format, NMonitoring::EFormat::SPACK);
    ASSERT_EQ(shardData.Key, TShardKey{});
}

TEST_F(TSingleShardUrlParserTest, HasMoreIndicatesOverflow) {
    auto resp = MakeHttpResponse(HTTP_OK, "somedata", {
            {"content-type", "application/x-solomon-spack"},
            {NAME_HAS_MORE, "1"}
    });

    Runtime_->Send(ParserId_, EdgeId_, std::make_unique<TUrlParserEvents::TParse>(std::move(resp)));
    auto parseResult = Runtime_->GrabEdgeEvent<TUrlParserEvents::TParseResult>();

    ASSERT_TRUE(parseResult);
    EXPECT_EQ(parseResult->Format, NMonitoring::EFormat::SPACK);
    EXPECT_EQ(parseResult->Compression, NMonitoring::ECompression::UNKNOWN);
    EXPECT_TRUE(parseResult->HasMore);

    auto& val = parseResult->Data;
    ASSERT_EQ(val.size(), 1u);
    auto& shardData = val[0];

    ASSERT_EQ(shardData.Data, "somedata");
    ASSERT_EQ(shardData.Format, NMonitoring::EFormat::SPACK);
    ASSERT_EQ(shardData.Key, TShardKey{});
}
