#include <solomon/services/fetcher/lib/fetcher_shard.h>
#include <solomon/services/fetcher/lib/url/fetcher_url_factory.h>
#include <solomon/services/fetcher/lib/url/parser.h>
#include <solomon/services/fetcher/testlib/http_client.h>
#include <solomon/services/fetcher/testlib/tvm.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/resource/resource.h>
#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NFetcher;
using namespace NSolomon::NTesting;

class TYasmUrlParserTest: public ::testing::Test {
public:
    TYasmUrlParserTest()
        : Shard_{CreateYasmAgentShard({"yasm_monitored"})}
        , UrlFactory_{CreateFetcherUrlFactory(MakeIntrusive<TMockTicketProvider>(), {}, {})}
        , Runtime_{TTestActorRuntime::CreateInited()}
    {
    }

    void SetUp() override {
        ParserId_ = Runtime_->Register(UrlFactory_->CreateUrlParser(Shard_, "my-host", nullptr));
        EdgeId_ = Runtime_->AllocateEdgeActor();
    }

    void TearDown() override {
        Runtime_->Send(ParserId_, EdgeId_, std::make_unique<NActors::TEvents::TEvPoison>());
        Runtime_->GrabEdgeEvent<NActors::TEvents::TEvPoisonTaken>(EdgeId_);
    }

protected:
    TFetcherShard Shard_;
    std::shared_ptr<IFetcherUrlFactory> UrlFactory_;
    THolder<TTestActorRuntime> Runtime_;
    NActors::TActorId ParserId_;
    NActors::TActorId EdgeId_;
};

TEST_F(TYasmUrlParserTest, ParseReturnsErrorOnInvalidProto) {
    auto resp = MakeHttpResponse(HTTP_OK, "broken-data", {{"content-type", "application/x-protobuf"}});
    ASSERT_TRUE(resp);

    Runtime_->Send(ParserId_, EdgeId_, std::make_unique<TUrlParserEvents::TParse>(std::move(resp)));
    auto parseResult = Runtime_->GrabEdgeEvent<TUrlParserEvents::TParseError>();

    ASSERT_TRUE(parseResult);
    EXPECT_EQ(parseResult->Status, yandex::solomon::common::UrlStatusType::PARSE_ERROR);
    EXPECT_EQ(parseResult->Message, "cannot parse TAgentResponse from plain response");
}

TEST_F(TYasmUrlParserTest, ParseProducesMultipleShardData) {
    // TODO: use proto with aggregated records only
    const TString data = NResource::Find("yasm_agent_response.proto");
    auto resp = MakeHttpResponse(HTTP_OK, data, {});

    Runtime_->Send(ParserId_, EdgeId_, std::make_unique<TUrlParserEvents::TParse>(std::move(resp)));
    auto parseResult = Runtime_->GrabEdgeEvent<TUrlParserEvents::TParseResult>(TDuration::Seconds(1));
    if (!parseResult) {
        auto parseError = Runtime_->GrabEdgeEvent<TUrlParserEvents::TParseError>(TDuration::Seconds(1));
        ASSERT_TRUE(parseError);
        FAIL() << "parse error: " << parseError->Message;
    }

    ASSERT_TRUE(parseResult);
    EXPECT_EQ(parseResult->Format, NMonitoring::EFormat::SPACK);
    EXPECT_EQ(parseResult->Compression, NMonitoring::ECompression::IDENTITY);
    EXPECT_FALSE(parseResult->HasMore);
}
