#include "mock_matcher.h"

#include <solomon/services/fetcher/lib/dns/continuous_resolver.h>
#include <solomon/services/fetcher/lib/download/download.h>
#include <solomon/services/fetcher/lib/url/url.h>
#include <solomon/services/fetcher/testlib/actor_system.h>
#include <solomon/services/fetcher/testlib/fetcher_shard.h>
#include <solomon/services/fetcher/testlib/http_client.h>
#include <solomon/services/fetcher/testlib/tvm.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace testing;
using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace NSolomon::NAuth::NTvm;

namespace {
    ITicketProviderPtr MOCK_TICKET_PROVIDER = MakeIntrusive<TMockTicketProvider>();
    const TIpv6Address IP_VLA{DC_ADDRS[size_t(EDc::VLA)]};
    const TIpv6Address IP_SAS{DC_ADDRS[size_t(EDc::SAS)]};
}

class TUrlActorTestBase: public ::testing::Test {
public:
    virtual std::shared_ptr<IFetcherUrlFactory> CreateUrlFactory() = 0;
    virtual THostAndLabels HostAndLabels() = 0;

    void SetUp() override {
        TFetcherShardFactory f;
        TIntrusiveConstPtr<TMockShard> mock = f.MakeShard();
        ShardConf_.Reset(new TFetcherShard{mock});
        ActorSystem_ = MakeActorSystem();
        ActorSystem_->Initialize();
        ActorSystem_->UpdateCurrentTime(TInstant::Zero());
        ShardMetricFactory_ = CreateShardMetricFactory(Registry_, EMetricVerbosity::All);
        UrlFactory_ = CreateUrlFactory();

        auto metrics = ShardMetricFactory_->MetricsFor("foo", "bar");
        TActorId parserId = ActorSystem_->Register(UrlFactory_->CreateUrlParser(*ShardConf_, "my.host", metrics));

        UrlActor_ = ActorSystem_->Register(CreateUrlActor(
            HostAndLabels(), *ShardConf_, *UrlFactory_,
            parserId,
            {}, // dnsResolverId
            {}, // dataSinkId
            {}, // statCollectorId
            {}, // authGatekeeper
            metrics, CreateFakeLimiter(), HttpClient_
        ));

        ActorSystem_->SetScheduledEventFilter([] (auto&&, auto&&, auto&&, auto) {
            return false;
        });

        ActorSystem_->WaitForBootstrap();
    }

    void TearDown() override {
        ActorSystem_.Reset();
        ShardMetricFactory_.Reset();
        ShardConf_.Reset();
    }

protected:
    TMetricRegistry Registry_;
    THolder<IShardMetricFactory> ShardMetricFactory_;
    TIntrusivePtr<TMockHttpClient> HttpClient_{new TMockHttpClient};
    std::shared_ptr<IFetcherUrlFactory> UrlFactory_;
    TActorId UrlActor_;
    THolder<TFetcherShard> ShardConf_;
    THolder<TFetcherActorRuntime> ActorSystem_;
};

class TUrlActorTest: public TUrlActorTestBase {
public:
    THostAndLabels HostAndLabels() override {
        bool ok;
        return {TIpv6Address::FromString("::3", ok)};
    }

    std::shared_ptr<IFetcherUrlFactory> CreateUrlFactory() override {
        return CreateFetcherUrlFactory(MOCK_TICKET_PROVIDER, {}, TClusterInfo{"cluster"});
    }
};

TEST_F(TUrlActorTest, ScheduleSimple) {
    auto ev = MakeHolder<TEvDownloadCompleted>();
    Y_UNUSED(ActorSystem_->CaptureScheduledEvents());

    ev->Result = IHttpClient::TResult::FromError(TRequestError::EType::ConnectFailed, "err");
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, ev.Release()});
    auto scheduled = ActorSystem_->CaptureScheduledEvents();
    ASSERT_THAT(scheduled, SizeIs(1u));
    ASSERT_THAT(scheduled.begin()->Deadline, Gt(TInstant::Zero() + ShardConf_->FetchInterval()));
    ASSERT_THAT(scheduled.begin()->Deadline, Lt(TInstant::Zero() + ShardConf_->FetchInterval() * 2));
    auto* wakeupEvent = scheduled.begin()->Event->Get<TEvents::TEvWakeup>();
    ASSERT_THAT(wakeupEvent->Tag, Ne(DEFUNCT_WAKEUP_TAG));
}

TEST_F(TUrlActorTest, ScheduleAfterLongDownload) {
    Y_UNUSED(ActorSystem_->CaptureScheduledEvents());

    auto startInstant = TInstant::MilliSeconds(
            RandomNumber(ShardConf_->FetchInterval().MilliSeconds())
    );

    ActorSystem_->UpdateCurrentTime(startInstant);

    auto ev = MakeHolder<TEvDownloadCompleted>();
    ev->Result = IHttpClient::TResult::FromError(TRequestError::EType::ConnectFailed, "err");

    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, ev.Release()});
    auto scheduled = ActorSystem_->CaptureScheduledEvents();

    ASSERT_THAT(scheduled, SizeIs(1u));
    ASSERT_THAT(scheduled.begin()->Deadline, Gt(TInstant::Zero() + ShardConf_->FetchInterval()));
    ASSERT_THAT(scheduled.begin()->Deadline, Lt(TInstant::Zero() + ShardConf_->FetchInterval() * 2));
}

TEST_F(TUrlActorTest, ScheduleAfterSkippedInterval) {
    auto ev = MakeHolder<TEvDownloadCompleted>();
    Y_UNUSED(ActorSystem_->CaptureScheduledEvents());

    ActorSystem_->UpdateCurrentTime(TInstant::Zero() + ShardConf_->FetchInterval());
    ev->Result = IHttpClient::TResult::FromError(TRequestError::EType::ConnectFailed, "err");
    ActorSystem_->UpdateCurrentTime(TInstant::Zero() + ShardConf_->FetchInterval() + TDuration::MilliSeconds(50));
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, ev.Release()});
    auto scheduled = ActorSystem_->CaptureScheduledEvents();
    ASSERT_THAT(scheduled, SizeIs(1u));
    ASSERT_THAT(scheduled.begin()->Deadline, Gt(TInstant::Zero() + ShardConf_->FetchInterval() * 2));
    ASSERT_THAT(scheduled.begin()->Deadline, Lt(TInstant::Zero() + ShardConf_->FetchInterval() * 3));
}

TEST_F(TUrlActorTest, ScheduleWhileDefunct) {
    Y_UNUSED(ActorSystem_->CaptureScheduledEvents());
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, new TEvHostResolveFail{EErrorType::NotFound, "", ""}});

    auto scheduled = ActorSystem_->CaptureScheduledEvents();
    ASSERT_THAT(scheduled, SizeIs(1));
    auto* wakeupEvent = scheduled.begin()->Event->Get<TEvents::TEvWakeup>();
    ASSERT_THAT(wakeupEvent->Tag, Eq(DEFUNCT_WAKEUP_TAG));
}

TEST_F(TUrlActorTest, DnsTimeoutDoesNotMakeActorDefunct) {
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, new TEvHostResolveFail{EErrorType::Timeout, "", ""}});

    auto ev = MakeHolder<TEvDownloadCompleted>();
    Y_UNUSED(ActorSystem_->CaptureScheduledEvents());
    ev->Result = IHttpClient::TResult::FromError(TRequestError::EType::ConnectFailed, "err");
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, ev.Release()});
    auto scheduled = ActorSystem_->CaptureScheduledEvents();
    ASSERT_THAT(scheduled, Not(IsEmpty()));
}

TEST_F(TUrlActorTest, RestoreActorFromDefunctStateAfterDnsFailuresAreGone) {
    Y_UNUSED(ActorSystem_->CaptureScheduledEvents());
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, new TEvHostResolveFail{EErrorType::NotFound, "", ""}});

    // URL actor become defunct
    {
        auto scheduled = ActorSystem_->CaptureScheduledEvents();
        ASSERT_THAT(scheduled, SizeIs(1));
        auto* wakeupEvent = scheduled.begin()->Event->Get<TEvents::TEvWakeup>();
        ASSERT_THAT(wakeupEvent->Tag, Eq(DEFUNCT_WAKEUP_TAG));
    }

    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, new TEvHostResolveOk{"", IP_VLA, EDc::VLA}});

    // URL actor become working
    {
        auto ev = MakeHolder<TEvDownloadCompleted>();
        ev->Result = IHttpClient::TResult::FromError(TRequestError::EType::ConnectFailed, "err");
        ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, ev.Release()});

        auto scheduled = ActorSystem_->CaptureScheduledEvents();
        ASSERT_THAT(scheduled, Not(IsEmpty()));
    }
}

class TLocalUrlActorTest: public TUrlActorTestBase {
public:
    THostAndLabels HostAndLabels() override {
        return THostAndLabels{"my.host"};
    }

    std::shared_ptr<IFetcherUrlFactory> CreateUrlFactory() override {
        return CreateFetcherUrlFactory(MOCK_TICKET_PROVIDER, {}, TClusterInfo{"cluster", EOperationMode::Local, EDc::SAS});
    }
};

TEST_F(TLocalUrlActorTest, StartsIdleOnNonLocalUrl) {
    Y_UNUSED(ActorSystem_->CaptureScheduledEvents());
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, new TEvHostResolveOk{"", IP_VLA, EDc::VLA}});

    auto scheduled = ActorSystem_->CaptureScheduledEvents();
    ASSERT_THAT(scheduled, IsEmpty());
}

TEST_F(TLocalUrlActorTest, BecomesIdleOnNonLocalUrl) {
    Y_UNUSED(ActorSystem_->CaptureScheduledEvents());
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, new TEvHostResolveOk{"", IP_SAS, EDc::SAS}});

    auto scheduled = ActorSystem_->CaptureScheduledEvents();
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, new TEvHostResolveOk{"", IP_VLA, EDc::VLA}});

    auto ev = MakeHolder<TEvDownloadCompleted>();
    // this would trigger scheduling of the next download in working state
    ev->Result = IHttpClient::TResult::FromError(TRequestError::EType::ConnectFailed, "err");
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, ev.Release()});
    scheduled = ActorSystem_->CaptureScheduledEvents();

    scheduled = ActorSystem_->CaptureScheduledEvents();
    ASSERT_THAT(scheduled, IsEmpty());
}

TEST_F(TLocalUrlActorTest, LeavesIdleState) {
    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, new TEvHostResolveOk{"", IP_VLA, EDc::VLA}});

    auto scheduled = ActorSystem_->CaptureScheduledEvents();
    ASSERT_THAT(scheduled, IsEmpty());

    ActorSystem_->Send(new IEventHandle{UrlActor_, TActorId{}, new TEvHostResolveOk{"", IP_SAS, EDc::VLA}});
    scheduled = ActorSystem_->CaptureScheduledEvents();
}
