#include "encoder.h"
#include "spack_encoder.h"

#include <infra/yasm/interfaces/internal/agent.pb.h>

#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

using namespace NMonitoring;
using namespace NYasm::NInterfaces::NInternal;

namespace NSolomon::NFetcher::NYasm {
namespace {

class TMultiShardEncoder: public IMultiShardEncoder {
public:
    explicit TMultiShardEncoder(ECompression compression)
        : Compression_{compression}
    {
    }

    void AddShard(TYasmShardKey key) override {
        auto& encoder = Impls_[std::move(key)];
        if (!encoder) {
            encoder = std::make_unique<TStreamSpackEncoder>();
        }
        CurrentShards_.push_back(encoder.get());
    }

    void SwitchShards() override {
        CurrentShards_.clear();
    }

    void Close(IDataConsumer* consumer) override {
        for (auto it = Impls_.begin(); it != Impls_.end();) {
            if (it->second) {
                auto data = it->second->Finish(Compression_);
                consumer->OnShardData(it->first, std::move(data));
                it->second->Reset();
                ++it;
            } else {
                Impls_.erase(it++);
            }
        }
    }

    void WriteLabel(TStringBuf name, TStringBuf value) override {
        for (auto* ptr: CurrentShards_) {
            ptr->AddLabel(name, value);
        }
    }

    bool SupportedValue(const TValue& value) override {
        switch (value.GetValueCase()) {
            case TValue::kNoneValue:
            case TValue::kVec:
            case TValue::kHyperLogLog:
            case TValue::VALUE_NOT_SET:
                return false;
            default:
                return true;
        }
    }

    void WriteValue(const ::NYasm::NInterfaces::NInternal::TValue& value) override {
        switch (value.GetValueCase()) {
            case TValue::kFloatValue:
                WriteValue(value.GetFloatValue());
                break;
            case TValue::kCountedSum:
                WriteValue(value.GetCountedSum());
                break;
            case TValue::kHgramSmall:
                WriteValue(value.GetHgramSmall());
                break;
            case TValue::kHgramNormal:
                WriteValue(value.GetHgramNormal());
                break;
            case TValue::kUgram:
                WriteValue(value.GetUgram());
                break;
            default:
                Y_FAIL("type %d is not supported", value.GetValueCase());
        }
    }

    template <typename T>
    void WriteValue(const T& value) {
        for (auto* ptr: CurrentShards_) {
            ptr->WriteValue(value);
        }
    }

private:
    const ECompression Compression_;
    TVector<TStreamSpackEncoder*> CurrentShards_;
    absl::flat_hash_map<TYasmShardKey, std::unique_ptr<TStreamSpackEncoder>, THash<TYasmShardKey>> Impls_;
};

} // namespace

THolder<IMultiShardEncoder> CreateMultiShardEncoder(ECompression compression) {
    return MakeHolder<TMultiShardEncoder>(compression);
}

} // namespace NSolomon::NFetcher::NYasm
