#pragma once

#include <util/generic/strbuf.h>
#include <util/generic/yexception.h>

namespace NSolomon::NFetcher::NYasm {

/**
 * Instance key format:
 * <pre>'instance_type_name|key1=value1;key2=value2;...;keyN=valueN|aggr1,aggr2,...,aggrM'</pre>
 */
class TInstanceKey {
    static constexpr char FIELD_DELIM = '|';
    static constexpr char KV_DELIM = '=';
    static constexpr char TAG_DELIM = ';';
    static constexpr char AGGR_DELIM = ',';

public:
    explicit TInstanceKey(TStringBuf value)
        : Value_{value.data()}
        , Size_{static_cast<ui32>(value.size())}
    {
        auto* end = Value_ + Size_;
        auto* it = std::find(Value_, end, FIELD_DELIM);
        Y_ENSURE(it != end, "invalid instance key: '" << value << '\'');
        TagsOffset_ = static_cast<ui32>(it - Value_);

        it = std::find(it + 1, end, FIELD_DELIM);
        AggrsOffset_ = static_cast<ui32>(it - Value_);
    }

    static bool IsCommonItype(TStringBuf value) {
        TStringBuf commonItype{"common"};
        if (!value.StartsWith(commonItype)) {
            return false;
        }

        if (value.size() == commonItype.size()) {
            return true;
        }

        return value[commonItype.size()] == FIELD_DELIM;
    }

    constexpr TStringBuf GetItype() const noexcept {
        return TStringBuf{Value_, Value_ + TagsOffset_};
    }

    constexpr TStringBuf GetTags() const noexcept {
        return TStringBuf{Value_ + TagsOffset_ + 1, Value_ + AggrsOffset_};
    }

    constexpr TStringBuf GetAggrs() const noexcept {
        return TStringBuf{Value_ + std::min(AggrsOffset_ + 1, Size_), Value_ + Size_};
    }

    template <typename TConsumer>
    void ForEachTag(TConsumer&& consumer) const {
        auto tags = GetTags();
        if (tags.empty()) {
            return;
        }

        auto* it = tags.cbegin();
        auto* end = tags.cend();

        while (true) {
            auto* keyEnd = std::find(it, end, KV_DELIM);
            Y_ENSURE(keyEnd != end, "invalid instance key: '" << *this << '\'');

            TStringBuf key{it, keyEnd};
            it = keyEnd + 1; // skip KV_DELIM

            auto* valueEnd = std::find(it, end, TAG_DELIM);
            TStringBuf value{it, valueEnd};

            consumer(key, value);

            if (valueEnd == end) {
                break;
            }
            it = valueEnd + 1; // skip TAG_DELIM
        }
    }

    TStringBuf GetHostName() const {
        auto tags = GetTags();
        if (tags.empty()) {
            return {};
        }

        auto* it = tags.cbegin();
        auto* end = tags.cend();

        TStringBuf groupValue; // group has less priority than host

        while (true) {
            auto* keyEnd = std::find(it, end, KV_DELIM);
            Y_ENSURE(keyEnd != end, "invalid instance key: '" << *this << '\'');

            TStringBuf key{it, keyEnd};
            if (key == TStringBuf{"host"}) {
                auto* valueEnd = std::find(keyEnd + 1, end, TAG_DELIM);
                return TStringBuf{keyEnd + 1, valueEnd};
            } else if (key == TStringBuf{"group"}) {
                auto* valueEnd = std::find(keyEnd + 1, end, TAG_DELIM);
                groupValue = TStringBuf{keyEnd + 1, valueEnd};
                it = valueEnd;
            } else {
                it = std::find(keyEnd + 1, end, TAG_DELIM); // skip until the next tag
            }

            if (it == end) {
                break;
            }
            ++it; // skip TAG_DELIM
        }

        return groupValue;
    }

    template <typename TConsumer>
    void ForEachAggr(TConsumer&& consumer) const {
        auto aggrs = GetAggrs();
        if (aggrs.empty()) {
            return;
        }

        auto* it = aggrs.cbegin();
        auto* end = aggrs.cend();

        while (true) {
            auto* aggrEnd = std::find(it, end, AGGR_DELIM);
            consumer(TStringBuf{it, aggrEnd});
            if (aggrEnd == end) {
                break;
            }
            it = aggrEnd + 1; // skip AGGR_DELIM
        }
    }

    friend IOutputStream& operator<<(IOutputStream& out, TInstanceKey key) {
        return out << TStringBuf(key.Value_, key.Size_);
    }

    friend std::ostream& operator<<(std::ostream& out, TInstanceKey key) {
        return out << TStringBuf(key.Value_, key.Size_);
    }

private:
    const char* Value_;
    const ui32 Size_;
    ui32 TagsOffset_;
    ui32 AggrsOffset_;
};

} // namespace NSolomon::NFetcher::NYasm
