#include "spack_encoder.h"
#include "hist_builder.h"

#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>
#include <infra/yasm/interfaces/internal/agent.pb.h>

#include <library/cpp/monlib/encode/spack/compression.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/encode/spack/varint.h>
#include <library/cpp/monlib/metrics/log_histogram_collector.h>

#include <util/stream/str.h>

using namespace NMonitoring;
using namespace NYasm::NInterfaces::NInternal;

namespace NSolomon::NFetcher::NYasm {
namespace {

static constexpr double LOG_HGRAM_BASE = 1.5;
static constexpr double LAST_INF_UGRAM_BOARD = std::numeric_limits<double>::max();

const IHistogramSnapshotPtr EMPTY_HIST = [] {
    return THistogramBuilder{0}.Finalize();
}();

void WriteLabels(IOutputStream* out, const std::vector<std::pair<ui32, ui32>>& labels) {
    WriteVarUInt32(out, static_cast<ui32>(labels.size()));
    for (auto&& label: labels) {
        WriteVarUInt32(out, label.first);
        WriteVarUInt32(out, label.second);
    }
}

template <typename T>
void WriteFixed(IOutputStream* out, T value) {
    out->Write(&value, sizeof(value));
}

void WriteHistogram(IOutputStream* out, const IHistogramSnapshot& histogram) {
    ui32 count = histogram.Count();
    WriteVarUInt32(out, count);

    for (ui32 i = 0; i < count; i++) {
        double bound = histogram.UpperBound(i);
        out->Write(&bound, sizeof(bound));
    }
    for (ui32 i = 0; i < count; i++) {
        ui64 value = histogram.Value(i);
        out->Write(&value, sizeof(value));
    }
}

NZoom::NHgram::TUgramBuckets FromProtoBuckets(const TUgram& ugram) {
    NZoom::NHgram::TUgramBuckets buckets;
    buckets.reserve(ugram.BucketsSize());

    for (auto&& bucket: ugram.GetBuckets()) {
        buckets.emplace_back(bucket.GetLowerBound(), bucket.GetUpperBound(), bucket.GetWeight());
    }

    return buckets;
}

NZoom::NHgram::TUgramBuckets PrepareUgramBounds(const TUgram& ugram) {
    auto original = FromProtoBuckets(ugram);
    auto compressed = NZoom::NHgram::TUgramCompressor::GetInstance().Compress(
            original,
            HISTOGRAM_MAX_BUCKETS_COUNT - 2
            );

    if (compressed.empty() && !original.empty()) {
        if (AllOf(original, [](const auto& bucket) {return bucket.Weight == 0.0;})) {
            compressed = NZoom::NHgram::TUgramBuckets{NZoom::NHgram::TUgramBucket{
                original.front().LowerBound,
                original.back().UpperBound,
                0.0
            }};
        }
    }

    return compressed;
}

IHistogramSnapshotPtr ConvertUgram(const TUgram& ugram) {
    auto buckets = PrepareUgramBounds(ugram);
    if (buckets.empty()) {
        return EMPTY_HIST;
    }

    THistogramBuilder histBuilder{buckets.size()};
    constexpr auto MIN_BOUND = Min<TBucketBound>();
    auto prevUpper = MIN_BOUND;

    for (auto&& bucket: buckets) {
        const auto lower = bucket.LowerBound;
        auto upper = bucket.UpperBound;
        if (upper == lower) {
            upper = std::nextafter(upper, LAST_INF_UGRAM_BOARD);
        }
        if (prevUpper == MIN_BOUND || prevUpper != lower) {
            histBuilder.Add(lower, 0);
            if (histBuilder.Size() >= HISTOGRAM_MAX_BUCKETS_COUNT - 1) {
                break;
            }
        }

        histBuilder.Add(upper, round(bucket.Weight));
        prevUpper = bucket.UpperBound;

        if (histBuilder.Size() >= HISTOGRAM_MAX_BUCKETS_COUNT - 1) {
            break;
        }
    }

    return histBuilder.Finalize();
}

constexpr char PackType(EMetricType type) {
    return (char) ((static_cast<ui8>(type) << 2) | static_cast<ui8>(EValueType::ONE_WITHOUT_TS));
}

} // namespace

void TStreamSpackEncoder::AddLabel(TStringBuf name, TStringBuf value) {
    Labels_.emplace_back(LabelNamesPool_.Put(name), LabelValuesPool_.Put(value));
}

void TStreamSpackEncoder::BeginMetric(EMetricType type) {
    MetricsOut_.Write(PackType(type));
    MetricsOut_.Write((char) 0x00); // flag byte is not supported
    WriteLabels(&MetricsOut_, Labels_);
}

void TStreamSpackEncoder::EndMetric() {
    Labels_.clear();
    MetricCount_++;
}

void TStreamSpackEncoder::WriteValue(const TFloatValue& value) {
    BeginMetric(EMetricType::GAUGE);
    WriteFixed(&MetricsOut_, value.GetValue());
    EndMetric();
}

void TStreamSpackEncoder::WriteValue(const TCountedSum& value) {
    BeginMetric(EMetricType::DSUMMARY);
    WriteFixed(&MetricsOut_, value.GetCount());
    WriteFixed(&MetricsOut_, value.GetSum());
    WriteFixed(&MetricsOut_, 0.0); // min
    WriteFixed(&MetricsOut_, 0.0); // max
    WriteFixed(&MetricsOut_, 0.0); // last
    EndMetric();
}

void TStreamSpackEncoder::WriteValue(const THgramSmall& value) {
    // TODO: better way to convert small histogram into log histogram
    TLogHistogramCollector collector;
    collector.AddZeros(value.GetZeros());
    for (auto v: value.GetValues()) {
        collector.Collect(v);
    }
    auto logHist = collector.Snapshot();

    BeginMetric(EMetricType::LOGHIST);

    WriteFixed(&MetricsOut_, logHist->Base());
    WriteFixed(&MetricsOut_, logHist->ZerosCount());
    WriteVarUInt32(&MetricsOut_, static_cast<ui32>(logHist->StartPower()));
    WriteVarUInt32(&MetricsOut_, logHist->Count());
    for (ui32 i = 0; i < logHist->Count(); ++i) {
        WriteFixed(&MetricsOut_, logHist->Bucket(i));
    }

    EndMetric();
}

void TStreamSpackEncoder::WriteValue(const THgramNormal& value) {
    BeginMetric(EMetricType::LOGHIST);

    WriteFixed(&MetricsOut_, LOG_HGRAM_BASE);
    WriteFixed(&MetricsOut_, value.GetZeros());
    WriteVarUInt32(&MetricsOut_, static_cast<ui32>(value.GetStartPower()));
    WriteVarUInt32(&MetricsOut_, static_cast<ui32>(value.ValuesSize()));

    for (size_t i = 0; i < value.ValuesSize(); ++i) {
        WriteFixed(&MetricsOut_, value.GetValues(i));
    }

    EndMetric();
}

void TStreamSpackEncoder::WriteValue(const TUgram& value) {
    auto hist = ConvertUgram(value);
    // only inf bucket, hist is empty
    if (hist->Count() == 1 && hist->Value(0u) == 0u) {
        Labels_.clear();
        return;
    }

    BeginMetric(EMetricType::HIST);
    WriteHistogram(&MetricsOut_, *hist);
    EndMetric();
}

TString TStreamSpackEncoder::Finish(ECompression compression) {
    TStringOutput dataOut{Data_};

    // (1) write header
    TSpackHeader header;
    header.Version = ESpackV1Version::SV1_01;
    header.TimePrecision = EncodeTimePrecision(ETimePrecision::SECONDS);
    header.Compression = EncodeCompression(compression);
    header.LabelNamesSize = static_cast<ui32>(LabelNamesPool_.SizeBytes());
    header.LabelValuesSize = static_cast<ui32>(LabelValuesPool_.SizeBytes());
    header.MetricCount = MetricCount_;
    header.PointsCount = MetricCount_; // each metric has only one point
    dataOut.Write(&header, sizeof(header));

    // if compression enabled all below writes must go through compressor
    IOutputStream* out = &dataOut;
    auto compressedOut = CompressedOutput(out, compression);
    if (compressedOut) {
        out = compressedOut.Get();
    }

    // (2) write string pools
    out->Write(LabelNamesPool_.UnderlyingBuf());
    out->Write(LabelValuesPool_.UnderlyingBuf());

    // (3) write common time
    WriteFixed<ui32>(out, 0); // common time is not supported

    // (4) write common labels' indexes
    WriteLabels(out, {}); // common labels are not supported

    // (5) write metrics
    MetricsOut_.Finish();
    out->Write(MetricsBuf_.data(), MetricsBuf_.size());

    dataOut.Finish();
    return std::move(Data_);
}

void TStreamSpackEncoder::Reset() {
    Data_.clear();
    MetricsBuf_.Clear();
    Labels_.clear();
    MetricCount_ = 0;
}

} // namespace NSolomon::NFetcher::NYasm
