#include <solomon/services/fetcher/lib/yasm/instance_key.h>

#include <infra/yasm/common/labels/tags/instance_key.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NFetcher::NYasm;

std::vector<std::pair<TStringBuf, TStringBuf>> ToTags(TInstanceKey key) {
    std::vector<std::pair<TStringBuf, TStringBuf>> tags;
    key.ForEachTag([&](auto key, auto value) {
        tags.emplace_back(key, value);
    });
    return tags;
}

std::vector<TStringBuf> ToAggrs(TInstanceKey key) {
    std::vector<TStringBuf> aggrs;
    key.ForEachAggr([&](auto aggr) {
       aggrs.push_back(aggr);
    });
    return aggrs;
}

TEST(TInstanceKeyTest, Invalid) {
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(
        TInstanceKey{""};
    , yexception, "invalid instance key: ''");

    ASSERT_THROW_MESSAGE_HAS_SUBSTR({
        TInstanceKey key{"itype-name|key1"};
        key.ForEachTag([](auto, auto) {});
    }, yexception, "invalid instance key: 'itype-name|key1'");

    ASSERT_THROW_MESSAGE_HAS_SUBSTR({
        TInstanceKey key{"itype-name|key1=value1;key2"};
        key.ForEachTag([](auto, auto) {});
    }, yexception, "invalid instance key: 'itype-name|key1=value1;key2'");
}

TEST(TInstanceKeyTest, SingleTag) {
    TInstanceKey key{"jugglerclient|ctype=production"};
    ASSERT_EQ(key.GetItype(), "jugglerclient");
    ASSERT_EQ(key.GetTags(), "ctype=production");
    ASSERT_EQ(key.GetAggrs(), "");

    auto tags = ToTags(key);
    ASSERT_EQ(tags.size(), 1u);
    ASSERT_EQ(tags[0], std::make_pair(TStringBuf("ctype"), TStringBuf("production")));

    auto aggrs = ToAggrs(key);
    ASSERT_TRUE(aggrs.empty());
}

TEST(TInstanceKeyTest, ManyTags) {
    TInstanceKey key{"jugglerclient|ctype=production;geo=msk;prj=juggler;tier=none"};
    ASSERT_EQ(key.GetItype(), "jugglerclient");
    ASSERT_EQ(key.GetTags(), "ctype=production;geo=msk;prj=juggler;tier=none");
    ASSERT_EQ(key.GetAggrs(), "");

    auto tags = ToTags(key);
    ASSERT_EQ(tags.size(), 4u);
    ASSERT_EQ(tags[0], std::make_pair(TStringBuf("ctype"), TStringBuf("production")));
    ASSERT_EQ(tags[1], std::make_pair(TStringBuf("geo"), TStringBuf("msk")));
    ASSERT_EQ(tags[2], std::make_pair(TStringBuf("prj"), TStringBuf("juggler")));
    ASSERT_EQ(tags[3], std::make_pair(TStringBuf("tier"), TStringBuf("none")));

    auto aggrs = ToAggrs(key);
    ASSERT_TRUE(aggrs.empty());
}

TEST(TInstanceKeyTest, TagsAndEmptyAggregates) {
    TInstanceKey key{"addrsupper|ctype=prestable;geo=sas;prj=addrs|"};
    ASSERT_EQ(key.GetItype(), "addrsupper");
    ASSERT_EQ(key.GetTags(), "ctype=prestable;geo=sas;prj=addrs");
    ASSERT_EQ(key.GetAggrs(), "");

    auto tags = ToTags(key);
    ASSERT_EQ(tags.size(), 3u);
    ASSERT_EQ(tags[0], std::make_pair(TStringBuf("ctype"), TStringBuf("prestable")));
    ASSERT_EQ(tags[1], std::make_pair(TStringBuf("geo"), TStringBuf("sas")));
    ASSERT_EQ(tags[2], std::make_pair(TStringBuf("prj"), TStringBuf("addrs")));

    auto aggrs = ToAggrs(key);
    ASSERT_TRUE(aggrs.empty());
}

TEST(TInstanceKeyTest, ManyTagsAndSingleAggregate) {
    TInstanceKey key{"addrsupper|ctype=prestable;geo=sas;prj=addrs|tier"};
    ASSERT_EQ(key.GetItype(), "addrsupper");
    ASSERT_EQ(key.GetTags(), "ctype=prestable;geo=sas;prj=addrs");
    ASSERT_EQ(key.GetAggrs(), "tier");

    auto tags = ToTags(key);
    ASSERT_EQ(tags.size(), 3u);
    ASSERT_EQ(tags[0], std::make_pair(TStringBuf("ctype"), TStringBuf("prestable")));
    ASSERT_EQ(tags[1], std::make_pair(TStringBuf("geo"), TStringBuf("sas")));
    ASSERT_EQ(tags[2], std::make_pair(TStringBuf("prj"), TStringBuf("addrs")));

    auto aggrs = ToAggrs(key);
    ASSERT_EQ(aggrs.size(), 1u);
    ASSERT_EQ(aggrs[0], "tier");
}

TEST(TInstanceKeyTest, ManyTagsAndManyAggregates) {
    TInstanceKey key{"addrsupper|ctype=prestable;geo=sas;prj=addrs|tier,unit"};
    ASSERT_EQ(key.GetItype(), "addrsupper");
    ASSERT_EQ(key.GetTags(), "ctype=prestable;geo=sas;prj=addrs");
    ASSERT_EQ(key.GetAggrs(), "tier,unit");

    auto tags = ToTags(key);
    ASSERT_EQ(tags.size(), 3u);
    ASSERT_EQ(tags[0], std::make_pair(TStringBuf("ctype"), TStringBuf("prestable")));
    ASSERT_EQ(tags[1], std::make_pair(TStringBuf("geo"), TStringBuf("sas")));
    ASSERT_EQ(tags[2], std::make_pair(TStringBuf("prj"), TStringBuf("addrs")));

    auto aggrs = ToAggrs(key);
    ASSERT_EQ(aggrs.size(), 2u);
    ASSERT_EQ(aggrs[0], "tier");
    ASSERT_EQ(aggrs[1], "unit");
}

TEST(TInstanceKeyTest, TypicalCommons) {
    TInstanceKey key{"common||ctype,geo,prj,tier"};
    ASSERT_EQ(key.GetItype(), "common");
    ASSERT_EQ(key.GetTags(), "");
    ASSERT_EQ(key.GetAggrs(), "ctype,geo,prj,tier");

    auto tags = ToTags(key);
    ASSERT_TRUE(tags.empty());

    auto aggrs = ToAggrs(key);
    ASSERT_EQ(aggrs.size(), 4u);
    ASSERT_EQ(aggrs[0], "ctype");
    ASSERT_EQ(aggrs[1], "geo");
    ASSERT_EQ(aggrs[2], "prj");
    ASSERT_EQ(aggrs[3], "tier");
}

TEST(TInstanceKeyTest, HostName_OnlyOne) {
    std::vector<TStringBuf> instanceKeys{
        TStringBuf{"addrsupper|host=abc|tier,unit"},
        TStringBuf{"addrsupper|host=abc"},
        TStringBuf{"addrsupper|group=abc|tier,unit"},
        TStringBuf{"addrsupper|group=abc"},
    };

    for (auto instanceKey: instanceKeys) {
        TInstanceKey key{instanceKey};
        ASSERT_EQ(key.GetHostName(), "abc") << "key=" << key;

        auto yasmKey = NTags::TInstanceKey::FromNamed(instanceKey);
        ASSERT_EQ(yasmKey.GetHostName().GetName(), "abc") << "key=" << key;
    }
}

TEST(TInstanceKeyTest, HostName_AtBegin) {
    std::vector<TStringBuf> instanceKeys{
        TStringBuf{"addrsupper|host=abc;ctype=prestable;geo=sas;prj=addrs|tier,unit"},
        TStringBuf{"addrsupper|host=abc;ctype=prestable;geo=sas;prj=addrs"},
        TStringBuf{"addrsupper|group=abc;ctype=prestable;geo=sas;prj=addrs|tier,unit"},
        TStringBuf{"addrsupper|group=abc;ctype=prestable;geo=sas;prj=addrs"},
    };

    for (auto instanceKey: instanceKeys) {
        TInstanceKey key{instanceKey};
        ASSERT_EQ(key.GetHostName(), "abc") << "key=" << key;

        auto yasmKey = NTags::TInstanceKey::FromNamed(instanceKey);
        ASSERT_EQ(yasmKey.GetHostName().GetName(), "abc") << "key=" << key;
    }
}

TEST(TInstanceKeyTest, HostName_AtEnd) {
    std::vector<TStringBuf> instanceKeys{
        TStringBuf{"addrsupper|ctype=prestable;geo=sas;prj=addrs;host=abc|tier,unit"},
        TStringBuf{"addrsupper|ctype=prestable;geo=sas;prj=addrs;host=abc"},
        TStringBuf{"addrsupper|ctype=prestable;geo=sas;prj=addrs;group=abc|tier,unit"},
        TStringBuf{"addrsupper|ctype=prestable;geo=sas;prj=addrs;group=abc"},
    };

    for (auto instanceKey: instanceKeys) {
        TInstanceKey key{instanceKey};
        ASSERT_EQ(key.GetHostName(), "abc") << "key=" << key;

        auto yasmKey = NTags::TInstanceKey::FromNamed(instanceKey);
        ASSERT_EQ(yasmKey.GetHostName().GetName(), "abc") << "key=" << key;
    }
}

TEST(TInstanceKeyTest, HostName_GroupAndHost) {
    std::vector<TStringBuf> instanceKeys{
        TStringBuf{"addrsupper|ctype=prestable;host=abc;geo=sas;group=xyz;prj=addrs|tier,unit"},
        TStringBuf{"addrsupper|ctype=prestable;host=abc;geo=sas;group=xyz;prj=addrs"},
        TStringBuf{"addrsupper|ctype=prestable;group=xyz;geo=sas;host=abc;prj=addrs|tier,unit"},
        TStringBuf{"addrsupper|ctype=prestable;group=xyz;geo=sas;host=abc;prj=addrs"},
    };

    for (auto instanceKey: instanceKeys) {
        TInstanceKey key{instanceKey};
        ASSERT_EQ(key.GetHostName(), "abc") << "key=" << key;

        auto yasmKey = NTags::TInstanceKey::FromNamed(instanceKey);
        ASSERT_EQ(yasmKey.GetHostName().GetName(), "abc") << "key=" << key;
    }
}
