#include <solomon/services/fetcher/lib/yasm/hist_builder.h>

#include <solomon/libs/cpp/testing/matchers.h> // operator== for histograms

#include <infra/yasm/interfaces/internal/agent.pb.h>

#include <library/cpp/monlib/consumers/collecting_consumer.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <tuple>
#include <iostream>

using namespace NMonitoring;
using namespace NSolomon::NFetcher::NYasm;
using namespace NSolomon;
using namespace NYasm::NInterfaces::NInternal;

TValue CreateUgram(const TVector<std::tuple<double, double, double>>& vals) {
    TValue result;
    auto* ugram = result.MutableUgram();

    for (auto [lo, hi, w]: vals) {
        auto* bucket = ugram->AddBuckets();
        bucket->SetLowerBound(lo);
        bucket->SetUpperBound(hi);
        bucket->SetWeight(w);
    }

    return result;
}

TValue CreateConstUgram(size_t bucketCount, double val = 0.) {
    TValue result;
    auto* ugram = result.MutableUgram();

    for (auto i = 0u; i < bucketCount; ++i) {
        auto* bucket = ugram->AddBuckets();
        bucket->SetLowerBound(i);
        bucket->SetUpperBound(i + 1);
        bucket->SetWeight(val);
    }

    return result;
}

TValue CreateConstUgramWithGaps(size_t bucketCount, double val = 0.) {
    TValue result;
    auto* ugram = result.MutableUgram();

    auto low = 0;
    for (auto i = 0u; i < bucketCount; ++i) {
        auto* bucket = ugram->AddBuckets();
        bucket->SetLowerBound(low);
        bucket->SetUpperBound(low + 2);
        bucket->SetWeight(val);

        low += 4;
    }

    return result;
}

TValue CreateFloat(double val) {
    TValue result;
    result.MutableFloatValue()->SetValue(val);
    return result;
}

TValue CreateNone() {
    TValue result;
    result.MutableNoneValue();
    return result;
}

TValue CreateVec() {
    TValue result;
    result.MutableVec();
    return result;
}

TValue CreateHyperLogLog() {
    TValue result;
    result.MutableHyperLogLog();
    return result;
}

IHistogramSnapshotPtr CreateHist(const TVector<std::pair<TBucketBound, TBucketValue>>& vs) {
    THistogramBuilder b{vs.size()};
    for (auto v: vs) {
        b.Add(v.first, v.second);
    }
    return b.Finalize();
}

class TValueTestBase: public testing::Test {
    void SetUp() override {
        Consumer_ = {};
    }

protected:
    TCollectingConsumer Consumer_;
};

class TUgramValueConverterTest: public TValueTestBase {
};

TEST_F(TUgramValueConverterTest, UgramEmpty) {
    auto empty = THistogramBuilder{0}.Finalize();
    auto val = CreateUgram({});
    auto ok = ConvertValue(val, Consumer_);
    ASSERT_FALSE(ok);

//    auto&& metrics = Consumer_.Metrics;
//    ASSERT_EQ(metrics.size(), 1u);
//    ASSERT_EQ(metrics[0].Kind, EMetricType::HIST);
//
//    auto* hist = (*metrics[0].Values)[0].GetValue().AsHistogram();
//    ASSERT_EQ(*hist, *empty);
}

TEST_F(TUgramValueConverterTest, UgramIsCompressed) {
    auto ugram = CreateConstUgram(52, 1);

    THistogramBuilder builder{HISTOGRAM_MAX_BUCKETS_COUNT};
    builder.Add(0, 0);
    builder.Add(5, 5);
    for (auto i = 6; i <= 52; ++i) {
        builder.Add(i, 1);
    }

    const auto expected = builder.Finalize();

    ConvertValue(ugram, Consumer_);
    auto&& metrics = Consumer_.Metrics;
    auto* hist = (*metrics[0].Values)[0].GetValue().AsHistogram();
    ASSERT_EQ(*hist, *expected);
    ASSERT_EQ(hist->Count(), HISTOGRAM_MAX_BUCKETS_COUNT - 1);
}

TEST_F(TUgramValueConverterTest, ZeroUgramIsCompressed) {
    auto ugram = CreateConstUgram(52);
    /// XXX recheck how it works now
    auto expected = CreateHist({{0, 0}, {52, 0}});

    ConvertValue(ugram, Consumer_);
    auto&& metrics = Consumer_.Metrics;
    auto* hist = (*metrics[0].Values)[0].GetValue().AsHistogram();
    ASSERT_EQ(*hist, *expected);
}

TEST_F(TUgramValueConverterTest, UgramEmptyIsSkipped) {
    auto ugram = CreateUgram({
        {5.0, 6.0, 10.1},
        {6.0, 7.0, 10.9},
        {7.0, 8.0, 10.5},
        {8.0, 9.0, 1.0},
        {9.0, 10.0, 0.3},
        {10.0, 11.0, 0.8}
    });

    auto expected = CreateHist({{5, 0}, {6, 10}, {7, 11}, {8, 11}, {9, 1}, {10, 0}, {11, 1}});

    auto ok  = ConvertValue(ugram, Consumer_);
    ASSERT_TRUE(ok);
    auto&& metrics = Consumer_.Metrics;
    ASSERT_EQ(metrics.size(), 1u);
    ASSERT_EQ(metrics[0].Kind, EMetricType::HIST);

    auto* hist = (*metrics[0].Values)[0].GetValue().AsHistogram();
    ASSERT_EQ(*hist, *expected);
}

TEST_F(TUgramValueConverterTest, UgramWithGaps) {
    auto ugram = CreateUgram({
        {5.0, 6.0, 10.1},
        {16.0, 17.0, 10.9},
        {18.0, 20.0, 10.5}
    });

    auto expected = CreateHist({{5, 0}, {6, 10}, {16, 0}, {17, 11}, {18, 0}, {20, 11}});

    ConvertValue(ugram, Consumer_);
    auto&& metrics = Consumer_.Metrics;
    auto* hist = (*metrics[0].Values)[0].GetValue().AsHistogram();
    ASSERT_EQ(*hist, *expected);
}

TEST_F(TUgramValueConverterTest, UgramWithPointBacket) {
    auto ugram = CreateUgram({
        {10000.0, 10000.0, 40}
    });

    auto expected = CreateHist({
        {1000.0, 0},
        {10000.000000000002, 40}
    });

    ConvertValue(ugram, Consumer_);
    auto&& metrics = Consumer_.Metrics;
    auto* hist = (*metrics[0].Values)[0].GetValue().AsHistogram();
    ASSERT_EQ(*hist, *expected);
}

TEST_F(TUgramValueConverterTest, UgramWithManyGaps) {
    auto ugram = CreateConstUgramWithGaps(50, 1);
    ConvertValue(ugram, Consumer_);
    auto* hist = (*Consumer_.Metrics[0].Values)[0].GetValue().AsHistogram();
    auto infVal = hist->Value(hist->Count() - 1);
    ASSERT_EQ(infVal, 0u);
}

class TFloatValueConverterTest: public TValueTestBase {
};

TEST_F(TFloatValueConverterTest, Simple) {
    auto ok = ConvertValue(CreateFloat(1.), Consumer_);
    ASSERT_TRUE(ok);
    auto&& metric = Consumer_.Metrics[0];
    ASSERT_EQ(metric.Kind, EMetricType::GAUGE);
    ASSERT_EQ(metric.Values->GetValueType(), EMetricValueType::DOUBLE);
    ASSERT_EQ(metric.Values->Size(), 1u);
    ASSERT_EQ((*metric.Values)[0].GetValue().AsDouble(), 1.);
}

class TUnsupportedTypesTest: public TValueTestBase {
};

TEST_F(TUnsupportedTypesTest, Unsupported) {
    ASSERT_FALSE(ConvertValue(TValue{}, Consumer_));
    ASSERT_FALSE(ConvertValue(CreateNone(), Consumer_));
    ASSERT_FALSE(ConvertValue(CreateVec(), Consumer_));
    ASSERT_FALSE(ConvertValue(CreateHyperLogLog(), Consumer_));
}
