#include "app.h"

#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <solomon/libs/cpp/signals/signals.h>
#include <solomon/libs/cpp/config/units.h>

#include <library/cpp/grpc/server/grpc_server.h>

#include <util/generic/size_literals.h>
#include <util/string/builder.h>

using namespace NActors;
using namespace NMonitoring;

namespace NSolomon {
namespace {
    THolder<NGrpc::TGRpcServer> CreateGrpcServer(const yandex::solomon::config::rpc::TGrpcServerConfig& config) {
        NGrpc::TServerOptions opts;

        opts.SetHost("[::]")
            .SetPort(config.GetPort(0))
            .SetWorkerThreads(2)
            .SetMaxMessageSize(config.HasMaxMessageSize() ? FromProtoDataSize(config.GetMaxMessageSize()) : 4_MB);

        return MakeHolder<NGrpc::TGRpcServer>(opts);
    }

    class TApp: public IApp {
    public:
        void Start() override {
            sigset_t blockMask;
            SigEmptySet(&blockMask);
            while (true) {
                SigSuspend(&blockMask);

                if (NeedTerminate) {
                    ActorRuntime_->EmergencyLog(TStringBuilder() << AppName() << " was terminated");
                    break;
                } else if (NeedReopenLog) {
                    NeedReopenLog = 0;
                    ActorRuntime_->ReopenLog();
                    TLogBackend::ReopenAllBackends();
                }
            }
        }

        void Stop() override {
            if (GrpcServer_) {
                GrpcServer_->Stop();
                GrpcServer_.Reset();
            }
        }

        TStringBuf AppName() const override {
            return AppName_;
        }

        TActorRuntime& Runtime() override {
            return *ActorRuntime_;
        }

        const TActorRuntime& Runtime() const override {
            return *ActorRuntime_;
        }

        std::shared_ptr<TMetricRegistry> MetricRegistry() override {
            return Registry_;
        }

    public:
        TStringBuf AppName_;
        std::shared_ptr<TMetricRegistry> Registry_;
        THolder<TActorRuntime> ActorRuntime_;
        THolder<NGrpc::TGRpcServer> GrpcServer_;
    };

    struct TContext: TAppBuilder::IContext {
        NMonitoring::TMetricRegistry& MetricRegistry;

        TContext(NMonitoring::TMetricRegistry& r)
            : MetricRegistry{r}
        {
        }

        TMetricRegistry& Registry() override {
            return MetricRegistry;
        }
    };

} // namespace
    THolder<IApp> TAppBuilder::Build() {
        auto app = MakeHolder<TApp>();

        Y_ENSURE(ActorSystemConfig_, "Actor system config must be specified");

        app->ActorRuntime_ = TActorRuntime::Create(*ActorSystemConfig_, app->MetricRegistry(), AppData_);
        app->ActorRuntime_->Start();

        if (ActorServiceFactory_) {
            TContext ctx{*app->MetricRegistry()};
            ActorServiceFactory_->CreateServices(app->Runtime(), ctx);
        }

        if (RpcConfig_) {
            app->GrpcServer_ = CreateGrpcServer(*RpcConfig_);
        }

        if (!GrpcServices_.empty()) {
            Y_ENSURE("gRPC server is not configured -- cannot initialize services");
            for (auto& service: GrpcServices_) {
                app->GrpcServer_->AddService(std::move(service));
            }

            app->GrpcServer_->Start();
        }

        return app;
    }

} // namespace NSolomon
