#pragma once

#include <solomon/protos/configs/rpc/rpc_config.pb.h>
#include <solomon/libs/cpp/actors/config/actor_system_config.pb.h>

#include <library/cpp/grpc/server/grpc_server.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/dynamic_counters/counters.h>

#include <util/generic/ptr.h>

#include <optional>

namespace NSolomon {
    class TActorRuntime;

    using IGrpcServicePtr = TIntrusivePtr<NGrpc::IGRpcService>;

    class IApp {
    public:
        virtual ~IApp() = default;

        virtual TStringBuf AppName() const = 0;

        virtual TActorRuntime& Runtime() = 0;
        virtual const TActorRuntime& Runtime() const = 0;

        virtual std::shared_ptr<NMonitoring::TMetricRegistry> MetricRegistry() = 0;

        virtual void Start() = 0;
        virtual void Stop() = 0;
    };

    class IActorServiceFactory;

    class TAppBuilder {
    public:
        struct IContext {
            virtual ~IContext() = default;
            virtual NMonitoring::TMetricRegistry& Registry() = 0;
        };

    public:
        virtual ~TAppBuilder() = default;

        explicit TAppBuilder(TStringBuf appName)
            : AppName_{appName}
        {
        }

        TAppBuilder& WithGrpcServerConf(yandex::solomon::config::rpc::TGrpcServerConfig rpcConfig) {
            RpcConfig_ = std::move(rpcConfig);
            return *this;
        }

        TAppBuilder& WithGrpcService(IGrpcServicePtr srv) {
            GrpcServices_.push_back(std::move(srv));
            return *this;
        }

        TAppBuilder& WithAppData(void* appData) {
            AppData_ = appData;
            return *this;
        }

        TAppBuilder& WithActorSystemConfig(NSolomon::TActorSystemConfig conf) {
            ActorSystemConfig_ = std::move(conf);
            return *this;
        }

        TAppBuilder& WithActorServiceFactory(IActorServiceFactory* factory) {
            Y_ENSURE(factory);
            ActorServiceFactory_ = factory;
            return *this;
        }

        virtual THolder<IApp> Build();

    protected:
        TStringBuf AppName_;
        std::optional<yandex::solomon::config::rpc::TGrpcServerConfig> RpcConfig_;
        TVector<IGrpcServicePtr> GrpcServices_;
        void* AppData_{nullptr};
        IActorServiceFactory* ActorServiceFactory_{nullptr};
        std::optional<TActorSystemConfig> ActorSystemConfig_;
    };

    class IActorServiceFactory {
    public:
        virtual ~IActorServiceFactory() = default;
        virtual void CreateServices(TActorRuntime& runtime, TAppBuilder::IContext&) = 0;
    };

} // namespace NSolomon
