#include <solomon/services/fetcher/ufetcher/app/app.h>
#include <solomon/services/fetcher/ufetcher/ufetcher_factory.h>

#include <solomon/services/fetcher/lib/dns/continuous_resolver.h>
#include <solomon/libs/cpp/dns/dns.h>
#include <solomon/services/fetcher/lib/host_groups/host_and_labels.h>
#include <solomon/services/fetcher/lib/data_sink/data_sink.h>
#include <solomon/services/fetcher/lib/sink/sink.h>

#include <solomon/libs/cpp/actors/config/actor_system_config.pb.h>
#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <solomon/libs/cpp/actors/runtime/config.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/charset/codepage.h>
#include <library/cpp/string_utils/url/url.h>

using namespace NActors;
using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace NMonitoring;

class TActorServiceFactory: public IActorServiceFactory {
public:
    void CreateServices(TActorRuntime& runtime, TAppBuilder::IContext& ctx) override {
        TDnsResolverActorConf conf {
            .DnsClient = DnsClient_,
            .MetricRegistry = ctx.Registry(),
        };

        auto aid = runtime.Register(CreateDnsResolverActor(std::move(conf)));
        runtime.RegisterLocalService(
            MakeDnsResolverId(),
            aid
        );
    }

private:
    IDnsClientPtr DnsClient_{CreateDnsClient()};
};

EFetcherShardType TypeFromString(TString type) {
    ToUpper(type.begin(), type.size());

    if (type == "YASM") {
        return EFetcherShardType::YasmAgent;
    } else if (type == "AGENT") {
        return EFetcherShardType::Agent;
    } else {
        ythrow yexception() << "URL type " << type << " is not supported";
    }
}

IActor* CreateIngestorDataSink(TStringBuf address, std::shared_ptr<TMetricRegistry> metrics) {
    yandex::solomon::config::rpc::TGrpcClientConfig conf;

    TStringBuf scheme, host;
    ui16 port{0};

    // just to ensure that we have valid input
    GetSchemeHostAndPort(address, scheme, host, port);

    *conf.AddAddresses() = address;
    auto ingestorClient = NIngestor::CreateIngestorGrpcClient(std::move(conf), *metrics);
    auto* overflows = metrics->Counter({{"sensor", "queueOverflow"}});

    return ::NSolomon::NFetcher::CreateDataSink(
        CreateShardWriterFactory(
            WrapProcessingClient(ingestorClient.release()),
            { .ShardMetrics = std::move(metrics), } 
        ),
        overflows
    );
}

int main(int argc, char** argv) try {
    using namespace NLastGetopt;

    TVector<TString> urls;
    TString type;

    TOpts opts;
    opts.AddLongOption('u', "url", "URL to fetch (may be specified more than once)")
        .RequiredArgument("URL")
        .EmplaceTo(&urls)
        .Required();

    opts.AddLongOption('t', "type", "URL(s) type. Allowed values: YASM|AGENT")
        .RequiredArgument("TYPE")
        .StoreResult(&type)
        .Required();

    auto& debugSink = opts.AddLongOption("debug", "outputs all collected data to stdout")
        .NoArgument()
        .Optional();

    auto& ingestorSink = opts.AddLongOption("ingestor")
        .RequiredArgument("ADDRESS")
        .Optional();

    opts.MutuallyExclusiveOpt(debugSink, ingestorSink);

    opts.AddHelpOption('h');
    opts.SetFreeArgsNum(0);

    TOptsParseResult r{&opts, argc, argv};

    auto asConf = MinimalActorSystemConfig();
    TActorServiceFactory serviceFactory;

    auto app = TAppBuilder{TStringBuf("ufetcher")}
        .WithActorSystemConfig(std::move(asConf))
        .WithActorServiceFactory(&serviceFactory)
        .Build();

    auto& runtime = app->Runtime();
    auto createDataSink = [&] (TOptsParseResult& r) {
        if (r.Has(&debugSink)) {
            return CreateDebugDataSink();
        } else if (r.Has(&ingestorSink)) {
            return CreateIngestorDataSink(TStringBuf{r.Get(&ingestorSink)}, app->MetricRegistry());
        } else {
            ythrow yexception() << "Data sink must be specified";
        }
    };

    auto urlFactory = TMicroUrlFactoryBuilder{}
        .WithShardType(TypeFromString(type))
        .WithDataSinkId(runtime.Register(createDataSink(r)))
        .WithDnsResolverId(MakeDnsResolverId())
        .Build();

    for (auto&& url: urls) {
        auto hostAndLabels = THostAndLabels::FromString(url);
        urlFactory->CreateUrlActor(app->Runtime(), std::move(hostAndLabels));
    }

    app->Start();
    return 0;
} catch (...) {
    Cerr << CurrentExceptionMessage();
    return -1;
}
