package ru.yandex.solomon.alert.gateway.context;

import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.discovery.DiscoveryServices;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.conf.GrpcConfigurationContext;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.client.AlertingClients;
import ru.yandex.solomon.alert.client.MuteApi;
import ru.yandex.solomon.alert.client.NotificationApi;
import ru.yandex.solomon.alert.client.TelegramApi;
import ru.yandex.solomon.alert.client.YaChatsApi;
import ru.yandex.solomon.alert.client.grpc.AlertTemplateClient;
import ru.yandex.solomon.alert.client.grpc.AlertingCluster;
import ru.yandex.solomon.alert.client.grpc.TelegramClient;
import ru.yandex.solomon.alert.client.grpc.YaChatsClient;
import ru.yandex.solomon.alert.gateway.converter.DashboardYasmQueryConverter;
import ru.yandex.solomon.alert.gateway.endpoint.AlertController;
import ru.yandex.solomon.alert.gateway.endpoint.AlertInternalController;
import ru.yandex.solomon.alert.gateway.endpoint.AlertTemplateDeprecatedController;
import ru.yandex.solomon.alert.gateway.endpoint.AlertTemplateNewController;
import ru.yandex.solomon.alert.gateway.endpoint.AlertsListController;
import ru.yandex.solomon.alert.gateway.endpoint.EscalationInternalController;
import ru.yandex.solomon.alert.gateway.endpoint.MuteController;
import ru.yandex.solomon.alert.gateway.endpoint.NotificationChannelController;
import ru.yandex.solomon.alert.gateway.endpoint.NotificationChannelForSeverityController;
import ru.yandex.solomon.alert.gateway.endpoint.TelegramController;
import ru.yandex.solomon.alert.gateway.endpoint.YaChatsController;
import ru.yandex.solomon.alert.gateway.endpoint.YasmController;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.config.protobuf.alerting.gateway.AlertingGatewayConfig;
import ru.yandex.solomon.config.protobuf.frontend.MonitoringConfig;
import ru.yandex.solomon.config.protobuf.frontend.TYasmGatewayConfig;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.spring.ConditionalOnBean;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@ConditionalOnBean(AlertingGatewayConfig.class)
@Import({
        GrpcConfigurationContext.class
})
public class AlertingGatewayContext {
    private final AlertingGatewayConfig config;

    @Autowired
    public AlertingGatewayContext(AlertingGatewayConfig config) {
        this.config = config;
    }

    @Bean
    public AlertingClient alertingClient(AlertingCluster cluster) {
        return AlertingClients.create(cluster);
    }

    @Bean
    public AlertingCluster alertingCluster(
            ClientOptionsFactory clientOptionsFactory)
    {
        var clientConfig = config.getClientConfig();
        var addresses = DiscoveryServices.resolve(clientConfig.getGrpcClientConfig().getAddressesList());
        var opts = clientOptionsFactory.newBuilder(
                "AlertingGatewayConfig.ClientConfig",
                clientConfig.getGrpcClientConfig())
            .build();
        return new AlertingCluster(addresses, opts);
    }

    @Bean
    public TelegramClient telegramClient(AlertingCluster cluster) {
        return new TelegramClient(cluster);
    }

    @Bean
    public AlertTemplateClient alertTemplateClient(AlertingCluster cluster) {
        return new AlertTemplateClient(cluster);
    }

    @Bean
    public YaChatsClient yaChatsClient(AlertingCluster cluster) {
        return new YaChatsClient(cluster);
    }

    @Bean(name = "NotificationChannelController")
    public NotificationChannelController notificationChannelController(
        NotificationApi api, ProjectsDao projectsDao, Authorizer authorizer)
    {
        return new NotificationChannelController(api, projectsDao, authorizer);
    }

    @Bean(name = "escalationInternalController")
    public EscalationInternalController escalationInternalController(
            NotificationApi api, Authorizer authorizer)
    {
        return new EscalationInternalController(api, authorizer);
    }

    @Bean(name = "NotificationChannelForSeverityController")
    public NotificationChannelForSeverityController notificationChannelForSeverityController(
            NotificationApi api, Authorizer authorizer)
    {
        return new NotificationChannelForSeverityController(api, authorizer);
    }

    @Bean(name = "AlertController")
    public AlertController alertController(AlertApi api, ProjectsDao projectsDao, Authorizer authorizer) {
        return new AlertController(api, projectsDao, authorizer);
    }

    @Bean(name = "AlertInternalController")
    public AlertInternalController alertInternalController(AlertApi api, ProjectsDao projectsDao, Authorizer authorizer) {
        return new AlertInternalController(api, projectsDao, authorizer);
    }

    @Bean(name = "AlertListController")
    public AlertsListController alertsListController(AlertApi api, Authorizer authorizer) {
        return new AlertsListController(api, authorizer);
    }

    @Bean(name = "MuteController")
    public MuteController muteController(MuteApi api, ProjectsDao projectsDao, Authorizer authorizer) {
        return new MuteController(api, projectsDao, authorizer);
    }

    @Bean(name = "TelegramController")
    public TelegramController telegramController(TelegramApi api) {
        return new TelegramController(api);
    }

    @Autowired
    @Bean(name = "AlertTemplateDeprecatedController")
    public AlertTemplateDeprecatedController alertTemplateDeprecatedController(AlertTemplateClient alertTemplateClient, Authorizer authorizer) {
        return new AlertTemplateDeprecatedController(alertTemplateClient, authorizer);
    }

    @Autowired
    @Bean(name = "AlertTemplateNewController")
    public AlertTemplateNewController alertTemplateNewController(AlertTemplateClient alertTemplateClient, Authorizer authorizer) {
        return new AlertTemplateNewController(alertTemplateClient, authorizer);
    }

    @Autowired
    @Bean(name = "YaChatsController")
    public YaChatsController yaChatsController(YaChatsApi api) {
        return new YaChatsController(api);
    }

    @Bean
    public DashboardYasmQueryConverter dashboardExpressionConverter(
            Optional<TYasmGatewayConfig> tYasmGatewayConfig,
            Optional<MonitoringConfig> monitoringConfig
    ) {
        var prefix = tYasmGatewayConfig.map(TYasmGatewayConfig::getYasmProjectsPrefix).orElse("yasm_");
        return new DashboardYasmQueryConverter(prefix, monitoringConfig.orElse(MonitoringConfig.newBuilder()
                .setBaseUrl("https://monitoring-preprod.yandex-team.ru/")
                .build()));
    }

    @Bean(name = "YasmController")
    public YasmController yasmController(
            Authorizer authorizer,
            DashboardYasmQueryConverter dashboardYasmQueryConverter)
    {
        return new YasmController(authorizer, dashboardYasmQueryConverter);
    }
}
