package ru.yandex.solomon.alert.gateway.converter;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.alert.gateway.yasm.dto.ConvertYasmQueriesDto;
import ru.yandex.solomon.alert.gateway.yasm.dto.ConvertYasmQueryDto;
import ru.yandex.solomon.config.protobuf.frontend.MonitoringConfig;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.yasm.expression.grammar.YasmToSelConverter;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class DashboardYasmQueryConverter {

    private final YasmToSelConverter converter;
    private final String yasmPrefix;
    private final MonitoringConfig monitoringConfig;

    public DashboardYasmQueryConverter(String yasmPrefix, MonitoringConfig monitoringConfig) {
        this.converter = new YasmToSelConverter(yasmPrefix);
        this.yasmPrefix = yasmPrefix;
        this.monitoringConfig = monitoringConfig;
    }

    public String prepareUrl(ConvertYasmQueriesDto request) {
        var signals = Nullables.orEmpty(request.signals);
        if (signals.isEmpty()) {
            throw new BadRequestException("Signals must be specified");
        }
        String project = null;
        for (ConvertYasmQueryDto signal : signals) {
            var list = Nullables.orEmpty(signal.tags).getOrDefault("itype", List.of());
            if (!list.isEmpty()) {
                project = list.get(0);
                break;
            }
        }
        if (project == null) {
            throw new BadRequestException("Signals must contain itype tag");
        }
        StringBuilder sb = new StringBuilder(monitoringConfig.getBaseUrl())
                .append("projects/")
                .append(yasmPrefix)
                .append(project)
                .append("/explorer/queries?");
        boolean normalizeGlobal = request.settings != null && Boolean.TRUE.equals(request.settings.normalize);
        for (int i = 0; i < signals.size(); i++) {
            var convertYasmQueryDto = signals.get(i);
            var tags = Nullables.orEmpty(convertYasmQueryDto.tags);
            if (!StringUtils.isEmpty(convertYasmQueryDto.host)) {
                tags.put("hosts", List.of(convertYasmQueryDto.host));
            }
            boolean normalize = convertYasmQueryDto.normalize == null
                ? normalizeGlobal
                : convertYasmQueryDto.normalize;
            var expression = converter.convertExpression(Nullables.orEmpty(convertYasmQueryDto.name), tags, Nullables.orEmpty(convertYasmQueryDto.title), normalize);
            if (i > 0) {
                sb.append("&");
            }
            sb.append("q.").append(i).append(".text=").append(UrlUtils.urlEncode(expression));
            if (Boolean.TRUE.equals(convertYasmQueryDto.rightAxis)) {
                sb.append("&q.").append(i).append(".axis=r");
            }
        }
        if (request.settings != null && Boolean.TRUE.equals(request.settings.stack)) {
            sb.append("&type=area");
        } else {
            sb.append("&type=line");
        }
        if (request.settings != null && !StringUtils.isEmpty(request.settings.range)) {
            sb.append("&range=").append(request.settings.range);
        }
        if (request.settings != null && !StringUtils.isEmpty(request.settings.rangeFrom)) {
            sb.append("&from=").append(request.settings.rangeFrom);
        }
        if (request.settings != null && !StringUtils.isEmpty(request.settings.rangeTo)) {
            sb.append("&to=").append(request.settings.rangeTo);
        }
        if (request.settings != null && !StringUtils.isEmpty(request.settings.refresh)) {
            sb.append("&refresh=").append(request.settings.refresh);
        }
        return sb.toString();
    }
}
