package ru.yandex.solomon.alert.gateway.dto.alert;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import org.apache.logging.log4j.util.Strings;

import ru.yandex.solomon.alert.protobuf.EAlertState;
import ru.yandex.solomon.alert.protobuf.NoPointsPolicy;
import ru.yandex.solomon.alert.protobuf.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TNotificationChannelOptions;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
@ApiModel("Alert")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class AlertDto {
    @ApiModelProperty(
            value = "unique identity for alert",
            readOnly = true,
            required = true,
            dataType = "java.util.UUID",
            position = 0)
    public String id;

    @ApiModelProperty(
            value = "Project if own by this alert",
            readOnly = true,
            required = true,
            dataType = "java.util.UUID",
            position = 1)
    public String projectId;

    @ApiModelProperty(
            value = "Human-readable name of alert",
            required = true,
            example = "Disk free space",
            position = 2)
    public String name;

    @ApiModelProperty(
            value = "actual version of notification",
            position = 3)
    public Integer version;

    @ApiModelProperty(
            value = "User created alert",
            readOnly = true,
            position = 4)
    public String createdBy;

    @ApiModelProperty(
            value = "Time when alert was created",
            readOnly = true,
            dataType = "ISO-8601",
            example = "2017-12-19T12:50:53.583Z",
            position = 5)
    public String createdAt;

    @ApiModelProperty(
            value = "User updated alert",
            readOnly = true,
            position = 6)
    public String updatedBy;

    @ApiModelProperty(
            value = "Time when alert was updated last time",
            readOnly = true,
            dataType = "ISO-8601",
            example = "2017-12-19T12:55:53.583Z",
            position = 7)
    public String updatedAt;

    @ApiModelProperty(
            value = "State of current alert, only ACTIVE alerts will be periodically checked",
            required = true,
            position = 8)
    public EAlertState state;

    @ApiModelProperty(
            value = "List of label key that should be use to group metrics, each group it's " +
                    "separate sub alert that check independently from other group. " +
                    "Can't change in alert from template, instead use field in template",
            position = 9)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<String> groupByLabels;

    @Deprecated
    @ApiModelProperty(
            value = "Deprecated. Notification channel ids that will be use to notify about " +
                    "alert evaluation status",
            position = 10)
    public List<String> notificationChannels;

    @ApiModelProperty(
            value = "Notification channels that will receive events",
            required = true,
            position = 11)
    public List<AssociatedChannelDto> channels;

    @ApiModelProperty(
            value = "Type specific configuration",
            required = true,
            position = 12)
    public Type type;

    @ApiModelProperty(
            value = "Templates that explain alert evaluation status, and what todo when alert occurs. " +
                    "Variables available to use into template depends on alert type.",
            required = false,
            position = 13)
    public Map<String, String> annotations;

    @Deprecated
    // TODO: describe period inside expression? e.g idleTime{host="test"}[10m] (gordiychuk@)
    @ApiModelProperty(
        value = "Alert window width in in milliseconds. Period plus aggregate allow " +
            "smooth frequently change parameter. Can't change in alert from template, instead use field in template",
        required = true,
        position = 14)
    public Long periodMillis;

    @Deprecated
    @ApiModelProperty(
            value = "Deprecated. Time (in seconds) to delay evaluation relatively now. " +
                    "Useful for delayed metrics like cluster aggregation. " +
                    "Should be a non negative integer. " +
                    "Can't change in alert from template, instead use field in template",
            required = false,
            position = 15)
    public int delaySeconds;

    @ApiModelProperty(
            value = "Alert window width in seconds. Can be used only if periodMillis is not set, null or 0. Can't change in alert from template, instead use field in template",
            position = 16)
    public Long windowSecs;

    @ApiModelProperty(
            value = "Alert delay in seconds. Can't change in alert from template, instead use field in template",
            position = 17)
    public Integer delaySecs;

    @ApiModelProperty(
            value = "Description about alert in markdown format",
            position = 18)
    public String description;

    @ApiModelProperty(
            value = "NO_DATA policy for selectors that resolve to empty set of metrics. " +
                    "Can't change in alert from template, instead use field in template",
            position = 19)
    public ResolvedEmptyPolicy resolvedEmptyPolicy;

    @ApiModelProperty(
            value = "NO_DATA policy for timeseries without data points. " +
                    "Can't change in alert from template, instead use field in template",
            position = 20)
    public NoPointsPolicy noPointsPolicy;

    @ApiModelProperty(
            value = "Alert labels.",
            position = 21)
    public Map<String, String> labels;

    @ApiModelProperty(
            value = "Templates that explain alert evaluation status, and what todo when alert occurs. " +
                    "Variables available to use into template depends on alert type.",
            position = 22)
    public Map<String, String> serviceProviderAnnotations;

    @ApiModelProperty(
            value = "Alert severity.",
            position = 23)
    public Severity severity;

    @ApiModelProperty(
            value = "Escalations that will receive events.",
            position = 24)
    public List<String> escalations;

    @ApiModelProperty(
        value = "Alert labels provided by template.",
        position = 21)
    public Map<String, String> serviceProviderLabels;

    public static AlertDto fromProto(@Nonnull TAlert proto) {
        AlertDto dto = new AlertDto();
        dto.id = proto.getId();
        dto.projectId = proto.getProjectId();
        dto.version = proto.getVersion();
        dto.name = proto.getName();
        dto.description = proto.getDescription();
        dto.createdBy = proto.getCreatedBy();
        dto.createdAt = Instant.ofEpochMilli(proto.getCreatedAt()).toString();
        dto.updatedBy = proto.getUpdatedBy();
        dto.updatedAt = Instant.ofEpochMilli(proto.getUpdatedAt()).toString();
        dto.state = proto.getState();
        dto.groupByLabels = ImmutableList.copyOf(proto.getGroupByLabelsList());
        dto.notificationChannels = ImmutableList.copyOf(proto.getNotificationChannelIdsList());
        dto.channels = proto.getConfiguredNotificationChannelsMap().entrySet().stream()
                .map(e -> AssociatedChannelDto.of(e.getKey(), ChannelConfigDto.fromProto(e.getValue())))
                .collect(Collectors.toList());
        dto.type = Type.fromProto(proto);
        dto.annotations = ImmutableMap.copyOf(proto.getAnnotationsMap());
        dto.serviceProviderAnnotations = ImmutableMap.copyOf(proto.getServiceProviderAnnotationsMap());
        dto.labels = ImmutableMap.copyOf(proto.getLabelsMap());
        dto.serviceProviderLabels = ImmutableMap.copyOf(proto.getServiceProviderLabelsMap());
        dto.periodMillis = proto.getPeriodMillis();
        dto.delaySeconds = proto.getDelaySeconds();
        dto.windowSecs = proto.getPeriodMillis() / 1000L;
        dto.delaySecs = proto.getDelaySeconds();
        dto.resolvedEmptyPolicy = proto.getResolvedEmptyPolicy();
        dto.noPointsPolicy = proto.getNoPointsPolicy();
        dto.severity = proto.getSeverity();
        dto.escalations = proto.getEscalationsList();
        return dto;
    }

    public TAlert toProto() {
        TAlert.Builder proto = TAlert.newBuilder()
                .setId(Nullables.orEmpty(id))
                .setProjectId(Nullables.orEmpty(projectId))
                .setVersion(Nullables.orDefault(version, 0))
                .setName(Nullables.orEmpty(name))
                .setDescription(Nullables.orEmpty(description))
                .addAllGroupByLabels(Nullables.orEmpty(groupByLabels))
                .setState(Nullables.orDefault(state, EAlertState.ACTIVE))
                .setCreatedBy(Nullables.orEmpty(createdBy))
                .setUpdatedBy(Nullables.orEmpty(updatedBy))
                .putAllAnnotations(Nullables.orEmpty(annotations))
                .putAllServiceProviderAnnotations(Nullables.orEmpty(serviceProviderAnnotations))
                .putAllLabels(Nullables.orEmpty(labels))
                .setResolvedEmptyPolicy(Nullables.orDefault(resolvedEmptyPolicy, ResolvedEmptyPolicy.RESOLVED_EMPTY_DEFAULT))
                .setNoPointsPolicy(Nullables.orDefault(noPointsPolicy, NoPointsPolicy.NO_POINTS_DEFAULT))
                .setSeverity(Nullables.orDefault(severity, Severity.SEVERITY_UNSPECIFIED))
                .addAllEscalations(Nullables.orEmpty(escalations))
                ;

        if (notificationChannels != null && !notificationChannels.isEmpty()) {
            // Deprecated channel settings.
            proto.addAllNotificationChannelIds(notificationChannels);
        }
        if (channels != null && !channels.isEmpty()) {
            // New channel settings.
            Map<String, TNotificationChannelOptions> channelsMap = channels.stream()
                    .collect(Collectors.toMap(channel -> channel.id, channel -> channel.config.toProto()));
            proto.putAllConfiguredNotificationChannels(channelsMap);
        }

        if (periodMillis != null && periodMillis != 0) {
            proto.setPeriodMillis(periodMillis);
        } else if (windowSecs != null && windowSecs != 0) {
            proto.setPeriodMillis(windowSecs * 1000L);
        }

        if (delaySeconds != 0) {
            proto.setDelaySeconds(delaySeconds);
        } else if (delaySecs != null && delaySecs != 0) {
            proto.setDelaySeconds(delaySecs);
        }

        if (Strings.isNotEmpty(createdAt)) {
            proto.setCreatedAt(Instant.parse(createdAt).toEpochMilli());
        }

        if (Strings.isNotEmpty(updatedAt)) {
            proto.setUpdatedAt(Instant.parse(updatedAt).toEpochMilli());
        }

        if (type != null) {
            type.fillProto(proto);
        }

        return proto.build();
    }

    @ApiModel(value = "Type", description = "Only one of the property can be specified")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class Type {
        @ApiModelProperty(
                value = "Compare values of a metrics with a user defined threshold",
                position = 1)
        public ThresholdDto threshold;

        @ApiModelProperty(
                value = "Evaluate used defined expression for metrics",
                notes = "This kind of alert should be use only when other kind not suitable",
                position = 2)
        public ExpressionDto expression;

        @ApiModelProperty(
                value = "Alert created by template",
                notes = "This kind of alert should be use only when other kind not suitable",
                position = 3)
        public AlertFromTemplateDto fromTemplate;

        public static Type fromProto(@Nonnull TAlert alert) {
            Type type = new Type();
            switch (alert.getTypeCase()) {
                case THRESHOLD:
                    type.threshold = ThresholdDto.fromProto(alert.getThreshold());
                    break;
                case EXPRESSION:
                    type.expression = ExpressionDto.fromProto(alert.getExpression());
                    break;
                case ALERT_FROM_TEMPLATE:
                    type.fromTemplate = AlertFromTemplateDto.fromProto(alert);
                    break;
                default:
                    throw new UnsupportedOperationException("Unsupported alert type: " + alert);
            }
            return type;
        }

        public void fillProto(@Nonnull TAlert.Builder proto) {
            if (threshold != null) {
                proto.setThreshold(threshold.toProto());
            }

            if (expression != null) {
                proto.setExpression(expression.toProto());
            }

            if (fromTemplate != null) {
                proto.setAlertFromTemplate(fromTemplate.toProto());
            }
        }
    }
}
