package ru.yandex.solomon.alert.gateway.dto.alert;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.AlertFromTemplate;
import ru.yandex.solomon.alert.protobuf.AlertParameter;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
@ApiModel(value = "AlertFromTemplate")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class AlertFromTemplateDto {

    @ApiModelProperty(
            value = "contains template id",
            example = "template_one")
    public String templateId;

    @ApiModelProperty(
            value = "Template version tag",
            example = "rc-1",
            required = true,
            position = 1)
    public String templateVersionTag;

    @ApiModelProperty(
            value = "A list of double parameters. Can't change after create.",
            position = 2)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<DoubleParameterValue> doubleValueParameters;

    @ApiModelProperty(
            value = "A list of integer parameters. Can't change after create.",
            position = 3)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<IntegerParameterValue> intValueParameters;

    @ApiModelProperty(
            value = "A list of text parameters. Can't change after create.",
            position = 4)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<TextParameterValue> textValueParameters;

    @ApiModelProperty(
            value = "A list of text list parameters. Can't change after create.",
            position = 5)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<TextListParameterValue> textListValueParameters;

    @ApiModelProperty(
            value = "A list of label list value parameters. Can't change after create.",
            position = 6)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<LabelListParameterValue> labelListValueParameters;

    @ApiModelProperty(
            value = "A list of double thresholds",
            position = 7)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<DoubleParameterValue> doubleValueThresholds;

    @ApiModelProperty(
            value = "A list of int thresholds",
            position = 8)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<IntegerParameterValue> intValueThresholds;

    @ApiModelProperty(
            value = "A list of text thresholds",
            position = 9)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<TextParameterValue> textValueThresholds;

    @ApiModelProperty(
            value = "A list of text list thresholds",
            position = 10)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<TextListParameterValue> textListValueThresholds;

    @ApiModelProperty(
            value = "A list of label list thresholds",
            position = 11)
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<LabelListParameterValue> labelListValueThresholds;

    @ApiModelProperty(
            value = "Reference to service provider. Can't change after create.",
            example = "service1",
            position = 13)
    public String serviceProvider;

    public static AlertFromTemplateDto fromProto(TAlert tAlert) {
        var alert = tAlert.getAlertFromTemplate();
        AlertFromTemplateDto dto = new AlertFromTemplateDto();
        dto.templateId = alert.getTemplateId();
        dto.templateVersionTag = alert.getTemplateVersionTag();
        dto.serviceProvider = tAlert.getServiceProvider();
        Map<AlertParameter.ParameterCase, List<AlertParameter>> collect = alert.getAlertParametersList().stream()
                .collect(Collectors.groupingBy(AlertParameter::getParameterCase));
        for (Map.Entry<AlertParameter.ParameterCase, List<AlertParameter>> entry : collect.entrySet()) {
            switch (entry.getKey()) {
                case DOUBLE_PARAMETER_VALUE -> dto.doubleValueParameters = entry.getValue().stream()
                        .map(DoubleParameterValue::fromProto)
                        .collect(Collectors.toList());
                case INTEGER_PARAMETER_VALUE -> dto.intValueParameters = entry.getValue().stream()
                        .map(IntegerParameterValue::fromProto)
                        .collect(Collectors.toList());
                case TEXT_PARAMETER_VALUE -> dto.textValueParameters = entry.getValue().stream()
                        .map(TextParameterValue::fromProto)
                        .collect(Collectors.toList());
                case TEXT_LIST_PARAMETER_VALUE -> dto.textListValueParameters = entry.getValue().stream()
                        .map(TextListParameterValue::fromProto)
                        .collect(Collectors.toList());
                case LABEL_LIST_PARAMETER_VALUE -> dto.labelListValueParameters = entry.getValue().stream()
                        .map(LabelListParameterValue::fromProto)
                        .collect(Collectors.toList());
            }
        }
        collect = alert.getAlertThresholdsList().stream()
                .collect(Collectors.groupingBy(AlertParameter::getParameterCase));
        for (Map.Entry<AlertParameter.ParameterCase, List<AlertParameter>> entry : collect.entrySet()) {
            switch (entry.getKey()) {
                case DOUBLE_PARAMETER_VALUE -> dto.doubleValueThresholds = entry.getValue().stream()
                        .map(DoubleParameterValue::fromProto)
                        .collect(Collectors.toList());
                case INTEGER_PARAMETER_VALUE -> dto.intValueThresholds = entry.getValue().stream()
                        .map(IntegerParameterValue::fromProto)
                        .collect(Collectors.toList());
                case TEXT_PARAMETER_VALUE -> dto.textValueThresholds = entry.getValue().stream()
                        .map(TextParameterValue::fromProto)
                        .collect(Collectors.toList());
                case TEXT_LIST_PARAMETER_VALUE -> dto.textListValueThresholds = entry.getValue().stream()
                        .map(TextListParameterValue::fromProto)
                        .collect(Collectors.toList());
                case LABEL_LIST_PARAMETER_VALUE -> dto.labelListValueThresholds = entry.getValue().stream()
                        .map(LabelListParameterValue::fromProto)
                        .collect(Collectors.toList());
            }
        }
        return dto;
    }

    public AlertFromTemplate toProto() {
        return AlertFromTemplate.newBuilder()
                .setTemplateId(Nullables.orEmpty(templateId))
                .setTemplateVersionTag(Nullables.orEmpty(templateVersionTag))
                .addAllAlertParameters(Nullables.orEmpty(doubleValueParameters).stream().map(DoubleParameterValue::toProto).collect(Collectors.toList()))
                .addAllAlertParameters(Nullables.orEmpty(intValueParameters).stream().map(IntegerParameterValue::toProto).collect(Collectors.toList()))
                .addAllAlertParameters(Nullables.orEmpty(textValueParameters).stream().map(TextParameterValue::toProto).collect(Collectors.toList()))
                .addAllAlertParameters(Nullables.orEmpty(textListValueParameters).stream().map(TextListParameterValue::toProto).collect(Collectors.toList()))
                .addAllAlertParameters(Nullables.orEmpty(labelListValueParameters).stream().map(LabelListParameterValue::toProto).collect(Collectors.toList()))
                .addAllAlertThresholds(Nullables.orEmpty(doubleValueThresholds).stream().map(DoubleParameterValue::toProto).collect(Collectors.toList()))
                .addAllAlertThresholds(Nullables.orEmpty(intValueThresholds).stream().map(IntegerParameterValue::toProto).collect(Collectors.toList()))
                .addAllAlertThresholds(Nullables.orEmpty(textValueThresholds).stream().map(TextParameterValue::toProto).collect(Collectors.toList()))
                .addAllAlertThresholds(Nullables.orEmpty(textListValueThresholds).stream().map(TextListParameterValue::toProto).collect(Collectors.toList()))
                .addAllAlertThresholds(Nullables.orEmpty(labelListValueThresholds).stream().map(LabelListParameterValue::toProto).collect(Collectors.toList()))
                .build();
    }

    @ApiModel(value = "DoubleParameterValue")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class DoubleParameterValue {
        @JsonProperty
        public Double value;
        @JsonProperty
        public String name;

        public static DoubleParameterValue fromProto(AlertParameter alertParameter) {
            DoubleParameterValue dto = new DoubleParameterValue();
            dto.name = alertParameter.getDoubleParameterValue().getName();
            dto.value = alertParameter.getDoubleParameterValue().getValue();
            return dto;
        }

        public static AlertParameter toProto(DoubleParameterValue doubleParameterValue) {
            return AlertParameter.newBuilder()
                    .setDoubleParameterValue(AlertParameter.DoubleParameterValue.newBuilder()
                            .setName(Nullables.orEmpty(doubleParameterValue.name))
                            .setValue(Nullables.orZero(doubleParameterValue.value))
                            .build())
                    .build();
        }
    }

    @ApiModel(value = "IntegerParameterValue")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class IntegerParameterValue {
        @JsonProperty
        public Integer value;
        @JsonProperty
        public String name;

        public static IntegerParameterValue fromProto(AlertParameter alertParameter) {
            IntegerParameterValue dto = new IntegerParameterValue();
            dto.name = alertParameter.getIntegerParameterValue().getName();
            dto.value = (int) alertParameter.getIntegerParameterValue().getValue();
            return dto;
        }

        public static AlertParameter toProto(IntegerParameterValue param) {
            return AlertParameter.newBuilder()
                    .setIntegerParameterValue(AlertParameter.IntegerParameterValue.newBuilder()
                            .setName(Nullables.orEmpty(param.name))
                            .setValue(Nullables.orZero(param.value))
                            .build())
                    .build();
        }
    }

    @ApiModel(value = "TextParameterValue")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class TextParameterValue {
        @JsonProperty
        public String value;
        @JsonProperty
        public String name;

        public static TextParameterValue fromProto(AlertParameter alertParameter) {
            TextParameterValue dto = new TextParameterValue();
            dto.name = alertParameter.getTextParameterValue().getName();
            dto.value = alertParameter.getTextParameterValue().getValue();
            return dto;
        }

        public static AlertParameter toProto(TextParameterValue param) {
            return AlertParameter.newBuilder()
                    .setTextParameterValue(AlertParameter.TextParameterValue.newBuilder()
                            .setName(Nullables.orEmpty(param.name))
                            .setValue(Nullables.orEmpty(param.value))
                            .build())
                    .build();
        }
    }

    @ApiModel(value = "TextListParameterValue")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class TextListParameterValue {
        @JsonProperty
        public List<String> value;
        @JsonProperty
        public String name;

        public static TextListParameterValue fromProto(AlertParameter alertParameter) {
            TextListParameterValue dto = new TextListParameterValue();
            dto.name = alertParameter.getTextListParameterValue().getName();
            dto.value = alertParameter.getTextListParameterValue().getValuesList();
            return dto;
        }

        public static AlertParameter toProto(TextListParameterValue param) {
            return AlertParameter.newBuilder()
                    .setTextListParameterValue(AlertParameter.TextListParameterValue.newBuilder()
                            .setName(Nullables.orEmpty(param.name))
                            .addAllValues(Nullables.orEmpty(param.value))
                            .build())
                    .build();
        }
    }

    @ApiModel(value = "LabelListParameterValue")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class LabelListParameterValue {
        @JsonProperty
        public List<String> value;
        @JsonProperty
        public String name;

        public static LabelListParameterValue fromProto(AlertParameter alertParameter) {
            LabelListParameterValue dto = new LabelListParameterValue();
            dto.name = alertParameter.getLabelListParameterValue().getName();
            dto.value = alertParameter.getLabelListParameterValue().getValuesList();
            return dto;
        }

        public static AlertParameter toProto(LabelListParameterValue param) {
            return AlertParameter.newBuilder()
                    .setLabelListParameterValue(AlertParameter.LabelListParameterValue.newBuilder()
                            .setName(Nullables.orEmpty(param.name))
                            .addAllValues(Nullables.orEmpty(param.value))
                            .build())
                    .build();
        }
    }
}
