package ru.yandex.solomon.alert.gateway.dto.alert;

import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.EAlertState;
import ru.yandex.solomon.alert.protobuf.EAlertType;
import ru.yandex.solomon.alert.protobuf.TListAlert;

/**
 * @author Vladimir Gordiychuk
 */
@ApiModel("AlertListItem")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class AlertListItemDto {
    @ApiModelProperty(
            value = "unique identity for alert",
            readOnly = true,
            required = true,
            dataType = "java.util.UUID",
            position = 0)
    public String id;

    @JsonProperty
    @ApiModelProperty(
            value = "Project if own by this alert",
            readOnly = true,
            required = true,
            dataType = "java.util.UUID",
            position = 1)
    public String projectId;

    @ApiModelProperty(
            value = "Human-readable name of alert",
            required = true,
            example = "Disk free space",
            position = 2)
    public String name;

    @ApiModelProperty(
            value = "Type of alert",
            required = true,
            position = 3)
    public EAlertType type;

    @ApiModelProperty(
            value = "State of current alert, only ACTIVE alerts will be periodically checked",
            required = true,
            position = 4)
    public EAlertState state;

    @ApiModelProperty(
            value = "True if current alert it's multi alert that unroll into multiple sub alerts, otherwise false",
            required = true,
            position = 5)
    public boolean multiAlert;

    @ApiModelProperty(
            value = "Statistic about alert evaluation, for multi alert statistic will contains count sub alerts in particular state",
            required = true,
            position = 6)
    public EvaluationStatsDto evaluationStats;

    @ApiModelProperty(
            value = "Statistic about alert notification",
            required = true,
            position = 7)
    public NotificationStatsDto notificationStats;

    @ApiModelProperty(
            value = "Type data for specific configuration",
            position = 8)
    public TypeData typeData;

    @ApiModelProperty(
            value = "Statistic about muted alert evaluation, for multi alert statistic will contain number of muted sub alerts in particular state",
            required = true,
            position = 9)
    public EvaluationStatsDto mutedStats;

    @ApiModelProperty(
            value = "Alert labels.",
            position = 10)
    public Map<String, String> labels;

    public static AlertListItemDto fromProto(@Nonnull TListAlert proto) {
        AlertListItemDto dto = new AlertListItemDto();
        dto.id = proto.getId();
        dto.projectId = proto.getProjectId();
        dto.name = proto.getName();
        dto.type = proto.getAlertType();
        dto.state = proto.getAlertState();
        dto.multiAlert = proto.getMultiAlert();
        dto.evaluationStats = EvaluationStatsDto.fromProto(proto.getEvaluationStats());
        dto.notificationStats = NotificationStatsDto.fromProto(proto.getNotificationStats());
        dto.mutedStats = EvaluationStatsDto.fromProto(proto.getMutedStats());
        dto.typeData = TypeData.fromProto(proto);
        dto.labels = proto.getLabelsMap();
        // TODO: proto.getEvaluationStatusCode() and proto.getMuteStatusCode() ?
        return dto;
    }

    @ApiModel(value = "TypeData", description = "Only one of the property can be specified")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class TypeData {

        @ApiModelProperty(
                value = "Alert created by template data",
                notes = "This kind of alert should be use only when other kind not suitable",
                position = 1)
        public FromTemplateData fromTemplate;

        public static TypeData fromProto(@Nonnull TListAlert alert) {
            switch (alert.getTypeCase()) {
                case TEMPLATE_DATA:
                    TypeData type = new TypeData();
                    type.fromTemplate = FromTemplateData.fromProto(alert.getTemplateData());
                    return type;
            }
            return null;
        }
    }

    @ApiModel(value = "FromTemplateData")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class FromTemplateData {
        @ApiModelProperty(
                value = "Template service provider",
                required = true,
                position = 1)
        @JsonProperty
        public String templateServiceProviderId;
        @ApiModelProperty(
                value = "Type of template",
                required = true,
                position = 2)
        @JsonProperty
        public EAlertType templateType;

        public static FromTemplateData fromProto(@Nonnull TListAlert.FromTemplateData data) {
            FromTemplateData dto = new FromTemplateData();
            dto.templateServiceProviderId = data.getTemplateServiceProviderId();
            dto.templateType = data.getTemplateType();
            return dto;
        }
    }
}
