package ru.yandex.solomon.alert.gateway.dto.alert;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.monitoring.api.v3.UnitFormat;
import ru.yandex.solomon.alert.protobuf.AlertTemplate;
import ru.yandex.solomon.alert.protobuf.AlertTemplateParameter;
import ru.yandex.solomon.alert.protobuf.NoPointsPolicy;
import ru.yandex.solomon.alert.protobuf.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
@ApiModel("AlertTemplate")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class AlertTemplateDto {
    @ApiModelProperty(
            value = "unique identity for alert template",
            required = true)
    public String id;

    @ApiModelProperty(
            value = "unique version tag for alert template",
            required = true)
    public String templateVersionTag;

    @ApiModelProperty(
            value = "service provider id for alert template",
            required = true)
    public String serviceProviderId;

    @ApiModelProperty(
            value = "Human-readable name of alert template",
            required = true,
            example = "Disk free space")
    public String name;

    @ApiModelProperty(value = "Description about alert in markdown format")
    public String description;

    @ApiModelProperty(value = "User created alert template",
            readOnly = true)
    public String createdBy;

    @ApiModelProperty(
            value = "Time when alert template was created",
            readOnly = true)
    public Instant createdAt;

    @ApiModelProperty(
            value = "User updated alert template",
            readOnly = true)
    public String updatedBy;

    @ApiModelProperty(
            value = "Time when alert template was updated last time",
            readOnly = true)
    public Instant updatedAt;

    @ApiModelProperty(
            value = "List of label key that should be use to group metrics, each group it's " +
                    "separate sub alert that check independently from other group.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<String> groupByLabels;

    @ApiModelProperty(
            value = "Type specific configuration",
            required = true)
    public AlertTemplateDto.Type type;

    @ApiModelProperty(
            value = "Templates that explain alert evaluation status, and what todo when alert occurs. " +
                    "Variables available to use into template depends on alert type")
    public Map<String, String> annotations;

    @ApiModelProperty(
            value = "Labels for that resource")
    public Map<String, String> labels;

    @ApiModelProperty(
            value = "Alert evaluation period in milliseconds. Period plus aggregate allow " +
                    "smooth frequently change parameter.",
            required = true)
    public Long periodMillis;

    @ApiModelProperty(
            value = "Time (in seconds) to delay evaluation relatively now. " +
                    "Useful for delayed metrics like cluster aggregation. " +
                    "Should be a non negative integer.")
    public int delaySeconds;

    @ApiModelProperty(
            value = "NO_DATA policy for selectors that resolve to empty set of metrics.")
    public ResolvedEmptyPolicy resolvedEmptyPolicy;

    @ApiModelProperty(
            value = "NO_DATA policy for timeseries without data points.")
    public NoPointsPolicy noPointsPolicy;

    @ApiModelProperty(
            value = "Template is default for resource.")
    public boolean isDefaultTemplate;

    @ApiModelProperty(
            value = "Template severity.")
    public Severity severity;

    @ApiModelProperty(
            value = "A list of double parameters")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<DoubleTemplateParameter> doubleParameters;

    @ApiModelProperty(
            value = "A list of integer parameters")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<IntegerTemplateParameter> intParameters;

    @ApiModelProperty(
            value = "A list of text parameters")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<TextTemplateParameter> textParameters;

    @ApiModelProperty(
            value = "A list of text list parameters")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<TextListTemplateParameter> textListParameters;

    @ApiModelProperty(
            value = "A list of label list value parameters")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<LabelListTemplateParameter> labelListParameters;

    @ApiModelProperty(
            value = "A list of double thresholds")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<DoubleTemplateParameter> doubleThresholds;

    @ApiModelProperty(
            value = "A list of int thresholds")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<IntegerTemplateParameter> intThresholds;

    @ApiModelProperty(
            value = "A list of text thresholds")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<TextTemplateParameter> textThresholds;

    @ApiModelProperty(
            value = "A list of text list thresholds")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<TextListTemplateParameter> textListThresholds;

    @ApiModelProperty(
            value = "A list of label list thresholds")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<LabelListTemplateParameter> labelListThresholds;

    public static AlertTemplateDto fromProto(@Nonnull AlertTemplate proto) {
        AlertTemplateDto dto = new AlertTemplateDto();
        dto.id = proto.getId();
        dto.templateVersionTag = proto.getTemplateVersionTag();
        dto.serviceProviderId = proto.getServiceProviderId();
        dto.name = proto.getName();
        dto.description = proto.getDescription();
        dto.createdBy = proto.getCreatedBy();
        dto.createdAt = Instant.ofEpochSecond(proto.getCreatedAt().getSeconds(), proto.getCreatedAt().getNanos());
        dto.updatedBy = proto.getModifiedBy();
        dto.updatedAt = Instant.ofEpochSecond(proto.getModifiedAt().getSeconds(), proto.getModifiedAt().getNanos());
        dto.groupByLabels = ImmutableList.copyOf(proto.getGroupByLabelsList());
        dto.type = AlertTemplateDto.Type.fromProto(proto);
        dto.annotations = ImmutableMap.copyOf(proto.getAnnotationsMap());
        dto.labels = ImmutableMap.copyOf(proto.getLabelsMap());
        dto.periodMillis = proto.getPeriodMillis();
        dto.delaySeconds = proto.getDelaySeconds();
        dto.resolvedEmptyPolicy = proto.getResolvedEmptyPolicy();
        dto.noPointsPolicy = proto.getNoPointsPolicy();
        dto.severity = proto.getSeverity();
        dto.isDefaultTemplate = proto.getDefault();
        Map<AlertTemplateParameter.ParameterCase, List<AlertTemplateParameter>> collect = proto.getAlertTemplateParametersList().stream()
                .collect(Collectors.groupingBy(AlertTemplateParameter::getParameterCase));
        for (var entry : collect.entrySet()) {
            switch (entry.getKey()) {
                case DOUBLE_PARAMETER -> dto.doubleParameters = entry.getValue().stream()
                        .map(DoubleTemplateParameter::fromProto)
                        .collect(Collectors.toList());
                case INTEGER_PARAMETER -> dto.intParameters = entry.getValue().stream()
                        .map(IntegerTemplateParameter::fromProto)
                        .collect(Collectors.toList());
                case TEXT_PARAMETER -> dto.textParameters = entry.getValue().stream()
                        .map(TextTemplateParameter::fromProto)
                        .collect(Collectors.toList());
                case TEXT_LIST_PARAMETER -> dto.textListParameters = entry.getValue().stream()
                        .map(TextListTemplateParameter::fromProto)
                        .collect(Collectors.toList());
                case LABEL_LIST_PARAMETER -> dto.labelListParameters = entry.getValue().stream()
                        .map(LabelListTemplateParameter::fromProto)
                        .collect(Collectors.toList());
            }
        }
        collect = proto.getAlertTemplateThresholdsList().stream()
                .collect(Collectors.groupingBy(AlertTemplateParameter::getParameterCase));
        for (var entry : collect.entrySet()) {
            switch (entry.getKey()) {
                case DOUBLE_PARAMETER -> dto.doubleThresholds = entry.getValue().stream()
                        .map(DoubleTemplateParameter::fromProto)
                        .collect(Collectors.toList());
                case INTEGER_PARAMETER -> dto.intThresholds = entry.getValue().stream()
                        .map(IntegerTemplateParameter::fromProto)
                        .collect(Collectors.toList());
                case TEXT_PARAMETER -> dto.textThresholds = entry.getValue().stream()
                        .map(TextTemplateParameter::fromProto)
                        .collect(Collectors.toList());
                case TEXT_LIST_PARAMETER -> dto.textListThresholds = entry.getValue().stream()
                        .map(TextListTemplateParameter::fromProto)
                        .collect(Collectors.toList());
                case LABEL_LIST_PARAMETER -> dto.labelListThresholds = entry.getValue().stream()
                        .map(LabelListTemplateParameter::fromProto)
                        .collect(Collectors.toList());
            }
        }
        return dto;
    }

    public AlertTemplate toProto() {
        AlertTemplate.Builder proto = AlertTemplate.newBuilder()
                .setId(Nullables.orEmpty(id))
                .setTemplateVersionTag(Nullables.orEmpty(templateVersionTag))
                .setServiceProviderId(Nullables.orEmpty(serviceProviderId))
                .setName(Nullables.orEmpty(name))
                .setDescription(Nullables.orEmpty(description))
                .setCreatedBy(Nullables.orEmpty(createdBy))
                .setModifiedBy(Nullables.orEmpty(updatedBy))
                .addAllGroupByLabels(Nullables.orEmpty(groupByLabels))
                .putAllAnnotations(Nullables.orEmpty(annotations))
                .putAllLabels(Nullables.orEmpty(labels))
                .setPeriodMillis(Nullables.orZero(periodMillis))
                .setDelaySeconds(Nullables.orZero(delaySeconds))
                .setResolvedEmptyPolicy(Nullables.orDefault(resolvedEmptyPolicy, ResolvedEmptyPolicy.RESOLVED_EMPTY_DEFAULT))
                .setNoPointsPolicy(Nullables.orDefault(noPointsPolicy, NoPointsPolicy.NO_POINTS_DEFAULT))
                .setSeverity(Nullables.orDefault(severity, Severity.SEVERITY_UNSPECIFIED))
                .setDefault(Nullables.orFalse(isDefaultTemplate))
                .addAllAlertTemplateParameters(Nullables.orEmpty(doubleParameters).stream().map(DoubleTemplateParameter::toProto).collect(Collectors.toList()))
                .addAllAlertTemplateParameters(Nullables.orEmpty(intParameters).stream().map(IntegerTemplateParameter::toProto).collect(Collectors.toList()))
                .addAllAlertTemplateParameters(Nullables.orEmpty(textParameters).stream().map(TextTemplateParameter::toProto).collect(Collectors.toList()))
                .addAllAlertTemplateParameters(Nullables.orEmpty(textListParameters).stream().map(TextListTemplateParameter::toProto).collect(Collectors.toList()))
                .addAllAlertTemplateParameters(Nullables.orEmpty(labelListParameters).stream().map(LabelListTemplateParameter::toProto).collect(Collectors.toList()))
                .addAllAlertTemplateThresholds(Nullables.orEmpty(doubleThresholds).stream().map(DoubleTemplateParameter::toProto).collect(Collectors.toList()))
                .addAllAlertTemplateThresholds(Nullables.orEmpty(intThresholds).stream().map(IntegerTemplateParameter::toProto).collect(Collectors.toList()))
                .addAllAlertTemplateThresholds(Nullables.orEmpty(textThresholds).stream().map(TextTemplateParameter::toProto).collect(Collectors.toList()))
                .addAllAlertTemplateThresholds(Nullables.orEmpty(textListThresholds).stream().map(TextListTemplateParameter::toProto).collect(Collectors.toList()))
                .addAllAlertTemplateThresholds(Nullables.orEmpty(labelListThresholds).stream().map(LabelListTemplateParameter::toProto).collect(Collectors.toList()));

        if (type != null) {
            type.fillProto(proto);
        }

        return proto.build();
    }

    @ApiModel(value = "AlertTemplateType", description = "Only one of the property can be specified")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class Type {
        @ApiModelProperty(
                value = "Compare values of a metrics with a user defined threshold",
                position = 1)
        public ThresholdAlertTemplateDto threshold;

        @ApiModelProperty(
                value = "Evaluate used defined expression for metrics",
                notes = "This kind of alert should be use only when other kind not suitable",
                position = 2)
        public ExpressionAlertTemplateDto expression;

        public static AlertTemplateDto.Type fromProto(@Nonnull AlertTemplate template) {
            AlertTemplateDto.Type type = new AlertTemplateDto.Type();
            switch (template.getTypeCase()) {
                case THRESHOLD:
                    type.threshold = ThresholdAlertTemplateDto.fromProto(template.getThreshold());
                    break;
                case EXPRESSION:
                    type.expression = ExpressionAlertTemplateDto.fromProto(template.getExpression());
                    break;
                default:
                    throw new UnsupportedOperationException("Unsupported alert template type: " + template);
            }
            return type;
        }

        public void fillProto(AlertTemplate.Builder proto) {
            if (threshold != null) {
                proto.setThreshold(threshold.toProto());
            }

            if (expression != null) {
                proto.setExpression(expression.toProto());
            }
        }
    }

    @ApiModel(value = "DoubleTemplateParameter")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class DoubleTemplateParameter {
        @JsonProperty
        public Double defaultValue;
        @JsonProperty
        public String name;
        @JsonProperty
        public String title;
        @JsonProperty
        public String description;
        @JsonProperty
        public UnitFormat unitFormat;

        public static DoubleTemplateParameter fromProto(AlertTemplateParameter alertParameter) {
            return fromProto(alertParameter.getDoubleParameter());
        }

        public static DoubleTemplateParameter fromProto(ru.yandex.solomon.alert.protobuf.DoubleTemplateParameter param) {
            DoubleTemplateParameter dto = new DoubleTemplateParameter();
            dto.name = param.getName();
            dto.defaultValue = param.getDefaultValue();
            dto.title = param.getTitle();
            dto.description = param.getDescription();
            dto.description = param.getDescription();
            dto.unitFormat = param.getUnitFormat();
            return dto;
        }

        public static AlertTemplateParameter toProto(DoubleTemplateParameter parameter) {
            return AlertTemplateParameter.newBuilder()
                    .setDoubleParameter(ru.yandex.solomon.alert.protobuf.DoubleTemplateParameter.newBuilder()
                            .setDefaultValue(Nullables.orZero(parameter.defaultValue))
                            .setName(Nullables.orEmpty(parameter.name))
                            .setTitle(Nullables.orEmpty(parameter.title))
                            .setDescription(Nullables.orEmpty(parameter.description))
                            .setUnitFormat(Nullables.orDefault(parameter.unitFormat, UnitFormat.UNIT_FORMAT_UNSPECIFIED))
                            .build())
                    .build();
        }
    }

    @ApiModel(value = "IntegerTemplateParameter")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class IntegerTemplateParameter {
        @JsonProperty
        public Integer defaultValue;
        @JsonProperty
        public String name;
        @JsonProperty
        public String title;
        @JsonProperty
        public String description;
        @JsonProperty
        public UnitFormat unitFormat;

        public static IntegerTemplateParameter fromProto(AlertTemplateParameter alertParameter) {
            IntegerTemplateParameter dto = new IntegerTemplateParameter();
            dto.name = alertParameter.getIntegerParameter().getName();
            dto.title = alertParameter.getIntegerParameter().getTitle();
            dto.defaultValue = (int) alertParameter.getIntegerParameter().getDefaultValue();
            dto.description = alertParameter.getIntegerParameter().getDescription();
            dto.unitFormat = alertParameter.getIntegerParameter().getUnitFormat();
            return dto;
        }

        public static AlertTemplateParameter toProto(IntegerTemplateParameter parameter) {
            return AlertTemplateParameter.newBuilder()
                    .setIntegerParameter(ru.yandex.solomon.alert.protobuf.IntegerTemplateParameter.newBuilder()
                            .setDefaultValue(Nullables.orZero(parameter.defaultValue))
                            .setName(Nullables.orEmpty(parameter.name))
                            .setTitle(Nullables.orEmpty(parameter.title))
                            .setDescription(Nullables.orEmpty(parameter.description))
                            .setUnitFormat(Nullables.orDefault(parameter.unitFormat, UnitFormat.UNIT_FORMAT_UNSPECIFIED))
                            .build())
                    .build();
        }
    }

    @ApiModel(value = "TextTemplateParameter")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class TextTemplateParameter {
        @JsonProperty
        public String defaultValue;
        @JsonProperty
        public String name;
        @JsonProperty
        public String title;
        @JsonProperty
        public String description;

        public static TextTemplateParameter fromProto(AlertTemplateParameter alertParameter) {
            TextTemplateParameter dto = new TextTemplateParameter();
            dto.name = alertParameter.getTextParameter().getName();
            dto.title = alertParameter.getTextParameter().getTitle();
            dto.defaultValue = alertParameter.getTextParameter().getDefaultValue();
            dto.description = alertParameter.getTextParameter().getDescription();
            return dto;
        }

        public static AlertTemplateParameter toProto(TextTemplateParameter parameter) {
            return AlertTemplateParameter.newBuilder()
                    .setTextParameter(ru.yandex.solomon.alert.protobuf.TextTemplateParameter.newBuilder()
                            .setDefaultValue(Nullables.orEmpty(parameter.defaultValue))
                            .setName(Nullables.orEmpty(parameter.name))
                            .setTitle(Nullables.orEmpty(parameter.title))
                            .setDescription(Nullables.orEmpty(parameter.description))
                            .build())
                    .build();
        }
    }

    @ApiModel(value = "TextListTemplateParameter")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class TextListTemplateParameter {
        @JsonProperty
        public List<String> defaultValue;
        @JsonProperty
        public String name;
        @JsonProperty
        public String title;
        @JsonProperty
        public String description;

        public static TextListTemplateParameter fromProto(AlertTemplateParameter alertParameter) {
            TextListTemplateParameter dto = new TextListTemplateParameter();
            dto.name = alertParameter.getTextListParameter().getName();
            dto.title = alertParameter.getTextListParameter().getTitle();
            dto.defaultValue = alertParameter.getTextListParameter().getDefaultValuesList();
            dto.description = alertParameter.getTextListParameter().getDescription();
            return dto;
        }

        public static AlertTemplateParameter toProto(TextListTemplateParameter parameter) {
            return AlertTemplateParameter.newBuilder()
                    .setTextListParameter(ru.yandex.solomon.alert.protobuf.TextListTemplateParameter.newBuilder()
                            .addAllDefaultValues(Nullables.orEmpty(parameter.defaultValue))
                            .setName(Nullables.orEmpty(parameter.name))
                            .setTitle(Nullables.orEmpty(parameter.title))
                            .setDescription(Nullables.orEmpty(parameter.description))
                            .build())
                    .build();
        }
    }

    @ApiModel(value = "LabelListTemplateParameter")
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class LabelListTemplateParameter {
        @JsonProperty
        public List<String> defaultValue;
        @JsonProperty
        public String name;
        @JsonProperty
        public String title;
        @JsonProperty
        public String labelKey;
        @JsonProperty
        public String selector;
        @JsonProperty
        public String description;
        @JsonProperty
        public String projectId;
        @JsonProperty
        public boolean multiselectable;

        public static LabelListTemplateParameter fromProto(AlertTemplateParameter alertParameter) {
            LabelListTemplateParameter dto = new LabelListTemplateParameter();
            dto.name = alertParameter.getLabelListParameter().getName();
            dto.title = alertParameter.getLabelListParameter().getTitle();
            dto.labelKey = alertParameter.getLabelListParameter().getLabelKey();
            dto.selector = alertParameter.getLabelListParameter().getSelectors();
            dto.defaultValue = alertParameter.getLabelListParameter().getDefaultValuesList();
            dto.description = alertParameter.getLabelListParameter().getDescription();
            dto.projectId = alertParameter.getLabelListParameter().getProjectId();
            dto.multiselectable = alertParameter.getLabelListParameter().getMultiselectable();
            return dto;
        }

        public static AlertTemplateParameter toProto(LabelListTemplateParameter parameter) {
            return AlertTemplateParameter.newBuilder()
                    .setLabelListParameter(ru.yandex.solomon.alert.protobuf.LabelListTemplateParameter.newBuilder()
                            .addAllDefaultValues(Nullables.orEmpty(parameter.defaultValue))
                            .setName(Nullables.orEmpty(parameter.name))
                            .setTitle(Nullables.orEmpty(parameter.title))
                            .setLabelKey(Nullables.orEmpty(parameter.labelKey))
                            .setSelectors(Nullables.orEmpty(parameter.selector))
                            .setDescription(Nullables.orEmpty(parameter.description))
                            .setProjectId(Nullables.orEmpty(parameter.projectId))
                            .setMultiselectable(parameter.multiselectable)
                            .build())
                    .build();
        }
    }
}
