package ru.yandex.solomon.alert.gateway.dto.alert;

import java.io.IOException;
import java.io.StringReader;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.TNotificationChannelOptions;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
@ApiModel("CreateAlertsFromTemplateCmd")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class CreateAlertsFromTemplateCmd {
    private static final TypeReference<LinkedHashMap<String, String>> RESOURCE_ID_TYPE = new TypeReference<>() {
    };

    @ApiModelProperty(
            value = "Service provider id",
            position = 1)
    public String serviceProviderId;

    @ApiModelProperty(
            value = "Templates to create",
            position = 2)
    public Set<String> templateIds;

    @ApiModelProperty(
            value = "Notification channels that will receive events",
            required = true,
            position = 3)
    public List<AssociatedChannelDto> channels;

    @ApiModelProperty(
            value = "Resource rows as text",
            position = 4)
    public String rowsText;

    @ApiModelProperty(
            value = "Rows format",
            position = 5)
    public String format;

    @ApiModelProperty(
            value = "Escalations that will receive events.",
            position = 6)
    public List<String> escalations;

    public CreateAlertsFromTemplateRequest toProto(@Nonnull String projectId, @Nonnull ObjectMapper mapper) {
        validateModel();
        try {
            Iterable<CSVRecord> records = CSVFormat.DEFAULT.builder()
                    .setDelimiter(",")
                    .build()
                    .parse(new StringReader(Nullables.orEmpty(rowsText)));
            Set<LinkedHashMap<String, String>> resources = new HashSet<>();
            if ("csv".equals(format)) {
                oldFormatParser(mapper, records, resources);
            } else {
                parseRow(records, resources);
            }
            if (resources.isEmpty()) {
                throw new BadRequestException("Resources should be specified");
            }
            for (LinkedHashMap<String, String> resource : resources) {
                if (resource.isEmpty()) {
                    throw new BadRequestException("Resource should be specified with keys");
                }
            }

            Map<String, TNotificationChannelOptions> channelsMap = Nullables.orEmpty(channels).stream()
                    .collect(Collectors.toMap(channel -> channel.id, channel -> channel.config.toProto()));

            return CreateAlertsFromTemplateRequest.newBuilder()
                    .addAllTemplateIds(templateIds)
                    .setServiceProviderId(serviceProviderId)
                    .putAllChannels(channelsMap)
                    .addAllEscalations(Nullables.orEmpty(escalations))
                    .setProjectId(projectId)
                    .addAllResources(resources.stream().map(map -> CreateAlertsFromTemplateRequest.Resource.newBuilder()
                            .putAllResourceParameters(map)
                            .build())
                            .collect(Collectors.toList()))
                    .build();
        } catch (IOException ioException) {
            throw new BadRequestException(ioException.getMessage());
        }
    }

    private void validateModel() {
        if (StringUtils.isEmpty(serviceProviderId)) {
            throw new BadRequestException("Service provider id should be specified");
        }
        if (templateIds == null || templateIds.isEmpty()) {
            throw new BadRequestException("Template ids should be specified");
        }
        if (channels == null || channels.isEmpty()) {
            throw new BadRequestException("Channels should be specified");
        }
        for (AssociatedChannelDto channel : channels) {
            if (StringUtils.isEmpty(channel.id)) {
                throw new BadRequestException("Channel id should be specified");
            }
        }
        for (var escalation : Nullables.orEmpty(escalations)) {
            if (StringUtils.isEmpty(escalation)) {
                throw new BadRequestException("Escalation id should be specified");
            }
        }
    }

    private void parseRow(Iterable<CSVRecord> records, Set<LinkedHashMap<String, String>> resources) throws IOException {
        for (CSVRecord record : records) {
            if (record.size() == 0) {
                throw new BadRequestException("Resource row id is empty");
            }
            LinkedHashMap<String, String> map = new LinkedHashMap<>();
            for (String kv : record) {
                final String[] split = kv.split(":");
                if (split.length != 2) {
                    throw new BadRequestException("Bad fragment " + kv + " must contain only one ':'");
                }
                map.put(split[0], split[1]);
            }
            resources.add(map);
        }
    }

    private void oldFormatParser(ObjectMapper mapper, Iterable<CSVRecord> records, Set<LinkedHashMap<String, String>> resources) throws IOException {
        int i = 0;
        for (CSVRecord record : records) {
            i++;
            if (i == 1) {
                // skip header
                continue;
            }
            if (record.size() == 0) {
                throw new BadRequestException("Resource row id is empty");
            }
            final String attributes = record.get(6);
            if (attributes == null) {
                throw new BadRequestException("attributes is empty for: " + record);
            }
            final CSVParser parseAttributesCsv = CSVParser.parse(attributes, CSVFormat.DEFAULT.builder()
                    .setSkipHeaderRecord(true)
                    .setDelimiter(";")
                    .build());
            final List<CSVRecord> records1 = parseAttributesCsv.getRecords();
            if (records1.size() != 1) {
                throw new BadRequestException("attributes wrong format for: " + record);
            }
            String resourceIdStr = "";
            String spId = "";
            String resourceType = "";
            final CSVRecord strings = records1.get(0);
            for (String string : strings) {
                if (string.startsWith("resource_id")) {
                    resourceIdStr = string.substring("resource_id".length() + 1);
                } else if (string.startsWith("service_provider_id")) {
                    spId = string.substring("service_provider_id".length() + 1);
                } else if (string.startsWith("resource_type")) {
                    resourceType = string.substring("resource_type".length() + 1);
                }
            }
            if (!spId.equals(serviceProviderId)) {
                throw new BadRequestException("wrong service provider id for: " + record);
            }
            try {
                LinkedHashMap<String, String> map = mapper.readValue(resourceIdStr.trim(), RESOURCE_ID_TYPE);
                if (!map.containsKey("resourceType") && !resourceType.isEmpty()) {
                    map.put("resourceType", resourceType);
                }
                resources.add(map);
            } catch (IOException ioException) {
                throw new BadRequestException("Resource row id bad format: " + resourceIdStr);
            }
        }
    }
}
