package ru.yandex.solomon.alert.gateway.dto.alert;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.protobuf.TSubAlert;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.model.protobuf.Label;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
@ApiModel(value = "SubAlert")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class SubAlertDto {
    @ApiModelProperty(
            value = "hex(sha1(parentId,labels))",
            required = true,
            position = 1)
    public String id;

    @ApiModelProperty(
            value = "Project if own by this alert",
            readOnly = true,
            dataType = "java.util.UUID",
            position = 2)
    public String projectId;

    @ApiModelProperty(
            value = "Unique list of labels of sub alert",
            required = true,
            position = 3)
    public Map<String, String> labels;

    @ApiModelProperty(
            value = "Parent contains definition for alert",
            required = true,
            readOnly = true,
            position = 3)
    public AlertDto parent;

    public static SubAlertDto fromProto(@Nonnull TSubAlert proto) {
        SubAlertDto dto = new SubAlertDto();
        dto.id = proto.getId();
        dto.projectId = proto.getProjectId();
        dto.labels = proto.getGroupKeyList()
                .stream()
                .collect(Collectors.toMap(
                        Label::getKey,
                        Label::getValue,
                        (u, v) -> {
                            throw new IllegalStateException("Duplicate key " + u);
                        },
                        () -> new LinkedHashMap<>(proto.getGroupKeyCount())));
        dto.parent = AlertDto.fromProto(proto.getParent());
        return dto;
    }

    public TSubAlert toProto() {
        return TSubAlert.newBuilder()
                .setId(Nullables.orEmpty(id))
                .setProjectId(Nullables.orEmpty(projectId))
                .addAllGroupKey(LabelConverter.labelsToProtoList(Labels.of(Nullables.orEmpty(labels))))
                .setParent(Nullables.orDefault(parent, new AlertDto()).toProto())
                .build();
    }
}
