package ru.yandex.solomon.alert.gateway.dto.alert;

import java.time.Instant;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.alert.protobuf.AlertMuteStatus;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TListSubAlert;
import ru.yandex.solomon.model.protobuf.Label;

/**
 * @author Vladimir Gordiychuk
 */
@ApiModel("SubAlertListItem")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class SubAlertListItemDto {
    @ApiModelProperty(
            value = "unique identity for alert",
            readOnly = true,
            dataType = "java.util.UUID",
            position = 1)
    public String id;

    @ApiModelProperty(
            value = "Project if own by this alert",
            readOnly = true,
            dataType = "java.util.UUID",
            position = 2)
    public String projectId;

    @ApiModelProperty(
            value = "Parent id contains definition for alert",
            required = true,
            position = 3)
    public String parentId;

    @ApiModelProperty(
            value = "Unique list of labels of sub alert",
            required = true,
            position = 4)
    public Map<String, String> labels;

    @ApiModelProperty(
            value = "Latest evaluation status",
            required = true,
            position = 5)
    public TEvaluationStatus.ECode evaluationStatusCode;

    @ApiModelProperty(
        value = "Latest evaluation time",
        required = true,
        position = 6)
    public String latestEval;

    @ApiModelProperty(
            value = "Statistic about alert notification",
            required = true,
            position = 7)
    public NotificationStatsDto notificationStats;

    @ApiModelProperty(
            value = "Filtered annotations",
            required = true,
            position = 8)
    public Map<String, String> annotations;

    @ApiModelProperty(
            value = "Mute status code",
            required = true,
            position = 9)
    public AlertMuteStatus.Code muteStatusCode;

    public static SubAlertListItemDto fromProto(@Nonnull TListSubAlert proto) {
        SubAlertListItemDto dto = new SubAlertListItemDto();
        dto.id = proto.getId();
        dto.projectId = proto.getProjectId();
        dto.parentId = proto.getParentId();
        dto.evaluationStatusCode = proto.getEvaluationStatusCode();
        dto.muteStatusCode = proto.getMuteStatusCode();
        dto.latestEval = Instant.ofEpochMilli(proto.getLatestEvalMillis()).toString();
        dto.labels = proto.getLabelsList()
                .stream()
                .collect(Collectors.toMap(
                        Label::getKey,
                        Label::getValue,
                        (u, v) -> {
                            throw new IllegalStateException("Duplicate key " + u);
                        },
                        () -> new LinkedHashMap<>(proto.getLabelsCount())));
        dto.notificationStats = NotificationStatsDto.fromProto(proto.getNotificationStats());
        dto.annotations = proto.getAnnotationsMap();
        return dto;
    }
}
